/**
 * 
 */
package com.amentra.metamatrix.solr;

import java.math.BigInteger;
import java.sql.Date;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Set;
import java.util.Map.Entry;

import org.apache.solr.client.solrj.SolrQuery;
import org.apache.solr.client.solrj.SolrServerException;
import org.apache.solr.client.solrj.impl.CommonsHttpSolrServer;
import org.apache.solr.client.solrj.response.QueryResponse;
import org.apache.solr.common.SolrDocument;
import org.apache.solr.common.SolrDocumentList;

import com.amentra.metamatrix.solr.visitor.SolrHierarchyVisitor;
import com.metamatrix.data.api.Batch;
import com.metamatrix.data.api.ConnectorLogger;
import com.metamatrix.data.api.SynchQueryExecution;
import com.metamatrix.data.basic.BasicBatch;
import com.metamatrix.data.exception.ConnectorException;
import com.metamatrix.data.language.IQuery;
import com.metamatrix.data.metadata.runtime.RuntimeMetadata;

/**
 * Supports read-only, synchronious queries that are translated to Solr searches and executed.
 * @author Michael Walker
 */
public class SyncQueryExecution implements SynchQueryExecution {
	private RuntimeMetadata md;
	private ConnectorLogger logger;
	private CommonsHttpSolrServer server;
	private SolrQuery solrQuery = new SolrQuery();
	private SolrDocumentList docs;
	private int maxBatchSize;
	private ArrayList<String> fieldList;
	private HashMap<String, Class> fieldMap;
	// Number of documents found by Solr
	private int numFound;
	// Actual number of docs to retrieve -- could be less than numFound if LIMIT is used
	private int numToRetrieve;
	private int numProcessed;
	private boolean useLowerCase;
	private int initialLimit;
	private int initialOffset;
	private SolrHierarchyVisitor visitor;

	public SyncQueryExecution(RuntimeMetadata md, CommonsHttpSolrServer server, ConnectorLogger logger, boolean useLowerCase) {
		this.md = md;
		this.server = server;
		this.logger = logger;
		this.useLowerCase = useLowerCase;
	}

	/**
	 * Uses a visitor to process the IQuery and extract the parameters for a SolrQuery search, and then executes the search,
	 * retrieving a list of Documents that match the criteria.
	 * @see com.metamatrix.data.api.SynchQueryExecution#execute(com.metamatrix.data.language.IQuery, int)
	 */
	public void execute(IQuery query, int maxBatchSize)
			throws ConnectorException {
		this.maxBatchSize = maxBatchSize;
		logger.logDetail("Solr Connector batch size: " + maxBatchSize);
		// Visit the MMX query and generate the Solr query
		visitor = new SolrHierarchyVisitor(md, logger, useLowerCase);
		visitor.visitNode(query);
		fieldList = visitor.getFieldList();
		fieldMap = visitor.getFieldMap();
			
		// Execute query in Solr and retrieve document result list.
	    solrQuery.setQuery(visitor.getQueryString());
	    logger.logDetail("Solr query string: " + visitor.getQueryString());
	    // Add only the fields of interest.
	    for(String field : fieldList) {
		    solrQuery.addField(field);
	    }

	    if(visitor.getLimit()== null) {
	    	initialLimit = -1;
	    } else {
	    	initialLimit = visitor.getLimit();
	    }
	    if(visitor.getOffset() == null) {
	    	initialOffset = 0;
	    } else {
	    	initialOffset = visitor.getOffset();
	    }
	    // If the limit is smaller than one batch, then set the query to retrieve one limit's worth of data.
    	// It's important not to pull all results in a single query, since SolrJ loads it all into memory. 
    	// Instead, it's advised to match the batch size to the query size.
    	if(initialLimit != -1 && initialLimit < maxBatchSize) {
    		solrQuery.setRows(initialLimit);
    	} else {
    		solrQuery.setRows(maxBatchSize);
    	}
    	logger.logTrace("Set limit to: " + visitor.getLimit());
	    if(visitor.getOffset() != null) {
	    	solrQuery.setStart(initialOffset);
	    	logger.logTrace("Set query offset to: " + initialOffset);
	    }
	    // TODO Could more stuff here as needed, such as Order by
	    //solrQuery.addSortField( "price", SolrQuery.ORDER.asc );

	    QueryResponse rsp = null;
		try {
			if(server == null) {
				throw new ConnectorException("Server connection was null. Failed to initialize connector properly.");
			}
			rsp = server.query(solrQuery);
		} catch (SolrServerException e) {
			logger.logError("Failed to execute Solr query: " + solrQuery);
			e.printStackTrace();
		}
	    docs = rsp.getResults();
	    numFound = (int) docs.getNumFound();
	    logger.logDetail("Total docs returned: " + numFound);
	    if(initialLimit != -1 && initialLimit < numFound) {
	    	numToRetrieve = initialLimit;
	    } else {
	    	numToRetrieve = numFound;
	    }
	}

	/**
	 * Pull the field values from the list of documents until a batch is fulfilled or the end of the 
	 * document list is reached. If a field contains multiple values, only the first value returned by Solr will be returned.
	 * Only string values will be handled.
	 * @see com.metamatrix.data.api.BatchedExecution#nextBatch()
	 */
	public Batch nextBatch() throws ConnectorException {
		int curBatchSize = 0;
		Batch batch = new BasicBatch();
		Iterator<SolrDocument> docItr = docs.iterator();
		
		while((curBatchSize + numProcessed) < numToRetrieve &&
				curBatchSize < maxBatchSize &&
				docItr.hasNext()) {
			SolrDocument doc = docItr.next();
			ArrayList row = new ArrayList();
			for(String fieldName : fieldList) {
				// TODO Consider multiple values for the field here. 
				// This could be handled via multivalue-concat, multivalue-expand, or name/value pair tables.
				row = addFieldToRow(fieldName, doc, row);
			}
			batch.addRow(row);
			curBatchSize++;
		}
		numProcessed += curBatchSize;
		// Processed all results
		if(numProcessed >= numToRetrieve) {
			batch.setLast();
		// Get the next batch
		} else {
			solrQuery.setStart(initialOffset + numProcessed);
			solrQuery.setRows(maxBatchSize);
		    QueryResponse rsp = null;
			try {
				rsp = server.query(solrQuery);
			} catch (SolrServerException e) {
				e.printStackTrace();
				throw new ConnectorException("Solr next batch call failed.");

			}
		    docs = rsp.getResults();
		}
		return batch;
	}
	
	/**
	 * Get the field value and cast it to the class defined in the MMX model, if possible.
	 * @throws ConnectorException 
	 */
	private ArrayList addFieldToRow(String fieldName, SolrDocument doc, ArrayList row) throws ConnectorException {
		Class fieldClass = fieldMap.get(fieldName);
		try {
			// String
			if(fieldClass.equals(Class.forName(String.class.getName()))) {
				row.add((String)doc.getFieldValue(fieldName));
			// Integer
			} else if(fieldClass.equals(Class.forName(Integer.class.getName()))) {
				row.add((Integer)doc.getFieldValue(fieldName));
			// Date
			// Solr will pass back java.util.Date, but we represent it as java.sql.Date
			// so it must be cast appropriately.
			} else if(fieldClass.equals(Class.forName(java.sql.Date.class.getName()))) {
				java.sql.Date value = new java.sql.Date(((java.util.Date) doc.getFieldValue(fieldName)).getTime());
				row.add(value);
			} else {
				logger.logWarning("Type " + fieldClass.getName() + " not supported by connector. Support must be added or field type must be changed. Attempting to cast to string.");
				row.add((String)doc.getFieldValue(fieldName));
			}
		} catch (ClassNotFoundException e) {
			throw new ConnectorException("Could not cast field. Check type in source model. Field name: " + fieldName);
		} catch (ClassCastException e) {
			throw new ConnectorException("Could not cast field. Check type in source model. Field name: " + fieldName);
		}
		return row;
	}
	
	/**
	 * Does nothing, since we call the query synchronously.
	 * @see com.metamatrix.data.api.Execution#cancel()
	 */
	public void cancel() throws ConnectorException {
		;
	}

	/**
	 * Does nothing, since there is no matching close() method for the QueryResponse.
	 * @see com.metamatrix.data.api.Execution#close()
	 */
	public void close() throws ConnectorException {
		;
	}

}
