package com.redhat.installer.asconfiguration.processpanel.postinstallation;

import com.izforge.izpack.installer.AutomatedInstallData;
import com.izforge.izpack.util.AbstractUIProcessHandler;
import com.redhat.installer.installation.processpanel.ProcessPanelHelper;
import com.redhat.installer.installation.util.InstallationUtilities;
import org.apache.commons.cli.MissingArgumentException;
import org.apache.commons.lang.exception.ExceptionUtils;
import org.wildfly.security.auth.principal.NamePrincipal;
import org.wildfly.security.auth.realm.FileSystemSecurityRealm;
import org.wildfly.security.auth.server.ModifiableRealmIdentity;
import org.wildfly.security.auth.server.RealmUnavailableException;
import org.wildfly.security.authz.MapAttributes;
import org.wildfly.security.credential.PasswordCredential;
import org.wildfly.security.password.PasswordFactory;
import org.wildfly.security.password.WildFlyElytronPasswordProvider;
import org.wildfly.security.password.interfaces.ClearPassword;
import org.wildfly.security.password.spec.ClearPasswordSpec;

import java.nio.file.Paths;
import java.security.NoSuchAlgorithmException;
import java.security.Security;
import java.security.spec.InvalidKeySpecException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class InstallFilesystemRealm {

    private static final String IDENTITY = "identity";
    private static final String ROLE = "role";
    private static final String PATH = "path";

    private static final String ARGUMENT_SPLIT_REGEX = "\\s*=\\s*";
    private static final String IDENTITY_SPLIT_REGEX = "\\s*:\\s*";
    public static final String ROLE_SPLIT_REGEX = "\\s*,\\s*";

    private final Map<String, String> parsedArguments = new HashMap<>();
    private final List<Identity> identities = new ArrayList<>();

    private FileSystemSecurityRealm realm;
    private AbstractUIProcessHandler mHandler;

    private static class Identity {
        final String principal;
        final String password;
        final List<String> role;

        private Identity(String principal, String password, List<String> role) {
            this.principal = principal;
            this.password = password;
            this.role = role;
        }
    }

    public boolean run(AbstractUIProcessHandler handler, String[] args) {
        mHandler = handler;
        boolean result = parseArguments(args) && createRealm() && createIdentities();
        printResult(result);
        return result;
    }

    private boolean parseArguments(String[] arguments) {
        for (String arg : arguments) {
            String[] split = arg.split(ARGUMENT_SPLIT_REGEX);
            String key = split[0];
            String value = split[1];
            if (key.equals(IDENTITY)) {
                parseIdentity(value);
            } else {
                parsedArguments.put(key, value);
            }
        }
        try {
            checkRequiredParameter(PATH);
        } catch (MissingArgumentException e) {
            printError(e);
            return false;
        }
        return true;
    }

    private void parseIdentity(String alias) {
        String[] split = alias.split(IDENTITY_SPLIT_REGEX);
        String principal = split[0];
        String password = split[1];
        List<String> roles = Arrays.asList(split[2].split(ROLE_SPLIT_REGEX));
        identities.add(new Identity(principal, password, roles));
    }

    private void checkRequiredParameter(String name) throws MissingArgumentException {
        if (!parsedArguments.containsKey(name)) {
            throw new MissingArgumentException(name + " is not defined.");
        }
    }

    private boolean createRealm() {
        Security.addProvider(new WildFlyElytronPasswordProvider());
        String realmPath = parsedArguments.get(PATH);
        InstallationUtilities.addFileToCleanupList(realmPath);
        realm = new FileSystemSecurityRealm(Paths.get(realmPath));
        return true;
    }

    private boolean createIdentities() {
        for (Identity identity : identities) {
            try {
                createIdentity(identity.principal, identity.password, identity.role);
            } catch (RealmUnavailableException | InvalidKeySpecException | NoSuchAlgorithmException e) {
                printError(e);
                return false;
            }
        }
        return true;
    }

    private void createIdentity(String principal, String password, List<String> role) throws RealmUnavailableException, InvalidKeySpecException, NoSuchAlgorithmException {
        ModifiableRealmIdentity modifiableIdentity = realm.getRealmIdentityForUpdate(new NamePrincipal(principal));
        if (!modifiableIdentity.exists()) {
            modifiableIdentity.create();
        }

        PasswordFactory passwordFactory = PasswordFactory.getInstance(ClearPassword.ALGORITHM_CLEAR);
        PasswordCredential clearPassword = new PasswordCredential(passwordFactory.generatePassword(new ClearPasswordSpec(password.toCharArray())));
        modifiableIdentity.setCredentials(Collections.singleton(clearPassword));

        MapAttributes attributes = new MapAttributes();
        attributes.addAll(ROLE, role);
        modifiableIdentity.setAttributes(attributes);
    }

    private void printError(Exception e) {
        ProcessPanelHelper.printToPanel(mHandler, e.getMessage(), true);
        ProcessPanelHelper.printToLog(ExceptionUtils.getStackTrace(e));
    }

    private void printResult(boolean result) {
        AutomatedInstallData idata = AutomatedInstallData.getInstance();
        if (result) {
            ProcessPanelHelper.printToPanel(mHandler, idata.langpack.getString("FilesystemRealmInstaller.install.success"), false);
        } else {
            ProcessPanelHelper.printToPanel(mHandler, idata.langpack.getString("FilesystemRealmInstaller.install.failure"), true);
        }
    }
}