package com.redhat.installer.installation.maven.validator;

import com.izforge.izpack.panels.ProcessingClient;
import com.izforge.izpack.panels.Validator;
import com.izforge.izpack.util.Debug;
import org.apache.commons.io.FilenameUtils;
import org.w3c.dom.Document;
import org.xml.sax.SAXException;

import javax.xml.XMLConstants;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.Source;
import javax.xml.transform.stream.StreamSource;
import javax.xml.validation.Schema;
import javax.xml.validation.SchemaFactory;
import java.io.*;
import java.net.URL;
import java.net.URLConnection;
import java.util.Arrays;
import java.util.Scanner;

/**
 * Created by thauser on 5/1/15.
 */


public class MavenSchemaValidator implements Validator {

    private String mavenSettingsVersion = "1.0.0";
    private String mavenSettingsXSD = "http://maven.apache.org/xsd/settings-" + mavenSettingsVersion + ".xsd";

    @Override
    public Status validate(ProcessingClient client) {
        Status failureStatus = Status.ERROR;
        if (client.hasParams() && Boolean.parseBoolean(client.getValidatorParams().get(WARNING_PARAM)))
            failureStatus = Status.WARNING;
        String userSettings = client.getText();
        File settingsFile = new File(userSettings);
        if (!settingsFile.exists() || settingsFile.isDirectory() || settingsFile.length() == 0) {
            return Status.OK;
        } else {
            if (isValidXml(settingsFile)) {
                if (determineSettingsVersion(settingsFile)) {
                    Status validationResult = (settingsFile.canWrite() && isSchemaValid(settingsFile)) ? Status.OK : failureStatus;
                    return validationResult;
                }
            }
            return failureStatus;
        }
    }

    public boolean isSchemaValid(File setFile) {
        try {
            URL schemaFile = new URL(mavenSettingsXSD);
            Source xmlFile = new StreamSource(setFile);
            SchemaFactory schemaFactory = SchemaFactory.newInstance(XMLConstants.W3C_XML_SCHEMA_NS_URI);
            Schema schema = schemaFactory.newSchema(schemaFile);
            javax.xml.validation.Validator validator = schema.newValidator();
            validator.validate(xmlFile);
        } catch (Exception e) {
            return false;
        }
        return true;
    }

    public boolean isValidXml(File file) {
        DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
        dbf.setNamespaceAware(true);
        try {
            InputStream in = new BufferedInputStream(new FileInputStream(file));
            String fileType = URLConnection.guessContentTypeFromStream(in);
            if (fileType == null || !fileType.equals("application/xml")) {
                return false;
            }
        } catch (IOException e) {
            return false;
        }
        return true;
    }

    private boolean determineSettingsVersion(File file) {
        DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
        dbf.setNamespaceAware(true);
        try {
            DocumentBuilder db = dbf.newDocumentBuilder();
            Document dom = db.parse(file);
            String schemaLocation = dom.getElementsByTagName("settings").item(0).getNamespaceURI();
            setMavenSettingsVersion(getVersion(schemaLocation));
        } catch (ParserConfigurationException | SAXException | IOException | NullPointerException e) {
            return false;
        }
        return true;
    }

    private String getVersion(String schemaLocation) {
        String[] splitLocation = schemaLocation.split("/");
        return Arrays.asList(splitLocation).get(splitLocation.length - 1);
    }

    private void setMavenSettingsVersion(String version) {
        mavenSettingsVersion = version;
        mavenSettingsXSD = "http://maven.apache.org/xsd/settings-" + mavenSettingsVersion + ".xsd";
    }

}
