package org.jboss.brmsbpmsuite.patching.integrationtests.model.distribution;

import java.io.File;

/**
 * Contains all supported distribution types that can be patched.
 */
public enum DistributionType {
    RHPAM_EAP7X_BC("rhpam-business-central-eap7-deployable",
                "standalone/deployments/business-central.war",
                "business-central-eap7-deployable/",
                "business-central-eap7-deployable/"),
    RHPAM_EE8_KIE("rhpam-kie-server-ee8",
            "kie-server.war",
            "kie-server-ee8/",
            "kie-server-ee8/"),
    RHPAM_EE7_CTRL("rhpam-controller-ee7",
            "controller.war",
            "controller-ee7/",
            "controller-ee7/"),
    RHPAM_JWS_KIE("rhpam-kie-server-jws",
            "kie-server.war",
            "kie-server-jws/",
            "kie-server-jws/"),
    RHPAM_JWS_CTRL("rhpam-controller-jws",
            "controller.war",
            "controller-jws/",
            "controller-jws/"),
    RHDM_EAP7X_DC("rhdm-decision-central-eap7-deployable",
                "standalone/deployments/decision-central.war",
                "decision-central-eap7-deployable/",
                "decision-central-eap7-deployable/"),
    RHDM_EE8_KIE("rhdm-kie-server-ee8",
                "kie-server.war",
                "kie-server-ee8/",
                "kie-server-ee8/"),
    RHDM_JWS_KIE("rhdm-kie-server-jws",
                "kie-server.war",
                "kie-server-jws/",
                "kie-server-jws/"),
    RHDM_EE7_CTRL("rhdm-controller-ee7",
                "controller.war",
                "controller-ee7/",
                "controller-ee7/"),
    RHDM_JWS_CTRL("rhdm-controller-jws",
                 "controller.war",
                 "controller-jws/",
                 "controller-jws/");

    /**
     * Name of distribution type. Passed to patch tool as a distribution type argument.
     */
    private final String name;

    /**
     * Relative path to patched directory within test distribution.
     * Added to testing distribution path and passed as an argument to patch tool.
     */
    private final String relativePath;

    /**
     * Relative path to distribution files directory within patch tool.
     */
    private final String relativePathPatchTool;

    /**
     * Relative path to root directory for distribution type files.
     */
    private final String relativeRootDirectoryPathPatchTool;

    DistributionType(final String name, final String relativePath, final String relativePathPatchTool,
            final String relativeRootDirectoryPathPatchTool) {
        this.name = name;
        this.relativePath = relativePath;
        this.relativePathPatchTool = relativePathPatchTool;
        this.relativeRootDirectoryPathPatchTool = relativeRootDirectoryPathPatchTool;
    }

    public String getName() {
        return name;
    }

    public String getRelativeRootDirectoryPathPatchTool() {
        return relativeRootDirectoryPathPatchTool;
    }

    /**
     * Gets appropriate relative path by defined relative path type. There are few relative path types.
     * They bound different relative paths to different patching process entities.
     * I.e. distribution files are located on different paths within patch tool and within testing distribution.
     * @param relativePathType Type of relative path that is returned.
     * @param additionalPrefixPath Additional custom path that is added as prefix path to returned relative path.
     * @return Relative path for this distribution type instance.
     */
    public String getRelativePath(final RelativePathType relativePathType, final String additionalPrefixPath) {
        switch (relativePathType) {
            case PATH_WITHIN_DISTRIBUTION:
                // Some distributions can have special content root dir, that is specific to them and cannot be
                // generalized, so it must be passed as a build parameter and then added to path.
                if (additionalPrefixPath == null || additionalPrefixPath.isEmpty()) {
                    return relativePath;
                } else {
                    return additionalPrefixPath + File.separator + relativePath;
                }
            case PATH_WITHIN_PATCH_TOOL:
                if (additionalPrefixPath == null || additionalPrefixPath.isEmpty()) {
                    return relativePathPatchTool;
                } else {
                    return additionalPrefixPath + File.separator + relativePathPatchTool;
                }
            default:
                throw new IllegalArgumentException("Relative path type " + relativePathType + " not supported!");
        }
    }

    public static DistributionType getDistributionTypeByName(final String name) {
        for (DistributionType type : DistributionType.values()) {
            if (type.getName().equals(name)) {
                return type;
            }
        }
        throw new IllegalArgumentException("Unrecognized distribution type '" + name + "'! ");
    }
}
