package org.jboss.brmsbpmsuite.patching.client;

import org.jboss.brmsbpmsuite.patching.phase.PatchingPhase;

import java.io.File;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;


/**
 * Configuration options needed for running the client patcher.
 */
public class ClientPatcherConfig {

    public static final SimpleDateFormat BACKUP_DIR_NAME_DATE_FORMAT = new SimpleDateFormat("yyyy-MM-dd-hh-mm-ss-SS");

    private File distributionRoot;
    private DistributionType distributionType;
    private TargetProduct product;
    private File patchBasedir;
    private File backupDir;
    private List<DistributionType> supportedDistroTypes;
    private List<PatchingPhase> phasesToExecute;
    private boolean isDistributionExploded = true;

    public static ClientPatcherConfig empty() {
        return new ClientPatcherConfig();
    }

    public ClientPatcherConfig() {

    }

    public File getDistributionRoot() {
        return distributionRoot;
    }

    public void setDistributionRoot(File distributionRoot) {
        this.distributionRoot = distributionRoot;
    }

    public DistributionType getDistributionType() {
        return distributionType;
    }

    public void setDistributionType(DistributionType distributionType) {
        this.distributionType = distributionType;
    }

    public TargetProduct getProduct() {
        return product;
    }

    public void setProduct(TargetProduct product) {
        this.product = product;
    }

    public File getPatchBasedir() {
        return patchBasedir;
    }

    public void setPatchBasedir(File patchBasedir) {
        this.patchBasedir = patchBasedir;
    }

    public File getBackupDir() {
        if (backupDir == null) {
            File executionDir = new File(System.getProperty("user.dir"));
            setBackupBaseDir(executionDir);
        }
        return backupDir;
    }

    public void setBackupBaseDir(File backupBaseDir) {
        String timestampDirName = BACKUP_DIR_NAME_DATE_FORMAT.format(new Date());
        backupDir = new File(backupBaseDir, timestampDirName);
    }

    public void setSupportedDistroTypes(
            List<DistributionType> supportedDistroTypes) {
        this.supportedDistroTypes = supportedDistroTypes;
    }

    public List<PatchingPhase> getPhasesToExecute() {

        // Patching phases were set external (Debug purpose).
        if (phasesToExecute != null) {
            return phasesToExecute;
        } else {
            ArrayList<PatchingPhase> phaseList = new ArrayList<PatchingPhase>();
            phaseList.add(PatchingPhase.CHECK_DISTRO);
            phaseList.add(PatchingPhase.CHECK_VERSION);
            phaseList.add(PatchingPhase.BACKUP);
            phaseList.add(PatchingPhase.APPLY);
            if (isDistributionExploded) {
                return phaseList;
            } else {
                // Explode distribution
                phaseList.add(PatchingPhase.EXPLODE_WAR);
                phaseList.add(PatchingPhase.ZIP_WAR);
                return phaseList;
            }
        }
    }

    public void setPhasesToExecute(List<PatchingPhase> phasesToExecute) {
        this.phasesToExecute = phasesToExecute;
    }

    public boolean isDistributionExploded() {
        return isDistributionExploded;
    }

    public void setDistributionExploded(boolean distributionExploded) {
        isDistributionExploded = distributionExploded;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        ClientPatcherConfig that = (ClientPatcherConfig) o;

        if (isDistributionExploded != that.isDistributionExploded) return false;
        if (distributionRoot != null ? !distributionRoot.equals(that.distributionRoot) : that.distributionRoot != null)
            return false;
        if (distributionType != that.distributionType) return false;
        if (product != that.product) return false;
        if (patchBasedir != null ? !patchBasedir.equals(that.patchBasedir) : that.patchBasedir != null) return false;
        if (backupDir != null ? !backupDir.equals(that.backupDir) : that.backupDir != null)
            return false;
        if (supportedDistroTypes != null ? !supportedDistroTypes.equals(that.supportedDistroTypes) : that.supportedDistroTypes != null)
            return false;
        return phasesToExecute != null ? phasesToExecute.equals(that.phasesToExecute) : that.phasesToExecute == null;
    }

    @Override
    public int hashCode() {
        int result = distributionRoot != null ? distributionRoot.hashCode() : 0;
        result = 31 * result + (distributionType != null ? distributionType.hashCode() : 0);
        result = 31 * result + (product != null ? product.hashCode() : 0);
        result = 31 * result + (patchBasedir != null ? patchBasedir.hashCode() : 0);
        result = 31 * result + (backupDir != null ? backupDir.hashCode() : 0);
        result = 31 * result + (supportedDistroTypes != null ? supportedDistroTypes.hashCode() : 0);
        result = 31 * result + (phasesToExecute != null ? phasesToExecute.hashCode() : 0);
        result = 31 * result + (isDistributionExploded ? 1 : 0);
        return result;
    }

    @Override
    public String toString() {
        return "ClientPatcherConfig{" +
                "distributionRoot=" + distributionRoot +
                ", distributionType=" + distributionType +
                ", product=" + product +
                ", patchBasedir=" + patchBasedir +
                ", backupBaseDir=" + backupDir +
                ", supportedDistroTypes=" + supportedDistroTypes +
                ", phasesToExecute=" + phasesToExecute +
                ", isDistributionExploded=" + isDistributionExploded +
                '}';
    }

    public void validate() {
        // we can't assume the distribution root is directory, because that is not the case for some of the distribution types
        // for example for individual WAS8 wars -- those are zipped files
        checkDistributionRootExists();
        checkDistributionTypeIsSupported();
    }

    private void checkDistributionRootExists() {
        if (!distributionRoot.exists()) {
            throw new InvalidDistributionRootException(
                    "Specified distribution root '" + distributionRoot + "' does not exist!");
        }
    }

    private void checkDistributionTypeIsSupported() {
        if (!supportedDistroTypes.contains(distributionType)) {
            throw new UnsupportedDistributionTypeException("Distribution type '" + distributionType + "' is not supported! " +
                    "This usually means that the type is known, but it is not supposed to be used for selected product.");
        }
    }

}
