package org.jboss.tools.central.internal;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

import java.io.File;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Collection;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.jboss.tools.central.JBossCentralActivator;
import org.jboss.tools.central.ShowJBossCentral;
import org.jboss.tools.central.test.Activator;
import org.junit.AfterClass;
import org.junit.BeforeClass;
import org.junit.Test;
import org.osgi.framework.Bundle;

public class CentralHelperTest {
	
	private static String openCentral;
	
	@BeforeClass
	public static void beforeClass() {
		openCentral = System.getProperty(ShowJBossCentral.ORG_JBOSS_TOOLS_CENTRAL_DONOTSHOW);
		System.setProperty(ShowJBossCentral.ORG_JBOSS_TOOLS_CENTRAL_DONOTSHOW, Boolean.TRUE.toString());
		System.setProperty("usage_reporting_enabled", Boolean.FALSE.toString());
	}
	
	@AfterClass
	public static void afterClass() {
		if (openCentral != null) {
			System.setProperty(ShowJBossCentral.ORG_JBOSS_TOOLS_CENTRAL_DONOTSHOW, openCentral);
		} else {
			System.clearProperty(ShowJBossCentral.ORG_JBOSS_TOOLS_CENTRAL_DONOTSHOW);
			
		}
	}
	
	@Test
	public void testGetCentralSyspropUrl() throws Exception {
		String url;
		try {
			url = "/foo/bar/central";
			System.setProperty(CentralHelper.JBOSS_CENTRAL_WEBPAGE_URL_KEY, url);
			assertEquals(url+"/index.html", CentralHelper.getCentralUrl(null));

			url = "file:///foo/bar/central/index.html";
			System.setProperty(CentralHelper.JBOSS_CENTRAL_WEBPAGE_URL_KEY, url);
			assertEquals(url, CentralHelper.getCentralUrl(null));

		
			url = "http://central.jboss.org/";
			System.setProperty(CentralHelper.JBOSS_CENTRAL_WEBPAGE_URL_KEY, url);
			assertEquals(url+"index.html", CentralHelper.getCentralUrl(null));
		} finally {
			System.clearProperty(CentralHelper.JBOSS_CENTRAL_WEBPAGE_URL_KEY);
		}
	}

	@Test
	public void testGetCentralUrlZip() throws Exception {
		Path zip = Paths.get("test-resources", "central.zip");
		String url = zip.toUri().toString();
		String sha1 = "566cafe";
		String expectedEnd = ".metadata"+File.separator+".plugins"+File.separator+"org.jboss.tools.central"+File.separator+"central"+File.separator+sha1+File.separator+"index.html";
		IProgressMonitor monitor = new NullProgressMonitor();
		try {
			String resolvedUrl = CentralHelper.getCentralUrl(url, monitor);
			assertTrue("Unexpected resolvedUrl" + resolvedUrl, resolvedUrl.endsWith(expectedEnd));
			
			//different url, same file, returns same sha1 based url
			zip = Paths.get("test-resources", "same-central.zip");
		    url = zip.toUri().toString();
		    String resolvedUrl2 = CentralHelper.getCentralUrl(url, monitor);
			
		    assertEquals(resolvedUrl, resolvedUrl2);
			
		} finally {
			System.clearProperty(CentralHelper.JBOSS_CENTRAL_WEBPAGE_URL_KEY);
		}
	}
	
	@Test
	public void testGetCentralUrlWithSpacesZip() throws Exception {
		Path zip = Paths.get("test-resources", "folder with spaces", "central.zip");
		String url = zip.toUri().toString();
		String sha1 = "566cafe";
		String expectedEnd = ".metadata"+File.separator+".plugins"+File.separator+"org.jboss.tools.central"+File.separator+"central"+File.separator+sha1+File.separator+"index.html";
		IProgressMonitor monitor = new NullProgressMonitor();
		try {
			String resolvedUrl = CentralHelper.getCentralUrl(url, monitor);
			assertTrue("Unexpected resolvedUrl" + resolvedUrl, resolvedUrl.endsWith(expectedEnd));
		} finally {
			System.clearProperty(CentralHelper.JBOSS_CENTRAL_WEBPAGE_URL_KEY);
		}
	}

	@Test
	public void testGetEmbeddedFilePathWithSpaces() throws Exception {
		String value = "test-resources/folder with spaces/central.zip";
		Bundle bundle = Activator.getDefault().getBundle();
		Path p = CentralHelper.getEmbeddedFilePath(bundle, value);
		assertTrue("Can't find " + p, Files.exists(p));
	}

	@Test
	public void testGetInstalledBundleIds() {
		Collection<String> bundleIds = CentralHelper.getInstalledBundleIds();
		assertTrue(bundleIds.contains(JBossCentralActivator.PLUGIN_ID));
	}
	
	//TODO setup jetty server to serve remote URL and test caching capabilities
}
