/*******************************************************************************
 * Copyright (c) 2007 Red Hat, Inc.
 * Distributed under license by Red Hat, Inc. All rights reserved.
 * This program is made available under the terms of the
 * Eclipse Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Red Hat, Inc. - initial API and implementation
 ******************************************************************************/
package com.jboss.jbds.product.intro.universal;

import java.io.IOException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.StringTokenizer;

import org.eclipse.core.runtime.FileLocator;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProduct;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Preferences;
import org.eclipse.help.internal.util.ProductPreferences;
import org.eclipse.help.internal.util.SequenceResolver;
import org.eclipse.ui.internal.intro.impl.model.ExtensionMap;
import org.eclipse.ui.internal.intro.universal.CustomizeAction;
import org.eclipse.ui.internal.intro.universal.ExtensionData;
import org.eclipse.ui.internal.intro.universal.IUniversalIntroConstants;
import org.eclipse.ui.internal.intro.universal.IntroData;
import org.eclipse.ui.internal.intro.universal.PageData;
import org.eclipse.ui.internal.intro.universal.UniversalIntroPlugin;
import org.eclipse.ui.internal.intro.universal.contentdetect.ContentDetector;
import org.eclipse.ui.internal.intro.universal.util.ImageUtil;
import org.eclipse.ui.internal.intro.universal.util.PreferenceArbiter;
import org.eclipse.ui.intro.IIntroSite;
import org.eclipse.ui.intro.config.IntroConfigurer;
import org.eclipse.ui.intro.config.IntroElement;
import org.osgi.framework.Bundle;

public class JBDSUniversalIntroConfigurer extends IntroConfigurer implements IUniversalIntroConstants {
	// Page ids
	String JBDS_ID_ROOT = ID_ROOT;
	String JBDS_ID_STANDBY = ID_STANDBY;
	String JBDS_ID_GETTINGSTARTED = "gettingstarted";
	String JBDS_ID_CREATENEW = "createnew";
	String JBDS_ID_PERSPECTIVES = "perspectives";
	String JBDS_ID_WEBRESOURCES = "webresources";
	String JBDS_ID_WORKBENCH = ID_WORKBENCH;
	
	//Theme property constants
	String JBDS_LAUNCHBAR_GETTINGSTARTED_ICON = "launchbarGettingStartedIcon"; //$NON-NLS-1$
	String JBDS_LAUNCHBAR_CREATENEW_ICON = "launchbarCreateNewIcon"; //$NON-NLS-1$
	String JBDS_LAUNCHBAR_PERSPECTIVES_ICON = "launchbarPerspectivesIcon"; //$NON-NLS-1$
	String JBDS_LAUNCHBAR_WEBRESOURCES_ICON = LAUNCHBAR_WEBRESOURCES_ICON; 
	String HIGH_CONTRAST_PREFIX = "highContrast-"; //$NON-NLS-1$
	String HIGH_CONTRAST_NAV_PREFIX = "highContrastNav-"; //$NON-NLS-1$
	String HIGH_CONTRAST = "high-contrast"; //$NON-NLS-1$

	
	private IntroData primaryIntroData;
	private IntroData[] secondaryIntroData;
	private SequenceResolver sequenceResolver;

	public JBDSUniversalIntroConfigurer() {
		loadData();
	}
	
	public String getVariable(String variableName) {
		if (variableName.equals(HIGH_CONTRAST)) {
			boolean highContrast = ImageUtil.isHighContrast();
			if (highContrast)
				return variableName;
			else
				return ""; //$NON-NLS-1$
		}
		IProduct product = Platform.getProduct();
		if (product != null) {
			String value = getProductProperty(product, variableName);
			if (value != null) {
				value = resolveVariable(product.getDefiningBundle(), value);
				return value;
			}
			if (variableName.startsWith(VAR_INTRO_DESCRIPTION_PREFIX))
				return ""; //$NON-NLS-1$
			Preferences prefs = UniversalIntroPlugin.getDefault()
					.getPluginPreferences();
			String key = product.getId() + "_" + variableName; //$NON-NLS-1$
			value = prefs.getString(key);
			if (value.length() == 0) {
				key = variableName;
				value = prefs.getString(key);
			}
			if (value.length() > 0)
				value = resolveVariable(product.getDefiningBundle(), value);
			else {
				value = getThemeProperty(variableName);
			}
			return value;
		}
		return null;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.ui.intro.config.IntroConfigurer#getMixinStyle(java.lang.String)
	 */
	public String getMixinStyle(String pageId, String extensionId) {
		if (primaryIntroData != null) {
			int importance = getImportance(primaryIntroData, pageId, extensionId);
			if (importance >= 0) {
				return ExtensionData.IMPORTANCE_STYLE_TABLE[importance];
			}
		}
		int[] importanceRefs = new int[ExtensionData.IMPORTANCE_TABLE.length];
		for (int i=0;i<secondaryIntroData.length;++i) {
			IntroData data = secondaryIntroData[i];
			int importance = getImportance(data, pageId, extensionId);
			if (importance >= 0) {
				++importanceRefs[importance];
			}
		}
		int maxIndex = 0;
		for (int i=1;i<importanceRefs.length;++i) {
			if (importanceRefs[i] > importanceRefs[maxIndex]) {
				maxIndex = i;
			}
		}
		if (importanceRefs[maxIndex] > 0) {
			return ExtensionData.IMPORTANCE_STYLE_TABLE[maxIndex];
		}
		return null;
	}

	/*
	 * Returns the given extension's importance as specified by the
	 * given intro data.
	 */
	private int getImportance(IntroData data, String pageId, String extensionId) {
		String pluginId = ExtensionMap.getInstance().getPluginId(extensionId);
		if (ContentDetector.isNew(pluginId)) {
			updateStartPage(pageId);
			return ExtensionData.NEW;
		}
		PageData pdata = data.getPage(pageId);
		if (pdata != null) {
			ExtensionData ed = pdata.findExtension(extensionId, false);
			if (ed != null) {
				return ed.getImportance();
			}
		}
		return -1;
	}

	/*
	 * Modify the start page if this is a root page and it's position
	 * in the root page list is earlier than the current start page
	 */
	private void updateStartPage(String pageId) {
		String currentStartPage = ExtensionMap.getInstance().getStartPage();
		String ids = getVariable(VAR_INTRO_ROOT_PAGES);
		if (ids != null) {
			StringTokenizer stok = new StringTokenizer(ids, ","); //$NON-NLS-1$
			while (stok.hasMoreTokens()) {
				String id = stok.nextToken().trim();
				if (id.equals(pageId)) {
					ExtensionMap.getInstance().setStartPage(pageId);
					return;
				}
				if (id.equals(currentStartPage)) {
					// The current start page has higher priority than the new page
					return;
				}
			}
		}
	}

	private String resolveVariable(Bundle bundle, String value) {
		if (value != null) {
			String path = null;
			if (value.startsWith("intro:")) { //$NON-NLS-1$
				bundle = UniversalIntroPlugin.getDefault().getBundle();
				path = value.substring(6);
			} else if (value.startsWith("product:")) { //$NON-NLS-1$
				path = value.substring(8);
			} else
				return value;
			try {
				URL url = bundle.getEntry(path);
				if (url != null) {
					URL localURL = FileLocator.toFileURL(url);
					return localURL.toString();
				}
			} catch (IOException e) {
				return value;
			}
		}
		return null;
	}


	private String getProductProperty(IProduct product, String variableName) {
		String value = product.getProperty(variableName);
		if (value == null) {
			// return default values
			if (variableName.equals(VAR_INTRO_BACKGROUND_IMAGE))
				return "css/graphics/root/welcomebckgrd.jpg"; //$NON-NLS-1$
		}
		return value;
	}

	public IntroElement[] getGroupChildren(String pageId, String groupId) {
		if (pageId.equals(ID_ROOT)) {
			if (groupId.equals(DIV_PAGE_LINKS))
				return getRootPageLinks(false);
			if (groupId.equals(DIV_ACTION_LINKS))
				return getRootPageActionLinks(false);
		} else if (pageId.equals(ID_STANDBY)) {
			if (groupId.equals(DIV_PAGE_LINKS))
				return getRootPageLinks(true);
			if (groupId.equals(DIV_ACTION_LINKS))
				return getRootPageActionLinks(true);
		} else {
			if (groupId.equals(DIV_PAGE_LINKS))
				return getNavLinks(pageId);
			if (groupId.equals(DIV_LAYOUT_TOP_LEFT)
					|| groupId.equals(DIV_LAYOUT_TOP_RIGHT)
					|| groupId.equals(DIV_LAYOUT_BOTTOM_LEFT)
					|| groupId.equals(DIV_LAYOUT_BOTTOM_RIGHT))
				return getContent(pageId, groupId);
		}
		return new IntroElement[0];
	}

	public IntroElement[] getLaunchBarShortcuts() {
		ArrayList links = new ArrayList();
		String ids = getVariable(VAR_INTRO_ROOT_PAGES);
		if (ids != null) {
			StringTokenizer stok = new StringTokenizer(ids, ","); //$NON-NLS-1$
			while (stok.hasMoreTokens()) {
				String id = stok.nextToken().trim();
				IntroElement page = createLaunchBarShortcut(id);
				if (page != null)
					links.add(page);
			}
		}
		return (IntroElement[]) links.toArray(new IntroElement[links.size()]);
	}

	private IntroElement[] getRootPageLinks(boolean standby) {
		ArrayList links = new ArrayList();
		String ids = getVariable(VAR_INTRO_ROOT_PAGES);
		if (ids != null) {
			StringTokenizer stok = new StringTokenizer(ids, ","); //$NON-NLS-1$
			while (stok.hasMoreTokens()) {
				String id = stok.nextToken().trim();
				IntroElement page = createRootPageLink(id, standby);
				if (page != null)
					links.add(page);
			}
		}
		String wb = getVariable(VAR_WORKBENCH_AS_ROOT_LINK);
		if (wb!=null && wb.equalsIgnoreCase("true")) { //$NON-NLS-1$
			IntroElement page = createRootPageLink(ID_WORKBENCH, standby);
			if (page !=null)
				links.add(page);
		}
		return (IntroElement[]) links.toArray(new IntroElement[links.size()]);
	}

	private IntroElement[] getRootPageActionLinks(boolean standby) {
		String wb = getVariable(VAR_WORKBENCH_AS_ROOT_LINK);
		if (wb==null || !wb.equalsIgnoreCase("true")) { //$NON-NLS-1$
			IntroElement page = createRootPageLink(ID_WORKBENCH, standby);
			if (page !=null)
				return new IntroElement[] { page };
		}
		return new IntroElement [0];
	}

	private IntroElement[] getNavLinks(String pageId) {
		ArrayList links = new ArrayList();
		String ids = getVariable(VAR_INTRO_ROOT_PAGES);		
		if (ImageUtil.isHighContrast()) {
			ids = ids + ',' + IUniversalIntroConstants.ID_WORKBENCH;
		}
		if (ids != null) {
			StringTokenizer stok = new StringTokenizer(ids, ","); //$NON-NLS-1$
			int [] counter = new int [1];
			while (stok.hasMoreTokens()) {
				String id = stok.nextToken().trim();
				IntroElement page = createNavLink(id, pageId, counter);
				if (page != null)
					links.add(page);
			}
		}

		return (IntroElement[]) links.toArray(new IntroElement[links.size()]);
	}

	private IntroElement createRootPageLink(String id, boolean standby) {

		if (id.equals(JBDS_ID_GETTINGSTARTED))
			return createRootLink(
					Messages.SharedIntroConfigurer_gettingstarted_name,
					createPageURL(id, standby),
					id,
					"gettingstarted_img", getThemeProperty(JBDS_LAUNCHBAR_GETTINGSTARTED_ICON), Messages.SharedIntroConfigurer_gettingstarted_alt, //$NON-NLS-1$ //$NON-NLS-2$
					Messages.SharedIntroConfigurer_gettingstarted_tooltip, "left"); //$NON-NLS-1$
		if (id.equals(JBDS_ID_CREATENEW))
			return createRootLink(
					Messages.SharedIntroConfigurer_createnew_name,
					createPageURL(id, standby),
					id,
					"createnew_img", getThemeProperty(JBDS_LAUNCHBAR_CREATENEW_ICON), Messages.SharedIntroConfigurer_createnew_alt, //$NON-NLS-1$ //$NON-NLS-2$
					Messages.SharedIntroConfigurer_createnew_tooltip, "left"); //$NON-NLS-1$
		if (id.equals(JBDS_ID_PERSPECTIVES))
			return createRootLink(
					Messages.SharedIntroConfigurer_perspectives_name,
					createPageURL(id, standby),
					id,
					"perspectives_img", getThemeProperty(JBDS_LAUNCHBAR_PERSPECTIVES_ICON), Messages.SharedIntroConfigurer_perspectives_alt, //$NON-NLS-1$ //$NON-NLS-2$
					Messages.SharedIntroConfigurer_perspectives_tooltip, "left"); //$NON-NLS-1$
		if (id.equals(JBDS_ID_WEBRESOURCES))
			return createRootLink(
					Messages.SharedIntroConfigurer_webresources_name,
					createPageURL(id, standby),
					id,
					"webresources_img", getThemeProperty(JBDS_LAUNCHBAR_WEBRESOURCES_ICON), //$NON-NLS-1$ //$NON-NLS-2$
					Messages.SharedIntroConfigurer_webresources_alt,
					Messages.SharedIntroConfigurer_webresources_tooltip,
					"right"); //$NON-NLS-1$
		if (id.equals(JBDS_ID_WORKBENCH))
			return createRootLink(
					Messages.SharedIntroConfigurer_workbench_name,
					"http://org.eclipse.ui.intro/switchToLaunchBar", //$NON-NLS-1$
					id,
					"workbench_img", null, //$NON-NLS-1$ //$NON-NLS-2$
					Messages.SharedIntroConfigurer_workbench_alt,
					Messages.SharedIntroConfigurer_workbench_tooltip,
					"right"); //$NON-NLS-1$
		return null;
	}

	private IntroElement createNavLink(String id, String pageId, int [] counter) {
		if (id.equals(JBDS_ID_GETTINGSTARTED))
			return createNavLink(Messages.SharedIntroConfigurer_gettingstarted_nav,
					createPageURL(id, false), id, "left nav_link"+(++counter[0])); //$NON-NLS-1$ 
		if (id.equals(JBDS_ID_CREATENEW))
			return createNavLink(Messages.SharedIntroConfigurer_createnew_nav,
					createPageURL(id, false), id, "left  nav_link"+(++counter[0])); //$NON-NLS-1$
		if (id.equals(JBDS_ID_PERSPECTIVES))
			return createNavLink(Messages.SharedIntroConfigurer_perspectives_nav,
					createPageURL(id, false), id, "left nav_link"+(++counter[0])); //$NON-NLS-1$
		if (id.equals(ID_WEBRESOURCES))
			return createNavLink(
					Messages.SharedIntroConfigurer_webresources_nav,
					createPageURL(id, false), id, "right nav_link"+(++counter[0])); //$NON-NLS-1$
		if (id.equals(ID_WORKBENCH))
			return createNavLink(
					Messages.SharedIntroConfigurer_workbench_name,
					"http://org.eclipse.ui.intro/switchToLaunchBar", //$NON-NLS-1$
					id, 
					"right nav_link"+(++counter[0])); //$NON-NLS-1$
		return null;
	}

	private String createPageURL(String id, boolean standby) {
		String url = "http://org.eclipse.ui.intro/showPage?id=" + id; //$NON-NLS-1$
//		if (standby)
//			url += "&standby=false"; //$NON-NLS-1$
		return url;
	}

	private IntroElement createLaunchBarShortcut(String id) {
		if (id.equals(JBDS_ID_GETTINGSTARTED))
			return createShortcutLink(
					getThemeProperty(JBDS_LAUNCHBAR_GETTINGSTARTED_ICON), Messages.SharedIntroConfigurer_gettingstarted_nav, 
					id);
		if (id.equals(JBDS_ID_CREATENEW))
			return createShortcutLink(
					getThemeProperty(JBDS_LAUNCHBAR_CREATENEW_ICON), Messages.SharedIntroConfigurer_createnew_nav, 
					id);
		if (id.equals(JBDS_ID_PERSPECTIVES))
			return createShortcutLink(
					getThemeProperty(JBDS_LAUNCHBAR_PERSPECTIVES_ICON), Messages.SharedIntroConfigurer_perspectives_nav, 
					id);
		if (id.equals(JBDS_ID_WEBRESOURCES))
			return createShortcutLink(
					getThemeProperty(JBDS_LAUNCHBAR_WEBRESOURCES_ICON), Messages.SharedIntroConfigurer_webresources_nav, 
					id);
		return null;
	}

	private IntroElement createRootLink(String name, String url, String id,
			String imgId, String imgSrc, String imgAlt, String imgText,
			String styleId) {
		IntroElement element = new IntroElement("link"); //$NON-NLS-1$
		element.setAttribute("label", name); //$NON-NLS-1$
		element.setAttribute("url", url); //$NON-NLS-1$
		element.setAttribute("id", id); //$NON-NLS-1$
		element.setAttribute("style-id", styleId);//$NON-NLS-1$
		IntroElement img = new IntroElement("img"); //$NON-NLS-1$
		img.setAttribute("id", imgId); //$NON-NLS-1$
		img.setAttribute("style-id", "content-img"); //$NON-NLS-1$ //$NON-NLS-2$
//		img.setAttribute("src", imgSrc); //$NON-NLS-1$
		boolean highContrast = ImageUtil.isHighContrast();
		if (highContrast) {
			String key = HIGH_CONTRAST_PREFIX+id;
			String value = getVariable(key);
			if (value!=null)
				img.setAttribute("src", value); //$NON-NLS-1$
		}
		img.setAttribute("alt", imgAlt); //$NON-NLS-1$
		IntroElement text = new IntroElement("text"); //$NON-NLS-1$
		text.setValue(imgText);
		element.addChild(img);
		element.addChild(text);
		return element;
	}

	private IntroElement createNavLink(String label, String url, String id,
			String styleId) {
		IntroElement element = new IntroElement("link"); //$NON-NLS-1$
		element.setAttribute("label", label); //$NON-NLS-1$
		element.setAttribute("url", url); //$NON-NLS-1$
		element.setAttribute("id", id); //$NON-NLS-1$
		boolean highContrast = ImageUtil.isHighContrast();
		if (highContrast) {
			IntroElement img = new IntroElement("img"); //$NON-NLS-1$
			img.setAttribute("style-id", "content-img"); //$NON-NLS-1$ //$NON-NLS-2$
			String key = HIGH_CONTRAST_NAV_PREFIX+id;
			String value = getVariable(key);
			if (value!=null)
				img.setAttribute("src", value); //$NON-NLS-1$
			img.setAttribute("alt", label); //$NON-NLS-1$
			element.addChild(img);		
			styleId += " "+HIGH_CONTRAST; //$NON-NLS-1$
		}
		element.setAttribute("style-id", styleId); //$NON-NLS-1$
		return element;
	}
	
	private IntroElement createShortcutLink(String icon, String tooltip,
			String id) {
		IntroElement element = new IntroElement("shortcut"); //$NON-NLS-1$
		element.setAttribute("icon", icon); //$NON-NLS-1$
		element.setAttribute("tooltip", tooltip); //$NON-NLS-1$
		element.setAttribute("url", createPageURL(id, false)); //$NON-NLS-1$
		return element;
	}

	private void loadData() {
		// load the active product's intro data first
		IProduct product = Platform.getProduct();
		if (product != null) {
			String dataFile = getVariable(VAR_INTRO_DATA);
			if (dataFile != null) {
				primaryIntroData = new IntroData(product.getId(), dataFile, true);
			}
		}
		List result = new ArrayList();
		Properties[] prefs = ProductPreferences.getProductPreferences(false);
		for (int i=0;i<prefs.length;++i) {
			String key = UniversalIntroPlugin.PLUGIN_ID + '/' + VAR_INTRO_DATA;
			String dataFile = prefs[i].getProperty(key);
			if (dataFile != null) {
				String pluginId = ProductPreferences.getPluginId(prefs[i]);
				Bundle bundle = Platform.getBundle(pluginId);
				if (bundle != null) {
					String pid = ProductPreferences.getProductId(prefs[i]);
					dataFile = resolveVariable(bundle, dataFile);
					result.add(new IntroData(pid, dataFile, false));
				}
			}
		}
		secondaryIntroData = (IntroData[])result.toArray(new IntroData[result.size()]);
	}

	private IntroElement[] getContent(String pageId, String groupId) {
		List result = new ArrayList();
		if (!ContentDetector.getNewContributors().isEmpty()) {
			IntroElement fallback = new IntroElement("anchor"); //$NON-NLS-1$
			fallback.setAttribute("id", NEW_CONTENT_ANCHOR); //$NON-NLS-1$
			result.add(fallback);
		}
		List anchors = getAnchors(pageId, groupId);
		if (anchors != null) {
			result.addAll(anchors);
		}
		IntroElement fallback = new IntroElement("anchor"); //$NON-NLS-1$
		fallback.setAttribute("id", DEFAULT_ANCHOR); //$NON-NLS-1$
		result.add(fallback);
		return (IntroElement[]) result.toArray(new IntroElement[result.size()]);
	}
	
	private List getAnchors(String pageId, String groupId) {
		List primaryAnchors = null;
		if (primaryIntroData != null) {
			primaryAnchors = getAnchors(primaryIntroData, pageId, groupId);
		}
		if (primaryAnchors == null) {
			primaryAnchors = Collections.EMPTY_LIST;
		}
		List secondaryAnchorsList = new ArrayList();
		for (int i=0;i<secondaryIntroData.length;++i) {
			IntroData idata = secondaryIntroData[i];
			List anchors = getAnchors(idata, pageId, groupId);
			if (anchors != null) {
				secondaryAnchorsList.add(anchors);
			}
		}
		List[] secondaryAnchors = (List[])secondaryAnchorsList.toArray(new List[secondaryAnchorsList.size()]);
		if (sequenceResolver == null) {
			sequenceResolver = new SequenceResolver();
		}
		return sequenceResolver.getSequence(primaryAnchors, secondaryAnchors);
	}
	
	private List getAnchors(IntroData data, String pageId, String groupId) {
		PageData pdata = data.getPage(pageId);
		if (pdata != null) {
			List anchors = new ArrayList();
			pdata.addAnchors(anchors, groupId);
			return anchors;
		}
		return null;
	}

	public String resolvePath(String extensionId, String path) {
		boolean extensionRelativePath = false;
		IPath ipath = new Path(path);
		String pageId = ipath.segment(0);
		String s2 = ipath.segment(1);
		if (s2.startsWith("@") && s2.length() > 1) { //$NON-NLS-1$
			extensionId = s2.substring(1);
		}
		if (!s2.equals("@")) { //$NON-NLS-1$
			extensionRelativePath = true;
		}
		if (!isHidden(extensionId, pageId)) {
			String resolvedPath = resolveExtensionPath(extensionId, pageId);
			if (resolvedPath != null) {
				if (extensionRelativePath) {
					IPath p2 = new Path(resolvedPath);
					IPath p1 = ipath.removeFirstSegments(2);
					resolvedPath = p2.removeLastSegments(1).append(p1).toString();
				}
				return resolvedPath;
			}
			return pageId + DEFAULT_CONTENT_PATH;
		}
		return null;
	}

	private String resolveExtensionPath(String extensionId, String pageId) {
		if (primaryIntroData != null) {
			PageData pdata = primaryIntroData.getPage(pageId);
			if (pdata != null) {
				String path = pdata.resolvePath(extensionId);
				if (path != null) {
					return path;
				}
			}
		}
		PreferenceArbiter arbiter = new PreferenceArbiter();
		for (int i=0;i<secondaryIntroData.length;++i) {
			IntroData idata = secondaryIntroData[i];
			PageData pdata = idata.getPage(pageId);
			if (pdata != null) {
				arbiter.consider(pdata.resolvePath(extensionId));
			}
		}
		String path = (String)arbiter.getWinner();
		if (path != null) {
			return path;
		}
		return resolveDefaultPath(pageId, extensionId);
	}
	
	private String resolveDefaultPath(String pageId, String extensionId) {
		String pluginId = ExtensionMap.getInstance().getPluginId(extensionId);
		if (ContentDetector.isNew(pluginId)) {
			return pageId + IUniversalIntroConstants.NEW_CONTENT_PATH;
		}
		if (primaryIntroData != null) {
			PageData pdata = primaryIntroData.getPage(pageId);
			if (pdata != null) {
				String path = pdata.resolveDefaultPath();
				if (path != null) {
					return path;
				}
			}
		}
		PreferenceArbiter arbiter = new PreferenceArbiter();
		for (int i=0;i<secondaryIntroData.length;++i) {
			IntroData idata = secondaryIntroData[i];
			PageData pdata = idata.getPage(pageId);
			if (pdata != null) {
				arbiter.consider(pdata.resolveDefaultPath());
			}
		}
		return (String)arbiter.getWinner();
	}

	private boolean isHidden(String extensionId, String pageId) {
		if (primaryIntroData != null) {
			PageData pdata = primaryIntroData.getPage(pageId);
			if (pdata != null) {
				return pdata.isHidden(extensionId);
			}
		}
		return false;
	}

//	public void init(IIntroSite site, Map themeProperties) {
//		super.init(site, themeProperties);
//		IConfigurationElement element = CustomizeAction.getPageElement();
//		if (element == null)
//			return;
//	}
}
