/**
 * Document-wide object manipulating drag and drop events.
 * When drag is in process, <code>window.drag</code> object of class <code>DnD.Drag</code> holds its state.
 * Draggables should create DnD.Drag instance and pass it to DnD.startDrag call to initialize drag.
 * Dropzones should implement accept method to be able to get drag results.
 * @author Maksim Kaszynski
 */
var DnD = {
	
	CODE_ACCEPT : "accept",
	
	CODE_DEFAULT: "default",
	
	CODE_REJECT: "reject",
	
	/**
	 * Switch mouse observing to DnD mode
	 * @param {DnD.Drag} drag
	 */
	startDrag: function(drag) {
		if (!window.drag) {
			this.init();
			window.drag = drag;
			Event.observe(document, 'mousemove', this.mousemove);
			Event.observe(document, 'mouseup', this.mouseup);
		} else {
			alert('drag in progress');
		}
	},
	
	endDrag: function(event) {
		Event.stopObserving(document, 'mousemove', this.mousemove);
		Event.stopObserving(document, 'mouseup', this.mouseup);
		var drag = window.drag;
		if (drag) {
			window.drag = null;
			var dropzone = drag.dropzone;
			drag.source.endDrag(event, drag);
			if (dropzone) {
				dropzone.onbeforedrag(event, drag);
				if (dropzone.accept(drag)) {
					dropzone.drop(event, drag);
				}
				dropzone.onafterdrag(event);
			}
		}
	},
	
	updateDrag: function(event) {
		var drag = window.drag;
	    var x = Event.pointerX(event);
	    var y = Event.pointerY(event);
		drag.indicator.position(x + 5, y + 14);
		Event.stop(event);
	},
	
	initialized : false,
	
	init: function() {
		if (!this.initialized) {
			this.mousemove = this.updateDrag.bindAsEventListener(this);
			this.mouseup = this.endDrag.bindAsEventListener(this);
			this.initialized = true;
		}	
			
	}
	


};

/**
 * @classDescription
 * Object holding drag state info
 */
DnD.Drag = Class.create();
DnD.Drag.prototype = {
	/**
	 * @constructor
	 * @param {DnD.Draggable} source
	 * @param {DnD.Indicator} indicator
	 */
	initialize: function(source, indicator, type) {
		this.source = source;
		this.indicator = indicator;
		this.type = type;
	},
	params: {},
	dragged: false,
	dropzone: null,
	getParameters: function() {
		var params = {};
		Object.extend(params, this.params);
	
		return params;
	}
};
/**
 * @author Maksim Kaszynski
 * base class for draggable component.
 */
DnD.Draggable = function() {};
DnD.Draggable.prototype = {
	
	/**
	 * @return type of draggable content
	 */
	getContentType: function() {
		return "ZZZZZ";
	},
	/**
	 * implementations are responsible for getting drag indicator
	 * @return DnD.Indicator
	 */
	getIndicator: function() {
		return null;
	},

	startDrag : function(event) {
		var type = this.getContentType();
		var indicator = this.getIndicator();
		var drag = new DnD.Drag(this, indicator, type);
		DnD.startDrag(drag);
		DnD.updateDrag(event);
		this.ondragstart(event, drag);
		if (indicator) {
			indicator.show();
		}
	},
	/**
	 * 
	 * @param {DnD.Drag} drag
	 */
	endDrag: function(event, drag) {
		var indicator = drag.indicator;
		if (indicator) {
			indicator.hide();
		}
		this.ondragend(event, drag);
	},
	/**
	 * cubclasses may define custom behavior
	 * @param {Object} drag
	 */
	ondragstart: function(event, drag) {
		
	},
	
	ondragend: function (event, drag) {
		
	}
};
/**
 * @author Maksim Kaszynski
 *	@classDescription
 * Base class for drop zones
 */
DnD.Dropzone = function(){};
DnD.Dropzone.DROP_TARGET_ID = "dropTargetId";
DnD.Dropzone.prototype = {
	/**
	 * 
	 * @param {Object} drag
	 * @return 
	 */
	accept: function(drag) {
		return this.getAcceptedTypes().indexOf(drag.type) > -1 ;
	},
	getAcceptedTypes: function() {
		return [];
	},
	getTypeMapping: function() {
		return {};
	},
	drop: function(event, drag){
	},		
	getIconCodeForType: function(type) {
		var types = this.getTypeMapping();
		if (type && types) {
			return types[type];
		}
		return null;
	},

	/**
	 * implementations call this method when mouse over them
	 * @param {Object} drag
	 */
	dragEnter: function(event) {
		var drag = window.drag;
		drag.dropzone = this;	
		var icon;
		var indicator = drag.indicator;
		if (this.accept(drag)) {
			icon = this.getIconCodeForType(drag.type);
			if (!indicator.hasTemplate(icon)) {
				icon = DnD.CODE_ACCEPT;
			}
			indicator.accept();
		} else {
			icon = DnD.CODE_REJECT;
			indicator.reject();
		}
		indicator.setLeftCell(icon);
	},
	
	
	/**
	 * Implementations call this method when mouse exits them
	 * @param {Object} drag
	 */
	dragLeave: function(event) {
		var drag = window.drag;
		drag.dropzone = null;
		var indicator = drag.indicator;
		indicator.setLeftCell(DnD.CODE_DEFAULT);
		indicator.leave();
	},
	
	onafterdrag: function(event) {
	},
	
	onbeforedrag: function(event, drag){
	},
	
	ondragenter: function(event) {
		
	},
	ondragexit : function(event) {
		
	}
	
};

