"use strict";
/*********************************************************************
 * Copyright (c) 2019 Red Hat, Inc.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 **********************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const client_node_1 = require("@kubernetes/client-node");
const axios_1 = require("axios");
const cli_ux_1 = require("cli-ux");
const fs = require("fs");
const https = require("https");
const yaml = require("js-yaml");
const lodash_1 = require("lodash");
const net = require("net");
const stream_1 = require("stream");
const constants_1 = require("../constants");
class KubeHelper {
    constructor(flags, context) {
        this.kc = new client_node_1.KubeConfig();
        this.portForwardHelper = new client_node_1.PortForward(this.kc, true);
        this.logHelper = new client_node_1.Log(this.kc);
        if (!context) {
            this.kc.loadFromDefault();
        }
        else {
            this.kc.loadFromString(context);
        }
        if (flags && flags.k8spodwaittimeout) {
            this.podWaitTimeout = parseInt(flags.k8spodwaittimeout, 10);
        }
        else {
            this.podWaitTimeout = 300000;
        }
        if (flags && flags.k8spodreadytimeout) {
            this.podReadyTimeout = parseInt(flags.k8spodreadytimeout, 10);
        }
        else {
            this.podReadyTimeout = 130000;
        }
    }
    deleteAllServices(namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sApi = this.kc.makeApiClient(client_node_1.CoreV1Api);
            try {
                const res = yield k8sApi.listNamespacedService(namespace, true);
                if (res && res.response && res.response.statusCode === 200) {
                    const serviceList = res.body;
                    const options = new client_node_1.V1DeleteOptions();
                    yield serviceList.items.forEach((service) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                        yield k8sApi.deleteNamespacedService(service.metadata.name, namespace, undefined, options);
                    }));
                }
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    getServicesBySelector(labelSelector = '', namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sCoreApi = this.kc.makeApiClient(client_node_1.CoreV1Api);
            try {
                const res = yield k8sCoreApi.listNamespacedService(namespace, true, 'true', undefined, undefined, labelSelector);
                if (res && res.body) {
                    return res.body;
                }
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
            throw new Error('ERR_LIST_SERVICES');
        });
    }
    waitForService(selector, namespace = '', intervalMs = 500, timeoutMs = 30000) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const iterations = timeoutMs / intervalMs;
            for (let index = 0; index < iterations; index++) {
                let currentServices = yield this.getServicesBySelector(selector, namespace);
                if (currentServices && currentServices.items.length > 0) {
                    return;
                }
                yield cli_ux_1.cli.wait(intervalMs);
            }
            throw new Error(`ERR_TIMEOUT: Timeout set to waiting for service ${timeoutMs}`);
        });
    }
    serviceAccountExist(name = '', namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sApi = this.kc.makeApiClient(client_node_1.CoreV1Api);
            try {
                const { body } = yield k8sApi.readNamespacedServiceAccount(name, namespace);
                return this.compare(body, name);
            }
            catch (_a) {
                return false;
            }
        });
    }
    createServiceAccount(name = '', namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sCoreApi = this.kc.makeApiClient(client_node_1.CoreV1Api);
            let sa = new client_node_1.V1ServiceAccount();
            sa.metadata = new client_node_1.V1ObjectMeta();
            sa.metadata.name = name;
            sa.metadata.namespace = namespace;
            try {
                return yield k8sCoreApi.createNamespacedServiceAccount(namespace, sa);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    deleteServiceAccount(name = '', namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sCoreApi = this.kc.makeApiClient(client_node_1.CoreV1Api);
            try {
                const options = new client_node_1.V1DeleteOptions();
                yield k8sCoreApi.deleteNamespacedServiceAccount(name, namespace, undefined, options);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    createServiceAccountFromFile(filePath, namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const yamlServiceAccount = this.safeLoadFromYamlFile(filePath);
            const k8sCoreApi = this.kc.makeApiClient(client_node_1.CoreV1Api);
            try {
                return yield k8sCoreApi.createNamespacedServiceAccount(namespace, yamlServiceAccount);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    replaceServiceAccountFromFile(filePath, namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const yamlServiceAccount = this.safeLoadFromYamlFile(filePath);
            if (!yamlServiceAccount || !yamlServiceAccount.metadata || !yamlServiceAccount.metadata.name) {
                throw new Error(`Service account read from ${filePath} must have name specified.`);
            }
            const k8sCoreApi = this.kc.makeApiClient(client_node_1.CoreV1Api);
            try {
                return yield k8sCoreApi.replaceNamespacedServiceAccount(yamlServiceAccount.metadata.name, namespace, yamlServiceAccount);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    roleExist(name = '', namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sRbacAuthApi = this.kc.makeApiClient(client_node_1.RbacAuthorizationV1Api);
            try {
                const { body } = yield k8sRbacAuthApi.readNamespacedRole(name, namespace);
                return this.compare(body, name);
            }
            catch (_a) {
                return false;
            }
        });
    }
    clusterRoleExist(name = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sRbacAuthApi = this.kc.makeApiClient(client_node_1.RbacAuthorizationV1Api);
            try {
                const { body } = yield k8sRbacAuthApi.readClusterRole(name);
                return this.compare(body, name);
            }
            catch (_a) {
                return false;
            }
        });
    }
    createRoleFromFile(filePath, namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const yamlRole = this.safeLoadFromYamlFile(filePath);
            const k8sRbacAuthApi = this.kc.makeApiClient(client_node_1.RbacAuthorizationV1Api);
            try {
                const res = yield k8sRbacAuthApi.createNamespacedRole(namespace, yamlRole);
                return res.response.statusCode;
            }
            catch (e) {
                if (e.response && e.response.statusCode && e.response.statusCode === 403) {
                    return e.response.statusCode;
                }
                else {
                    throw this.wrapK8sClientError(e);
                }
            }
        });
    }
    replaceRoleFromFile(filePath, namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const yamlRole = this.safeLoadFromYamlFile(filePath);
            const k8sRbacAuthApi = this.kc.makeApiClient(client_node_1.RbacAuthorizationV1Api);
            if (!yamlRole.metadata || !yamlRole.metadata.name) {
                throw new Error(`Role read from ${filePath} must have name specified`);
            }
            try {
                const res = yield k8sRbacAuthApi.replaceNamespacedRole(yamlRole.metadata.name, namespace, yamlRole);
                return res.response.statusCode;
            }
            catch (e) {
                if (e.response && e.response.statusCode && e.response.statusCode === 403) {
                    return e.response.statusCode;
                }
                else {
                    throw this.wrapK8sClientError(e);
                }
            }
        });
    }
    createClusterRoleFromFile(filePath) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const yamlRole = this.safeLoadFromYamlFile(filePath);
            const k8sRbacAuthApi = this.kc.makeApiClient(client_node_1.RbacAuthorizationV1Api);
            try {
                const res = yield k8sRbacAuthApi.createClusterRole(yamlRole);
                return res.response.statusCode;
            }
            catch (e) {
                if (e.response && e.response.statusCode && e.response.statusCode === 403) {
                    return e.response.statusCode;
                }
                else {
                    throw this.wrapK8sClientError(e);
                }
            }
        });
    }
    replaceClusterRoleFromFile(filePath) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const yamlRole = this.safeLoadFromYamlFile(filePath);
            const k8sRbacAuthApi = this.kc.makeApiClient(client_node_1.RbacAuthorizationV1Api);
            if (!yamlRole.metadata || !yamlRole.metadata.name) {
                throw new Error(`Cluster Role read from ${filePath} must have name specified`);
            }
            try {
                const res = yield k8sRbacAuthApi.replaceClusterRole(yamlRole.metadata.name, yamlRole);
                return res.response.statusCode;
            }
            catch (e) {
                if (e.response && e.response.statusCode && e.response.statusCode === 403) {
                    return e.response.statusCode;
                }
                else {
                    throw this.wrapK8sClientError(e);
                }
            }
        });
    }
    deleteRole(name = '', namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sCoreApi = this.kc.makeApiClient(client_node_1.RbacAuthorizationV1Api);
            try {
                const options = new client_node_1.V1DeleteOptions();
                yield k8sCoreApi.deleteNamespacedRole(name, namespace, undefined, options);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    deleteClusterRole(name = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sCoreApi = this.kc.makeApiClient(client_node_1.RbacAuthorizationV1Api);
            try {
                const options = new client_node_1.V1DeleteOptions();
                yield k8sCoreApi.deleteClusterRole(name, undefined, options);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    roleBindingExist(name = '', namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sRbacAuthApi = this.kc.makeApiClient(client_node_1.RbacAuthorizationV1Api);
            try {
                const { body } = yield k8sRbacAuthApi.readNamespacedRoleBinding(name, namespace);
                return this.compare(body, name);
            }
            catch (_a) {
                return false;
            }
        });
    }
    clusterRoleBindingExist(name = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sRbacAuthApi = this.kc.makeApiClient(client_node_1.RbacAuthorizationV1Api);
            try {
                const { body } = yield k8sRbacAuthApi.readClusterRoleBinding(name);
                return this.compare(body, name);
            }
            catch (_a) {
                return false;
            }
        });
    }
    createAdminRoleBinding(name = '', serviceAccount = '', namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sRbacAuthApi = this.kc.makeApiClient(client_node_1.RbacAuthorizationV1Api);
            let rb = new client_node_1.V1RoleBinding();
            rb.metadata = new client_node_1.V1ObjectMeta();
            rb.metadata.name = name;
            rb.metadata.namespace = namespace;
            rb.roleRef = new client_node_1.V1RoleRef();
            rb.roleRef.kind = 'ClusterRole';
            rb.roleRef.name = 'admin';
            let subject = new client_node_1.V1Subject();
            subject.kind = 'ServiceAccount';
            subject.name = serviceAccount;
            subject.namespace = namespace;
            rb.subjects = [subject];
            try {
                return yield k8sRbacAuthApi.createNamespacedRoleBinding(namespace, rb);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    createRoleBindingFromFile(filePath, namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const yamlRoleBinding = this.safeLoadFromYamlFile(filePath);
            const k8sRbacAuthApi = this.kc.makeApiClient(client_node_1.RbacAuthorizationV1Api);
            try {
                return yield k8sRbacAuthApi.createNamespacedRoleBinding(namespace, yamlRoleBinding);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    replaceRoleBindingFromFile(filePath, namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const yamlRoleBinding = this.safeLoadFromYamlFile(filePath);
            if (!yamlRoleBinding.metadata || !yamlRoleBinding.metadata.name) {
                throw new Error(`Role binding read from ${filePath} must have name specified`);
            }
            const k8sRbacAuthApi = this.kc.makeApiClient(client_node_1.RbacAuthorizationV1Api);
            try {
                return yield k8sRbacAuthApi.replaceNamespacedRoleBinding(yamlRoleBinding.metadata.name, namespace, yamlRoleBinding);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    createClusterRoleBinding(name, saName, saNamespace = '', roleName = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const clusterRoleBinding = {
                apiVersion: 'rbac.authorization.k8s.io/v1',
                metadata: {
                    name: `${name}`
                },
                subjects: [
                    {
                        kind: 'ServiceAccount',
                        name: `${saName}`,
                        namespace: `${saNamespace}`
                    }
                ],
                roleRef: {
                    kind: 'ClusterRole',
                    name: `${roleName}`,
                    apiGroup: 'rbac.authorization.k8s.io'
                }
            };
            const k8sRbacAuthApi = this.kc.makeApiClient(client_node_1.RbacAuthorizationV1Api);
            try {
                return yield k8sRbacAuthApi.createClusterRoleBinding(clusterRoleBinding);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    replaceClusterRoleBinding(name, saName, saNamespace = '', roleName = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const clusterRoleBinding = {
                apiVersion: 'rbac.authorization.k8s.io/v1',
                metadata: {
                    name: `${name}`
                },
                subjects: [
                    {
                        kind: 'ServiceAccount',
                        name: `${saName}`,
                        namespace: `${saNamespace}`
                    }
                ],
                roleRef: {
                    kind: 'ClusterRole',
                    name: `${roleName}`,
                    apiGroup: 'rbac.authorization.k8s.io'
                }
            };
            const k8sRbacAuthApi = this.kc.makeApiClient(client_node_1.RbacAuthorizationV1Api);
            try {
                return yield k8sRbacAuthApi.replaceClusterRoleBinding(name, clusterRoleBinding);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    deleteRoleBinding(name = '', namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sRbacAuthApi = this.kc.makeApiClient(client_node_1.RbacAuthorizationV1Api);
            try {
                const options = new client_node_1.V1DeleteOptions();
                return yield k8sRbacAuthApi.deleteNamespacedRoleBinding(name, namespace, undefined, options);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    deleteClusterRoleBinding(name = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sRbacAuthApi = this.kc.makeApiClient(client_node_1.RbacAuthorizationV1Api);
            try {
                const options = new client_node_1.V1DeleteOptions();
                return yield k8sRbacAuthApi.deleteClusterRoleBinding(name, undefined, options);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    getConfigMap(name = '', namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sCoreApi = this.kc.makeApiClient(client_node_1.CoreV1Api);
            try {
                const { body } = yield k8sCoreApi.readNamespacedConfigMap(name, namespace);
                return this.compare(body, name) && body;
            }
            catch (_a) {
                return;
            }
        });
    }
    createConfigMapFromFile(filePath, namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const yamlConfigMap = this.safeLoadFromYamlFile(filePath);
            const k8sCoreApi = this.kc.makeApiClient(client_node_1.CoreV1Api);
            try {
                return yield k8sCoreApi.createNamespacedConfigMap(namespace, yamlConfigMap);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    patchConfigMap(name, patch, namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sCoreApi = this.kc.makeApiClient(PatchedK8sApi);
            try {
                return yield k8sCoreApi.patchNamespacedConfigMap(name, namespace, patch);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    deleteConfigMap(name, namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sCoreApi = this.kc.makeApiClient(client_node_1.CoreV1Api);
            try {
                const options = new client_node_1.V1DeleteOptions();
                yield k8sCoreApi.deleteNamespacedConfigMap(name, namespace, undefined, options);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    readNamespacedPod(podName, namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sCoreApi = this.kc.makeApiClient(client_node_1.CoreV1Api);
            try {
                const res = yield k8sCoreApi.readNamespacedPod(podName, namespace);
                if (res && res.body) {
                    return res.body;
                }
            }
            catch (_a) {
                return;
            }
        });
    }
    podsExistBySelector(selector, namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sCoreApi = this.kc.makeApiClient(client_node_1.CoreV1Api);
            let res;
            try {
                res = yield k8sCoreApi.listNamespacedPod(namespace, true, undefined, undefined, undefined, selector);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
            if (!res || !res.body || !res.body.items) {
                throw new Error(`Get pods by selector "${selector}" returned an invalid response`);
            }
            return (res.body.items.length > 0);
        });
    }
    getPodPhase(selector, namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sCoreApi = this.kc.makeApiClient(client_node_1.CoreV1Api);
            let res;
            try {
                res = yield k8sCoreApi.listNamespacedPod(namespace, true, undefined, undefined, undefined, selector);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
            if (!res || !res.body || !res.body.items) {
                throw new Error(`Get pods by selector "${selector}" returned an invalid response`);
            }
            if (res.body.items.length !== 1) {
                throw new Error(`Get pods by selector "${selector}" returned ${res.body.items.length} pods (1 was expected)`);
            }
            if (!res.body.items[0].status || !res.body.items[0].status.phase) {
                throw new Error(`Get pods by selector "${selector}" returned a pod with an invalid state`);
            }
            return res.body.items[0].status.phase;
        });
    }
    getPodReadyConditionStatus(selector, namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sCoreApi = this.kc.makeApiClient(client_node_1.CoreV1Api);
            let res;
            try {
                res = yield k8sCoreApi.listNamespacedPod(namespace, true, undefined, undefined, undefined, selector);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
            if (!res || !res.body || !res.body.items) {
                throw new Error(`Get pods by selector "${selector}" returned an invalid response`);
            }
            if (res.body.items.length < 1) {
                // No pods found by the specified selector. So, it's not ready.
                return 'False';
            }
            if (res.body.items.length > 1) {
                throw new Error(`Get pods by selector "${selector}" returned ${res.body.items.length} pods (1 was expected)`);
            }
            if (!res.body.items[0].status) {
                throw new Error(`Get pods by selector "${selector}" returned a pod with an invalid state`);
            }
            if (!res.body.items[0].status.conditions || !(res.body.items[0].status.conditions.length > 0)) {
                throw new Error(`Get pods by selector "${selector}" returned a pod with an invalid status.conditions`);
            }
            const conditions = res.body.items[0].status.conditions;
            for (let condition of conditions) {
                if (condition.type === 'Ready') {
                    return condition.status;
                }
            }
            throw new Error(`Get pods by selector "${selector}" returned a pod without a status.condition of type "Ready"`);
        });
    }
    waitForPodPhase(selector, targetPhase, namespace = '', intervalMs = 500, timeoutMs = this.podWaitTimeout) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const iterations = timeoutMs / intervalMs;
            for (let index = 0; index < iterations; index++) {
                let currentPhase = yield this.getPodPhase(selector, namespace);
                if (targetPhase === currentPhase) {
                    return;
                }
                yield cli_ux_1.cli.wait(intervalMs);
            }
            throw new Error(`ERR_TIMEOUT: Timeout set to pod wait timeout ${this.podWaitTimeout}`);
        });
    }
    waitForPodPending(selector, namespace = '', intervalMs = 500, timeoutMs = this.podWaitTimeout) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const iterations = timeoutMs / intervalMs;
            let podExist;
            let currentPhase;
            for (let index = 0; index < iterations; index++) {
                podExist = yield this.podsExistBySelector(selector, namespace);
                if (podExist) {
                    currentPhase = yield this.getPodPhase(selector, namespace);
                    if (currentPhase === 'Pending' || currentPhase === 'Running') {
                        return;
                    }
                    else {
                        throw new Error(`ERR_UNEXPECTED_PHASE: ${currentPhase} (Pending expected) `);
                    }
                }
                yield cli_ux_1.cli.wait(intervalMs);
            }
            throw new Error(`ERR_TIMEOUT: Timeout set to pod wait timeout ${this.podWaitTimeout}. podExist: ${podExist}, currentPhase: ${currentPhase}`);
        });
    }
    waitForPodReady(selector, namespace = '', intervalMs = 500, timeoutMs = this.podReadyTimeout) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const iterations = timeoutMs / intervalMs;
            for (let index = 0; index < iterations; index++) {
                let readyStatus = yield this.getPodReadyConditionStatus(selector, namespace);
                if (readyStatus === 'True') {
                    return;
                }
                if (readyStatus !== 'False') {
                    throw new Error(`ERR_BAD_READY_STATUS: ${readyStatus} (True or False expected) `);
                }
                yield cli_ux_1.cli.wait(intervalMs);
            }
            throw new Error(`ERR_TIMEOUT: Timeout set to pod ready timeout ${this.podReadyTimeout}`);
        });
    }
    waitUntilPodIsDeleted(selector, namespace = '', intervalMs = 500, timeoutMs = this.podReadyTimeout) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const iterations = timeoutMs / intervalMs;
            for (let index = 0; index < iterations; index++) {
                let readyStatus = yield this.getPodReadyConditionStatus(selector, namespace);
                if (readyStatus === 'False') {
                    return;
                }
                if (readyStatus !== 'True') {
                    throw new Error(`ERR_BAD_READY_STATUS: ${readyStatus} (True or False expected) `);
                }
                yield cli_ux_1.cli.wait(intervalMs);
            }
            throw new Error(`ERR_TIMEOUT: Timeout set to pod ready timeout ${this.podReadyTimeout}`);
        });
    }
    deletePod(name, namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            this.kc.loadFromDefault();
            const k8sCoreApi = this.kc.makeApiClient(client_node_1.CoreV1Api);
            const options = new client_node_1.V1DeleteOptions();
            try {
                return yield k8sCoreApi.deleteNamespacedPod(name, namespace, undefined, options);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    // make sure that flag is specified for command that it's invoked
    waitLatestReplica(deploymentName, namespace = '', intervalMs = 500, timeoutMs = this.podWaitTimeout) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const iterations = timeoutMs / intervalMs;
            for (let index = 0; index < iterations; index++) {
                const deployment = yield this.getDeployment(deploymentName, namespace);
                if (!deployment) {
                    throw new Error(`Deployment ${namespace}/${deploymentName} is not found.`);
                }
                const deploymentStatus = deployment.status;
                if (!deploymentStatus) {
                    throw new Error(`Deployment ${namespace}/${deploymentName} does not have any status`);
                }
                if (deploymentStatus.unavailableReplicas && deploymentStatus.unavailableReplicas > 0) {
                    yield cli_ux_1.cli.wait(intervalMs);
                }
                else {
                    return;
                }
            }
            throw new Error(`ERR_TIMEOUT: Timeout set to pod wait timeout ${this.podWaitTimeout}`);
        });
    }
    deploymentExist(name = '', namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sApi = this.kc.makeApiClient(client_node_1.AppsV1Api);
            try {
                const { body } = yield k8sApi.readNamespacedDeployment(name, namespace);
                return this.compare(body, name);
            }
            catch (_a) {
                return false;
            }
        });
    }
    deploymentReady(name = '', namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sApi = this.kc.makeApiClient(client_node_1.AppsV1Api);
            try {
                const res = yield k8sApi.readNamespacedDeployment(name, namespace);
                return (res && res.body &&
                    res.body.status && res.body.status.readyReplicas
                    && res.body.status.readyReplicas > 0);
            }
            catch (_a) {
                return false;
            }
        });
    }
    deploymentStopped(name = '', namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sApi = this.kc.makeApiClient(client_node_1.AppsV1Api);
            try {
                const res = yield k8sApi.readNamespacedDeployment(name, namespace);
                if (res && res.body && res.body.spec && res.body.spec.replicas) {
                    throw new Error(`Deployment '${name}' without replicas in spec is fetched`);
                }
                return res.body.spec.replicas === 0;
            }
            catch (_a) {
                return false;
            }
        });
    }
    isDeploymentPaused(name = '', namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sApi = this.kc.makeApiClient(client_node_1.AppsV1Api);
            try {
                const res = yield k8sApi.readNamespacedDeployment(name, namespace);
                if (!res || !res.body || !res.body.spec) {
                    throw new Error('E_BAD_DEPLOY_RESPONSE');
                }
                return res.body.spec.paused || false;
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    pauseDeployment(name = '', namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sApi = this.kc.makeApiClient(PatchedK8sAppsApi);
            try {
                const patch = {
                    spec: {
                        paused: true
                    }
                };
                yield k8sApi.patchNamespacedDeployment(name, namespace, patch);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    resumeDeployment(name = '', namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sApi = this.kc.makeApiClient(PatchedK8sAppsApi);
            try {
                const patch = {
                    spec: {
                        paused: false
                    }
                };
                yield k8sApi.patchNamespacedDeployment(name, namespace, patch);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    scaleDeployment(name = '', namespace = '', replicas) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sAppsApi = this.kc.makeApiClient(PatchedK8sAppsApi);
            const patch = {
                spec: {
                    replicas
                }
            };
            let res;
            try {
                res = yield k8sAppsApi.patchNamespacedDeploymentScale(name, namespace, patch);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
            if (!res || !res.body) {
                throw new Error('Patch deployment scale returned an invalid response');
            }
        });
    }
    createDeployment(name, image, serviceAccount, pullPolicy, configMapEnvSource, namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sAppsApi = this.kc.makeApiClient(client_node_1.AppsV1Api);
            let deployment = new client_node_1.V1Deployment();
            deployment.metadata = new client_node_1.V1ObjectMeta();
            deployment.metadata.name = name;
            deployment.metadata.namespace = namespace;
            deployment.spec = new client_node_1.V1DeploymentSpec();
            deployment.spec.selector = new client_node_1.V1LabelSelector();
            deployment.spec.selector.matchLabels = { app: name };
            deployment.spec.template = new client_node_1.V1PodTemplateSpec();
            deployment.spec.template.metadata = new client_node_1.V1ObjectMeta();
            deployment.spec.template.metadata.name = name;
            deployment.spec.template.metadata.labels = { app: name };
            deployment.spec.template.spec = new client_node_1.V1PodSpec();
            deployment.spec.template.spec.serviceAccountName = serviceAccount;
            let opContainer = new client_node_1.V1Container();
            opContainer.name = name;
            opContainer.image = image;
            opContainer.imagePullPolicy = pullPolicy;
            let envFromSource = new client_node_1.V1EnvFromSource();
            envFromSource.configMapRef = new client_node_1.V1ConfigMapEnvSource();
            envFromSource.configMapRef.name = configMapEnvSource;
            opContainer.envFrom = [envFromSource];
            deployment.spec.template.spec.containers = [opContainer];
            try {
                return yield k8sAppsApi.createNamespacedDeployment(namespace, deployment);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    createDeploymentFromFile(filePath, namespace = '', containerImage = '', containerIndex = 0) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const yamlDeployment = this.safeLoadFromYamlFile(filePath);
            if (containerImage) {
                yamlDeployment.spec.template.spec.containers[containerIndex].image = containerImage;
            }
            const k8sAppsApi = this.kc.makeApiClient(client_node_1.AppsV1Api);
            try {
                return yield k8sAppsApi.createNamespacedDeployment(namespace, yamlDeployment);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    replaceDeploymentFromFile(filePath, namespace = '', containerImage = '', containerIndex = 0) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const yamlDeployment = this.safeLoadFromYamlFile(filePath);
            if (containerImage) {
                yamlDeployment.spec.template.spec.containers[containerIndex].image = containerImage;
            }
            if (!yamlDeployment.metadata || !yamlDeployment.metadata.name) {
                throw new Error(`Deployment read from ${filePath} must have name specified`);
            }
            // updating restartedAt to make sure that rollout will be restarted
            let annotations = yamlDeployment.spec.template.metadata.annotations;
            if (!annotations) {
                annotations = {};
                yamlDeployment.spec.template.metadata.annotations = annotations;
            }
            annotations['kubectl.kubernetes.io/restartedAt'] = new Date().toISOString();
            const k8sAppsApi = this.kc.makeApiClient(client_node_1.AppsV1Api);
            try {
                return yield k8sAppsApi.replaceNamespacedDeployment(yamlDeployment.metadata.name, namespace, yamlDeployment);
            }
            catch (e) {
                if (e.response && e.response.body && e.response.body.message && e.response.body.message.toString().endsWith('field is immutable')) {
                    try {
                        yield k8sAppsApi.deleteNamespacedDeployment(yamlDeployment.metadata.name, namespace);
                        return yield k8sAppsApi.createNamespacedDeployment(namespace, yamlDeployment);
                    }
                    catch (e) {
                        throw this.wrapK8sClientError(e);
                    }
                }
                throw this.wrapK8sClientError(e);
            }
        });
    }
    deleteAllDeployments(namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sAppsApi = this.kc.makeApiClient(client_node_1.AppsV1Api);
            try {
                yield k8sAppsApi.deleteCollectionNamespacedDeployment(namespace);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    getDeploymentsBySelector(labelSelector = '', namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sAppsApi = this.kc.makeApiClient(client_node_1.AppsV1Api);
            try {
                const res = yield k8sAppsApi.listNamespacedDeployment(namespace, true, 'true', undefined, undefined, labelSelector);
                if (res && res.body) {
                    return res.body;
                }
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
            throw new Error('ERR_LIST_NAMESPACES');
        });
    }
    getDeployment(name, namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sAppsApi = this.kc.makeApiClient(client_node_1.AppsV1Api);
            try {
                const res = yield k8sAppsApi.readNamespacedDeployment(name, namespace);
                if (res && res.body) {
                    return res.body;
                }
            }
            catch (error) {
                if (error.response && error.response.statusCode === 404) {
                    return;
                }
                throw this.wrapK8sClientError(error);
            }
            throw new Error('ERR_GET_DEPLOYMENT');
        });
    }
    createPod(name, image, serviceAccount, restartPolicy, pullPolicy, configMapEnvSource, namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sCoreApi = this.kc.makeApiClient(client_node_1.CoreV1Api);
            let pod = new client_node_1.V1Pod();
            pod.metadata = new client_node_1.V1ObjectMeta();
            pod.metadata.name = name;
            pod.metadata.labels = { app: name };
            pod.metadata.namespace = namespace;
            pod.spec = new client_node_1.V1PodSpec();
            pod.spec.containers;
            pod.spec.restartPolicy = restartPolicy;
            pod.spec.serviceAccountName = serviceAccount;
            let opContainer = new client_node_1.V1Container();
            opContainer.name = name;
            opContainer.image = image;
            opContainer.imagePullPolicy = pullPolicy;
            let envFromSource = new client_node_1.V1EnvFromSource();
            envFromSource.configMapRef = new client_node_1.V1ConfigMapEnvSource();
            envFromSource.configMapRef.name = configMapEnvSource;
            opContainer.envFrom = [envFromSource];
            pod.spec.containers = [opContainer];
            try {
                return yield k8sCoreApi.createNamespacedPod(namespace, pod);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    compare(body, name) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            if (body && body.metadata && body.metadata.name && body.metadata.name === name) {
                return true;
            }
            else {
                return false;
            }
        });
    }
    ingressExist(name = '', namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sExtensionsApi = this.kc.makeApiClient(client_node_1.ExtensionsV1beta1Api);
            try {
                const { body } = yield k8sExtensionsApi.readNamespacedIngress(name, namespace);
                return this.compare(body, name);
            }
            catch (_a) {
                return false;
            }
        });
    }
    deleteAllIngresses(namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sExtensionsApi = this.kc.makeApiClient(client_node_1.ExtensionsV1beta1Api);
            try {
                yield k8sExtensionsApi.deleteCollectionNamespacedIngress(namespace);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    createCrdFromFile(filePath) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const yamlCrd = this.safeLoadFromYamlFile(filePath);
            const k8sApiextensionsApi = this.kc.makeApiClient(client_node_1.ApiextensionsV1beta1Api);
            try {
                return yield k8sApiextensionsApi.createCustomResourceDefinition(yamlCrd);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    replaceCrdFromFile(filePath, resourceVersion) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const yamlCrd = this.safeLoadFromYamlFile(filePath);
            if (!yamlCrd.metadata || !yamlCrd.metadata.name) {
                throw new Error(`CRD read from ${filePath} must have name specified`);
            }
            yamlCrd.metadata.resourceVersion = resourceVersion;
            const k8sApiextensionsApi = this.kc.makeApiClient(client_node_1.ApiextensionsV1beta1Api);
            try {
                return yield k8sApiextensionsApi.replaceCustomResourceDefinition(yamlCrd.metadata.name, yamlCrd);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    crdExist(name = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sApiextensionsApi = this.kc.makeApiClient(client_node_1.ApiextensionsV1beta1Api);
            try {
                const { body } = yield k8sApiextensionsApi.readCustomResourceDefinition(name);
                return this.compare(body, name);
            }
            catch (_a) {
                return false;
            }
        });
    }
    getCrd(name = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sApiextensionsApi = this.kc.makeApiClient(client_node_1.ApiextensionsV1beta1Api);
            try {
                const { body } = yield k8sApiextensionsApi.readCustomResourceDefinition(name);
                return body;
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    deleteCrd(name = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sApiextensionsApi = this.kc.makeApiClient(client_node_1.ApiextensionsV1beta1Api);
            try {
                const options = new client_node_1.V1DeleteOptions();
                yield k8sApiextensionsApi.deleteCustomResourceDefinition(name, undefined, options);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    createCheClusterFromFile(filePath, flags, useDefaultCR) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            let yamlCr = this.safeLoadFromYamlFile(filePath);
            yamlCr = this.overrideDefaultValues(yamlCr, flags['che-operator-cr-patch-yaml']);
            const cheNamespace = flags.chenamespace;
            if (useDefaultCR) {
                // If we don't use an explicitly provided CheCluster CR,
                // then let's modify the default example CR with values
                // derived from the other parameters
                const cheImage = flags.cheimage;
                const imageAndTag = cheImage.split(':', 2);
                yamlCr.spec.server.cheImage = imageAndTag[0];
                yamlCr.spec.server.cheImageTag = imageAndTag.length === 2 ? imageAndTag[1] : 'latest';
                yamlCr.spec.server.cheDebug = flags.debug ? flags.debug.toString() : 'false';
                yamlCr.spec.auth.openShiftoAuth = flags['os-oauth'];
                if (flags.tls) {
                    yamlCr.spec.server.tlsSupport = flags.tls;
                    yamlCr.spec.k8s.tlsSecretName = 'che-tls';
                }
                yamlCr.spec.server.selfSignedCert = flags['self-signed-cert'];
                if (flags.domain) {
                    yamlCr.spec.k8s.ingressDomain = flags.domain;
                }
                const pluginRegistryUrl = flags['plugin-registry-url'];
                if (pluginRegistryUrl) {
                    yamlCr.spec.server.pluginRegistryUrl = pluginRegistryUrl;
                    yamlCr.spec.server.externalPluginRegistry = true;
                }
                const devfileRegistryUrl = flags['devfile-registry-url'];
                if (devfileRegistryUrl) {
                    yamlCr.spec.server.devfileRegistryUrl = devfileRegistryUrl;
                    yamlCr.spec.server.externalDevfileRegistry = true;
                }
                yamlCr.spec.storage.postgresPVCStorageClassName = flags['postgres-pvc-storage-class-name'];
                yamlCr.spec.storage.workspacePVCStorageClassName = flags['workspace-pvc-storage-class-name'];
                if (flags.cheimage === constants_1.DEFAULT_CHE_IMAGE &&
                    yamlCr.spec.server.cheImageTag !== 'nightly' &&
                    yamlCr.spec.server.cheImageTag !== 'latest') {
                    // We obviously are using a release version of crwctl with the default `cheimage`
                    // => We should use the operator defaults for docker images
                    yamlCr.spec.server.cheImage = '';
                    yamlCr.spec.server.cheImageTag = '';
                    yamlCr.spec.server.pluginRegistryImage = '';
                    yamlCr.spec.server.devfileRegistryImage = '';
                    yamlCr.spec.auth.identityProviderImage = '';
                }
            }
            const customObjectsApi = this.kc.makeApiClient(client_node_1.CustomObjectsApi);
            try {
                const { body } = yield customObjectsApi.createNamespacedCustomObject('org.eclipse.che', 'v1', cheNamespace, 'checlusters', yamlCr);
                return body;
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    overrideDefaultValues(yamlCr, filePath) {
        if (filePath) {
            const patchCr = this.safeLoadFromYamlFile(filePath);
            return lodash_1.merge(yamlCr, patchCr);
        }
        else {
            return yamlCr;
        }
    }
    cheClusterExist(name = '', namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const customObjectsApi = this.kc.makeApiClient(client_node_1.CustomObjectsApi);
            try {
                const { body } = yield customObjectsApi.getNamespacedCustomObject('org.eclipse.che', 'v1', namespace, 'checlusters', name);
                return this.compare(body, name);
            }
            catch (_a) {
                return false;
            }
        });
    }
    deleteCheCluster(name = '', namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const customObjectsApi = this.kc.makeApiClient(client_node_1.CustomObjectsApi);
            try {
                const options = new client_node_1.V1DeleteOptions();
                yield customObjectsApi.deleteNamespacedCustomObject('org.eclipse.che', 'v1', namespace, 'checlusters', name, options);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    currentContext() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return this.kc.getCurrentContext();
        });
    }
    getContext(name) {
        return this.kc.getContextObject(name);
    }
    /**
     * Retrieve the default token from the default serviceAccount.
     */
    getDefaultServiceAccountToken() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sCoreApi = this.kc.makeApiClient(client_node_1.CoreV1Api);
            const namespaceName = 'default';
            const saName = 'default';
            let res;
            // now get the matching secrets
            try {
                res = yield k8sCoreApi.listNamespacedSecret(namespaceName);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
            if (!res || !res.body) {
                throw new Error('Unable to get default service account');
            }
            const v1SecretList = res.body;
            if (!v1SecretList.items || v1SecretList.items.length === 0) {
                throw new Error(`Unable to get default service account token since there is no secret in '${namespaceName}' namespace`);
            }
            let v1DefaultSATokenSecret = v1SecretList.items.find(secret => secret.metadata.annotations
                && secret.metadata.annotations['kubernetes.io/service-account.name'] === saName
                && secret.type === 'kubernetes.io/service-account-token');
            if (!v1DefaultSATokenSecret) {
                throw new Error(`Secret for '${saName}' service account is not found in namespace '${namespaceName}'`);
            }
            return Buffer.from(v1DefaultSATokenSecret.data.token, 'base64').toString();
        });
    }
    checkKubeApi() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const currentCluster = this.kc.getCurrentCluster();
            if (!currentCluster) {
                throw new Error('Failed to get current Kubernetes cluster: returned null');
            }
            const token = yield this.getDefaultServiceAccountToken();
            const agent = new https.Agent({
                rejectUnauthorized: false
            });
            let endpoint = '';
            try {
                endpoint = `${currentCluster.server}/healthz`;
                let response = yield axios_1.default.get(`${endpoint}`, { httpsAgent: agent, headers: { Authorization: 'bearer ' + token } });
                if (!response || response.status !== 200 || response.data !== 'ok') {
                    throw new Error('E_BAD_RESP_K8S_API');
                }
            }
            catch (error) {
                if (error.response && error.response.status === 403) {
                    throw new Error(`E_K8S_API_FORBIDDEN - Message: ${error.response.data.message}`);
                }
                if (error.response && error.response.status === 401) {
                    throw new Error(`E_K8S_API_UNAUTHORIZED - Message: ${error.response.data.message}`);
                }
                if (error.response) {
                    // The request was made and the server responded with a status code
                    // that falls out of the range of 2xx
                    throw new Error(`E_K8S_API_UNKNOWN_ERROR - Status: ${error.response.status}`);
                }
                else if (error.request) {
                    // The request was made but no response was received
                    // `error.request` is an instance of XMLHttpRequest in the browser and an instance of
                    // http.ClientRequest in node.js
                    throw new Error(`E_K8S_API_NO_RESPONSE - Endpoint: ${endpoint} - Error message: ${error.message}`);
                }
                else {
                    // Something happened in setting up the request that triggered an Error
                    throw new Error(`E_CHECTL_UNKNOWN_ERROR - Message: ${error.message}`);
                }
            }
        });
    }
    isOpenShift() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sApiApi = this.kc.makeApiClient(client_node_1.ApisApi);
            let res;
            try {
                res = yield k8sApiApi.getAPIVersions();
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
            if (!res || !res.body) {
                throw new Error('Get API versions returned an invalid response');
            }
            const v1APIGroupList = res.body;
            for (const v1APIGroup of v1APIGroupList.groups) {
                if (v1APIGroup.name === 'apps.openshift.io') {
                    return true;
                }
            }
            return false;
        });
    }
    getIngressHost(name = '', namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sExtensionsApi = this.kc.makeApiClient(client_node_1.ExtensionsV1beta1Api);
            try {
                const res = yield k8sExtensionsApi.readNamespacedIngress(name, namespace);
                if (res && res.body &&
                    res.body.spec &&
                    res.body.spec.rules &&
                    res.body.spec.rules.length > 0) {
                    return res.body.spec.rules[0].host || '';
                }
                throw new Error('ERR_INGRESS_NO_HOST');
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    getIngressProtocol(name = '', namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sExtensionsApi = this.kc.makeApiClient(client_node_1.ExtensionsV1beta1Api);
            try {
                const res = yield k8sExtensionsApi.readNamespacedIngress(name, namespace);
                if (!res || !res.body || !res.body.spec) {
                    throw new Error('ERR_INGRESS_NO_HOST');
                }
                if (res.body.spec.tls && res.body.spec.tls.length > 0) {
                    return 'https';
                }
                else {
                    return 'http';
                }
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    getIngressesBySelector(labelSelector = '', namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sV1Beta = this.kc.makeApiClient(client_node_1.ExtensionsV1beta1Api);
            try {
                const res = yield k8sV1Beta.listNamespacedIngress(namespace, true, 'true', undefined, undefined, labelSelector);
                if (res && res.body) {
                    return res.body;
                }
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
            throw new Error('ERR_LIST_INGRESSES');
        });
    }
    apiVersionExist(expectedVersion) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sCoreApi = this.kc.makeApiClient(client_node_1.ApisApi);
            // if matching APi Version
            try {
                const res = yield k8sCoreApi.getAPIVersions();
                if (res && res.body && res.body.groups) {
                    return res.body.groups.some(version => version.name === expectedVersion);
                }
                else {
                    return false;
                }
            }
            catch (_a) {
                return false;
            }
        });
    }
    getSecret(name = '', namespace = 'default') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sCoreApi = this.kc.makeApiClient(client_node_1.CoreV1Api);
            // now get the matching secrets
            try {
                const res = yield k8sCoreApi.readNamespacedSecret(name, namespace);
                if (res && res.body && res.body) {
                    return res.body;
                }
                else {
                    return;
                }
            }
            catch (_a) {
                return;
            }
        });
    }
    persistentVolumeClaimExist(name = '', namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sCoreApi = this.kc.makeApiClient(client_node_1.CoreV1Api);
            try {
                const { body } = yield k8sCoreApi.readNamespacedPersistentVolumeClaim(name, namespace);
                return this.compare(body, name);
            }
            catch (_a) {
                return false;
            }
        });
    }
    deletePersistentVolumeClaim(name = '', namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sCoreApi = this.kc.makeApiClient(client_node_1.CoreV1Api);
            try {
                const options = new client_node_1.V1DeleteOptions();
                yield k8sCoreApi.deleteNamespacedPersistentVolumeClaim(name, namespace, undefined, options);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    getPersistentVolumeClaimsBySelector(labelSelector = '', namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sCoreApi = this.kc.makeApiClient(client_node_1.CoreV1Api);
            try {
                const res = yield k8sCoreApi.listNamespacedPersistentVolumeClaim(namespace, true, 'true', undefined, undefined, labelSelector);
                if (res && res.body) {
                    return res.body;
                }
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
            throw new Error('ERR_LIST_PVCS');
        });
    }
    listNamespace() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sApi = this.kc.makeApiClient(client_node_1.CoreV1Api);
            try {
                const res = yield k8sApi.listNamespace();
                if (res && res.body) {
                    return res.body;
                }
                else {
                    return {
                        items: []
                    };
                }
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    listNamespacedPod(namespace, fieldSelector, labelSelector) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sApi = this.kc.makeApiClient(client_node_1.CoreV1Api);
            try {
                const res = yield k8sApi.listNamespacedPod(namespace, true, undefined, undefined, fieldSelector, labelSelector);
                if (res && res.body) {
                    return res.body;
                }
                else {
                    return {
                        items: []
                    };
                }
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    /**
     * Reads log by chunk and writes into a file.
     */
    readNamespacedPodLog(pod, namespace, container, filename, follow) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return new Promise((resolve, reject) => {
                const stream = new stream_1.Writable();
                stream._write = function (chunk, encoding, done) {
                    fs.appendFileSync(filename, chunk, { encoding });
                    done();
                };
                this.logHelper.log(namespace, pod, container, stream, error => {
                    stream.end();
                    if (error) {
                        reject(error);
                    }
                    else {
                        resolve();
                    }
                }, { follow });
            });
        });
    }
    /**
     * Forwards port, based on the example
     * https://github.com/kubernetes-client/javascript/blob/master/examples/typescript/port-forward/port-forward.ts
     */
    portForward(podName, namespace, port) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            try {
                const server = net.createServer((socket) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.portForwardHelper.portForward(namespace, podName, [port], socket, null, socket);
                }));
                server.listen(port, 'localhost');
                return;
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    /**
     * Checks if message is present and returns error with it
     * or returns error with the specified error if message is not found.
     *
     * @param e k8s error to wrap
     */
    wrapK8sClientError(e) {
        if (e.response && e.response.body && e.response.body.message)
            return new Error(e.response.body.message);
        else
            return new Error(e);
    }
    safeLoadFromYamlFile(filePath) {
        return yaml.safeLoad(fs.readFileSync(filePath).toString());
    }
}
exports.KubeHelper = KubeHelper;
class PatchedK8sApi extends client_node_1.CoreV1Api {
    patchNamespacedConfigMap(...args) {
        const oldDefaultHeaders = this.defaultHeaders;
        this.defaultHeaders = Object.assign({ 'Content-Type': 'application/strategic-merge-patch+json' }, this.defaultHeaders);
        const returnValue = super.patchNamespacedConfigMap.apply(this, args);
        this.defaultHeaders = oldDefaultHeaders;
        return returnValue;
    }
}
class PatchedK8sAppsApi extends client_node_1.AppsV1Api {
    patchNamespacedDeployment(...args) {
        const oldDefaultHeaders = this.defaultHeaders;
        this.defaultHeaders = Object.assign({ 'Content-Type': 'application/strategic-merge-patch+json' }, this.defaultHeaders);
        const returnValue = super.patchNamespacedDeployment.apply(this, args);
        this.defaultHeaders = oldDefaultHeaders;
        return returnValue;
    }
    patchNamespacedDeploymentScale(...args) {
        const oldDefaultHeaders = this.defaultHeaders;
        this.defaultHeaders = Object.assign({ 'Content-Type': 'application/strategic-merge-patch+json' }, this.defaultHeaders);
        const returnValue = super.patchNamespacedDeploymentScale.apply(this, args);
        this.defaultHeaders = oldDefaultHeaders;
        return returnValue;
    }
}
//# sourceMappingURL=kube.js.map