"use strict";
/**
 * Copyright (c) 2019-2021 Red Hat, Inc.
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *   Red Hat, Inc. - initial API and implementation
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.OLMTasks = void 0;
const tslib_1 = require("tslib");
const cli_ux_1 = require("cli-ux");
const yaml = require("js-yaml");
const Listr = require("listr");
const path = require("path");
const kube_1 = require("../../api/kube");
const version_1 = require("../../api/version");
const constants_1 = require("../../constants");
const util_1 = require("../../util");
const common_tasks_1 = require("./common-tasks");
class OLMTasks {
    constructor() {
        this.prometheusRoleName = 'prometheus-k8s';
        this.prometheusRoleBindingName = 'prometheus-k8s';
    }
    /**
     * Returns list of tasks which perform preflight platform checks.
     */
    startTasks(flags, command) {
        const kube = new kube_1.KubeHelper(flags);
        return new Listr([
            this.isOlmPreInstalledTask(command, kube),
            common_tasks_1.createNamespaceTask(flags.chenamespace, this.getOlmNamespaceLabels(flags)),
            {
                enabled: () => flags['cluster-monitoring'] && flags.platform === 'openshift',
                title: `Create Role ${this.prometheusRoleName} in namespace ${flags.chenamespace}`,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const yamlFilePath = path.join(flags.templates, '..', 'installers', 'prometheus-role.yaml');
                    const exist = yield kube.roleExist(this.prometheusRoleName, flags.chenamespace);
                    if (exist) {
                        task.title = `${task.title}...It already exists.`;
                    }
                    else {
                        yield kube.createRoleFromFile(yamlFilePath, flags.chenamespace);
                        task.title = `${task.title}...done.`;
                    }
                }),
            },
            {
                enabled: () => flags['cluster-monitoring'] && flags.platform === 'openshift',
                title: `Create RoleBinding ${this.prometheusRoleBindingName} in namespace ${flags.chenamespace}`,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const exist = yield kube.roleBindingExist(this.prometheusRoleBindingName, flags.chenamespace);
                    const yamlFilePath = path.join(flags.templates, '..', 'installers', 'prometheus-role-binding.yaml');
                    if (exist) {
                        task.title = `${task.title}...It already exists.`;
                    }
                    else {
                        yield kube.createRoleBindingFromFile(yamlFilePath, flags.chenamespace);
                        task.title = `${task.title}...done.`;
                    }
                }),
            },
            {
                title: 'Create operator group',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    if (yield kube.operatorGroupExists(constants_1.OPERATOR_GROUP_NAME, flags.chenamespace)) {
                        task.title = `${task.title}...It already exists.`;
                    }
                    else {
                        yield kube.createOperatorGroup(constants_1.OPERATOR_GROUP_NAME, flags.chenamespace);
                        task.title = `${task.title}...created new one.`;
                    }
                }),
            },
            {
                title: 'Configure context information',
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    ctx.defaultCatalogSourceNamespace = util_1.isKubernetesPlatformFamily(flags.platform) ? constants_1.DEFAULT_OLM_KUBERNETES_NAMESPACE : constants_1.DEFAULT_OPENSHIFT_MARKET_PLACE_NAMESPACE;
                    // catalog source name for stable Che version
                    ctx.catalogSourceNameStable = util_1.isKubernetesPlatformFamily(flags.platform) ? constants_1.KUBERNETES_OLM_CATALOG : constants_1.OPENSHIFT_OLM_CATALOG;
                    ctx.sourceName = flags['catalog-source-name'] || constants_1.CUSTOM_CATALOG_SOURCE_NAME;
                    ctx.generalPlatformName = util_1.isKubernetesPlatformFamily(flags.platform) ? 'kubernetes' : 'openshift';
                    if (flags.version) {
                        // Convert version flag to channel (see subscription object), starting CSV and approval starategy
                        flags.version = version_1.VersionHelper.removeVPrefix(flags.version, true);
                        // Need to point to specific CSV
                        ctx.startingCSV = `eclipse-che.v${flags.version}`;
                        // Set approval starategy to manual to prevent autoupdate to the latest version right before installation
                        ctx.approvalStarategy = 'Manual';
                    }
                    else {
                        ctx.startingCSV = flags['starting-csv'];
                        if (ctx.startingCSV) {
                            // Ignore auto-update flag, otherwise it will automatically update to the latest version and starting-csv will not have any effect.
                            ctx.approvalStarategy = 'Manual';
                        }
                        else if (flags['auto-update'] === undefined) {
                            ctx.approvalStarategy = 'Automatic';
                        }
                        else {
                            ctx.approvalStarategy = flags['auto-update'] ? 'Automatic' : 'Manual';
                        }
                    }
                    task.title = `${task.title}...done.`;
                }),
            },
            {
                enabled: () => !version_1.VersionHelper.isDeployingStableVersion(flags) && !flags['catalog-source-name'] && !flags['catalog-source-yaml'] && flags['olm-channel'] !== constants_1.OLM_STABLE_CHANNEL_NAME,
                title: `Create nightly index CatalogSource in the namespace ${flags.chenamespace}`,
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    if (!(yield kube.catalogSourceExists(constants_1.NIGHTLY_CATALOG_SOURCE_NAME, flags.chenamespace))) {
                        const catalogSourceImage = `quay.io/eclipse/eclipse-che-${ctx.generalPlatformName}-opm-catalog:preview`;
                        const nigthlyCatalogSource = this.constructIndexCatalogSource(flags.chenamespace, catalogSourceImage);
                        yield kube.createCatalogSource(nigthlyCatalogSource);
                        yield kube.waitCatalogSource(flags.chenamespace, constants_1.NIGHTLY_CATALOG_SOURCE_NAME);
                    }
                    else {
                        task.title = `${task.title}...It already exists.`;
                    }
                }),
            },
            {
                enabled: () => flags['catalog-source-yaml'],
                title: 'Create custom catalog source from file',
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const customCatalogSource = kube.readCatalogSourceFromFile(flags['catalog-source-yaml']);
                    if (!(yield kube.catalogSourceExists(customCatalogSource.metadata.name, flags.chenamespace))) {
                        customCatalogSource.metadata.name = ctx.sourceName;
                        customCatalogSource.metadata.namespace = flags.chenamespace;
                        yield kube.createCatalogSource(customCatalogSource);
                        yield kube.waitCatalogSource(flags.chenamespace, constants_1.CUSTOM_CATALOG_SOURCE_NAME);
                        task.title = `${task.title}...created new one, with name ${constants_1.CUSTOM_CATALOG_SOURCE_NAME} in the namespace ${flags.chenamespace}.`;
                    }
                    else {
                        task.title = `${task.title}...It already exists.`;
                    }
                }),
            },
            {
                title: 'Create operator subscription',
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    if (yield kube.operatorSubscriptionExists(constants_1.SUBSCRIPTION_NAME, flags.chenamespace)) {
                        task.title = `${task.title}...It already exists.`;
                    }
                    else {
                        let subscription;
                        if (flags['catalog-source-yaml'] || flags['catalog-source-name']) {
                            // custom Che CatalogSource
                            const catalogSourceNamespace = flags['catalog-source-namespace'] || flags.chenamespace;
                            subscription = this.constructSubscription(constants_1.SUBSCRIPTION_NAME, flags['package-manifest-name'], flags.chenamespace, catalogSourceNamespace, flags['olm-channel'], ctx.sourceName, ctx.approvalStarategy, ctx.startingCSV);
                        }
                        else if (version_1.VersionHelper.isDeployingStableVersion(flags) || flags['olm-channel'] === constants_1.OLM_STABLE_CHANNEL_NAME) {
                            // stable Che CatalogSource
                            subscription = this.constructSubscription(constants_1.SUBSCRIPTION_NAME, constants_1.DEFAULT_CHE_OLM_PACKAGE_NAME, flags.chenamespace, ctx.defaultCatalogSourceNamespace, constants_1.OLM_STABLE_CHANNEL_NAME, ctx.catalogSourceNameStable, ctx.approvalStarategy, ctx.startingCSV);
                        }
                        else {
                            // nightly Che CatalogSource
                            subscription = this.constructSubscription(constants_1.SUBSCRIPTION_NAME, `eclipse-che-preview-${ctx.generalPlatformName}`, flags.chenamespace, flags.chenamespace, constants_1.OLM_NIGHTLY_CHANNEL_NAME, constants_1.NIGHTLY_CATALOG_SOURCE_NAME, ctx.approvalStarategy, ctx.startingCSV);
                        }
                        yield kube.createOperatorSubscription(subscription);
                        task.title = `${task.title}...created new one.`;
                    }
                }),
            },
            {
                title: 'Wait while subscription is ready',
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const installPlan = yield kube.waitOperatorSubscriptionReadyForApproval(flags.chenamespace, constants_1.SUBSCRIPTION_NAME, 600);
                    ctx.installPlanName = installPlan.name;
                    task.title = `${task.title}...done.`;
                }),
            },
            {
                title: 'Approve installation',
                enabled: ctx => ctx.approvalStarategy === 'Manual',
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield kube.approveOperatorInstallationPlan(ctx.installPlanName, flags.chenamespace);
                    task.title = `${task.title}...done.`;
                }),
            },
            {
                title: 'Wait while operator installed',
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield kube.waitUntilOperatorIsInstalled(ctx.installPlanName, flags.chenamespace);
                    task.title = `${task.title}...done.`;
                }),
            },
            {
                title: 'Set custom operator image',
                enabled: () => flags['che-operator-image'],
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const csvList = yield kube.getClusterServiceVersions(flags.chenamespace);
                    if (csvList.items.length < 1) {
                        throw new Error('Failed to get CSV for Che operator');
                    }
                    const csv = csvList.items[0];
                    const jsonPatch = [{ op: 'replace', path: '/spec/install/spec/deployments/0/spec/template/spec/containers/0/image', value: flags['che-operator-image'] }];
                    yield kube.patchClusterServiceVersion(csv.metadata.namespace, csv.metadata.name, jsonPatch);
                    task.title = `${task.title}... changed to ${flags['che-operator-image']}.`;
                }),
            },
            {
                title: 'Prepare CodeReady Workspaces cluster CR',
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const cheCluster = yield kube.getCheCluster(flags.chenamespace);
                    if (cheCluster) {
                        task.title = `${task.title}...It already exists..`;
                        return;
                    }
                    if (!ctx.customCR) {
                        ctx.defaultCR = yield this.getCRFromCSV(kube, flags.chenamespace);
                    }
                    task.title = `${task.title}...Done.`;
                }),
            },
            common_tasks_1.createEclipseCheCluster(flags, kube),
        ], { renderer: flags['listr-renderer'] });
    }
    preUpdateTasks(flags, command) {
        const kube = new kube_1.KubeHelper(flags);
        return new Listr([
            this.isOlmPreInstalledTask(command, kube),
            {
                title: 'Check if operator group exists',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    if (!(yield kube.operatorGroupExists(constants_1.OPERATOR_GROUP_NAME, flags.chenamespace))) {
                        command.error(`Unable to find operator group ${constants_1.OPERATOR_GROUP_NAME}`);
                    }
                    task.title = `${task.title}...done.`;
                }),
            },
            {
                title: 'Check if operator subscription exists',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    if (!(yield kube.operatorSubscriptionExists(constants_1.SUBSCRIPTION_NAME, flags.chenamespace))) {
                        command.error(`Unable to find operator subscription ${constants_1.SUBSCRIPTION_NAME}`);
                    }
                    task.title = `${task.title}...done.`;
                }),
            },
        ], { renderer: flags['listr-renderer'] });
    }
    updateTasks(flags, command) {
        const kube = new kube_1.KubeHelper(flags);
        return new Listr([
            {
                title: 'Get operator installation plan',
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const subscription = yield kube.getOperatorSubscription(constants_1.SUBSCRIPTION_NAME, flags.chenamespace);
                    if (subscription.status) {
                        if (subscription.status.state === 'AtLatestKnown') {
                            task.title = `Everything is up to date. Installed the latest known version '${subscription.status.currentCSV}'.`;
                            return;
                        }
                        // Retrieve current and next version from the subscription status
                        const installedCSV = subscription.status.installedCSV;
                        if (installedCSV) {
                            ctx.currentVersion = installedCSV.substr(installedCSV.lastIndexOf('v') + 1);
                        }
                        const currentCSV = subscription.status.currentCSV;
                        ctx.nextVersion = currentCSV.substr(currentCSV.lastIndexOf('v') + 1);
                        if (subscription.status.state === 'UpgradePending' && subscription.status.conditions) {
                            const installCondition = subscription.status.conditions.find(condition => condition.type === 'InstallPlanPending' && condition.status === 'True');
                            if (installCondition) {
                                ctx.installPlanName = subscription.status.installplan.name;
                                task.title = `${task.title}...done.`;
                                return;
                            }
                        }
                        if (subscription.status.state === 'UpgradeAvailable' && installedCSV === currentCSV) {
                            command.error('Another update is in progress');
                        }
                    }
                    command.error('Unable to find installation plan to update.');
                }),
            },
            {
                title: 'Approve installation',
                enabled: (ctx) => ctx.installPlanName,
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield kube.approveOperatorInstallationPlan(ctx.installPlanName, flags.chenamespace);
                    task.title = `${task.title}...done.`;
                }),
            },
            {
                title: 'Wait while newer operator installed',
                enabled: (ctx) => ctx.installPlanName,
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield kube.waitUntilOperatorIsInstalled(ctx.installPlanName, flags.chenamespace, 60);
                    ctx.highlightedMessages.push(`Operator is updated from ${ctx.currentVersion} to ${ctx.nextVersion} version`);
                    task.title = `${task.title}...done.`;
                }),
            },
            common_tasks_1.patchingEclipseCheCluster(flags, kube, command),
        ], { renderer: flags['listr-renderer'] });
    }
    deleteTasks(flags) {
        const kube = new kube_1.KubeHelper(flags);
        return [
            {
                title: 'Check if OLM is pre-installed on the platform',
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    ctx.isPreInstalledOLM = Boolean(yield kube.isPreInstalledOLM());
                    task.title = `${task.title}: ${ctx.isPreInstalledOLM}...OK`;
                }),
            },
            {
                title: `Delete(OLM) operator subscription ${constants_1.SUBSCRIPTION_NAME}`,
                enabled: ctx => ctx.isPreInstalledOLM,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield kube.deleteOperatorSubscription(constants_1.SUBSCRIPTION_NAME, flags.chenamespace);
                    task.title = `${task.title}...OK`;
                }),
            },
            {
                title: 'Delete(OLM) CodeReady Workspaces cluster service versions',
                enabled: ctx => ctx.isPreInstalledOLM,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const csvs = yield kube.getClusterServiceVersions(flags.chenamespace);
                    const csvsToDelete = csvs.items.filter(csv => csv.metadata.name.startsWith(constants_1.CVS_PREFIX));
                    csvsToDelete.forEach(csv => kube.deleteClusterServiceVersion(flags.chenamespace, csv.metadata.name));
                    task.title = `${task.title}...OK`;
                }),
            },
            {
                title: `Delete(OLM) operator group ${constants_1.OPERATOR_GROUP_NAME}`,
                enabled: ctx => ctx.isPreInstalledOLM,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield kube.deleteOperatorGroup(constants_1.OPERATOR_GROUP_NAME, flags.chenamespace);
                    task.title = `${task.title}...OK`;
                }),
            },
            {
                title: `Delete(OLM) custom catalog source ${constants_1.CUSTOM_CATALOG_SOURCE_NAME}`,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield kube.deleteCatalogSource(flags.chenamespace, constants_1.CUSTOM_CATALOG_SOURCE_NAME);
                    task.title = `${task.title}...OK`;
                }),
            },
            {
                title: `Delete(OLM) nigthly catalog source ${constants_1.NIGHTLY_CATALOG_SOURCE_NAME}`,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield kube.deleteCatalogSource(flags.chenamespace, constants_1.NIGHTLY_CATALOG_SOURCE_NAME);
                    task.title = `${task.title}...OK`;
                }),
            },
            {
                title: `Delete role ${this.prometheusRoleName}`,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield kube.deleteRole(this.prometheusRoleName, flags.chenamespace);
                    task.title = yield `${task.title}...OK`;
                }),
            },
            {
                title: `Delete role binding ${this.prometheusRoleName}`,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield kube.deleteRoleBinding(this.prometheusRoleName, flags.chenamespace);
                    task.title = yield `${task.title}...OK`;
                }),
            },
        ];
    }
    isOlmPreInstalledTask(command, kube) {
        return {
            title: 'Check if OLM is pre-installed on the platform',
            task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                if (!(yield kube.isPreInstalledOLM())) {
                    cli_ux_1.cli.warn('Looks like your platform hasn\'t got embedded OLM, so you should install it manually. For quick start you can use:');
                    cli_ux_1.cli.url('install.sh', 'https://raw.githubusercontent.com/operator-framework/operator-lifecycle-manager/master/deploy/upstream/quickstart/install.sh');
                    command.error('OLM is required for installation CodeReady Workspaces with installer flag \'olm\'');
                }
                task.title = `${task.title}...done.`;
            }),
        };
    }
    constructSubscription(name, packageName, namespace, sourceNamespace, channel, sourceName, installPlanApproval, startingCSV) {
        return {
            apiVersion: 'operators.coreos.com/v1alpha1',
            kind: 'Subscription',
            metadata: {
                name,
                namespace,
            },
            spec: {
                channel,
                installPlanApproval,
                name: packageName,
                source: sourceName,
                sourceNamespace,
                startingCSV,
            },
        };
    }
    constructIndexCatalogSource(namespace, catalogSourceImage) {
        return {
            apiVersion: 'operators.coreos.com/v1alpha1',
            kind: 'CatalogSource',
            metadata: {
                name: constants_1.NIGHTLY_CATALOG_SOURCE_NAME,
                namespace,
            },
            spec: {
                image: catalogSourceImage,
                sourceType: 'grpc',
                updateStrategy: {
                    registryPoll: {
                        interval: '15m',
                    },
                },
            },
        };
    }
    getCRFromCSV(kube, cheNamespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const subscription = yield kube.getOperatorSubscription(constants_1.SUBSCRIPTION_NAME, cheNamespace);
            const currentCSV = subscription.status.currentCSV;
            const csv = yield kube.getCSV(currentCSV, cheNamespace);
            if (csv && csv.metadata.annotations) {
                const CRRaw = csv.metadata.annotations['alm-examples'];
                return yaml.safeLoad(CRRaw)[0];
            }
            else {
                throw new Error(`Unable to retrieve Che cluster CR definition from CSV: ${currentCSV}`);
            }
        });
    }
    getOlmNamespaceLabels(flags) {
        // The label values must be strings
        if (flags['cluster-monitoring'] && flags.platform === 'openshift') {
            return { 'openshift.io/cluster-monitoring': 'true' };
        }
        return {};
    }
}
exports.OLMTasks = OLMTasks;
//# sourceMappingURL=olm.js.map