/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2011, Red Hat, Inc., and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.as.clustering.jgroups.subsystem;

import static org.jboss.as.controller.descriptions.ModelDescriptionConstants.ADD;
import static org.jboss.as.controller.descriptions.ModelDescriptionConstants.OPERATION_NAME;
import static org.jboss.as.controller.descriptions.ModelDescriptionConstants.OP_ADDR;

import java.lang.reflect.Field;
import java.security.AccessController;
import java.security.PrivilegedAction;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.ResourceBundle;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;

import javax.management.MBeanServer;

import org.jboss.as.clustering.jgroups.ChannelFactory;
import org.jboss.as.clustering.jgroups.JGroupsMessages;
import org.jboss.as.clustering.jgroups.ProtocolConfiguration;
import org.jboss.as.clustering.jgroups.ProtocolDefaults;
import org.jboss.as.clustering.jgroups.ProtocolStackConfiguration;
import org.jboss.as.clustering.jgroups.RelayConfiguration;
import org.jboss.as.clustering.jgroups.RemoteSiteConfiguration;
import org.jboss.as.clustering.jgroups.SaslConfiguration;
import org.jboss.as.clustering.jgroups.TransportConfiguration;
import org.jboss.as.controller.AbstractAddStepHandler;
import org.jboss.as.controller.OperationContext;
import org.jboss.as.controller.OperationFailedException;
import org.jboss.as.controller.PathAddress;
import org.jboss.as.controller.ServiceVerificationHandler;
import org.jboss.as.controller.descriptions.DescriptionProvider;
import org.jboss.as.controller.descriptions.ModelDescriptionConstants;
import org.jboss.as.controller.operations.common.Util;
import org.jboss.as.controller.registry.Resource;
import org.jboss.as.domain.management.SecurityRealm;
import org.jboss.as.jmx.MBeanServerService;
import org.jboss.as.network.SocketBinding;
import org.jboss.as.server.ServerEnvironment;
import org.jboss.as.server.ServerEnvironmentService;
import org.jboss.as.threads.ThreadsServices;
import org.jboss.dmr.ModelNode;
import org.jboss.dmr.Property;
import org.jboss.msc.inject.Injector;
import org.jboss.msc.service.ServiceBuilder;
import org.jboss.msc.service.ServiceController;
import org.jboss.msc.service.ServiceTarget;
import org.jboss.msc.value.InjectedValue;
import org.jboss.msc.value.Value;
import org.jboss.threads.JBossExecutors;

/**
 * @author Paul Ferraro
 */
public class ProtocolStackAdd extends AbstractAddStepHandler implements DescriptionProvider {

    public static final ProtocolStackAdd INSTANCE = new ProtocolStackAdd();

    static ModelNode createOperation(ModelNode address, ModelNode existing) {
        ModelNode operation = Util.getEmptyOperation(ModelDescriptionConstants.ADD, address);
        populate(existing, operation);
        return operation;
    }

    // need to override the description provider in order to not include the attribute protocols
    @Override
    public ModelNode getModelDescription(Locale locale) {
        ResourceBundle resources = ResourceBundle.getBundle(JGroupsExtension.RESOURCE_NAME, (locale == null) ? Locale.getDefault() : locale);

        ModelNode stack = new ModelNode();
        stack.get(ModelDescriptionConstants.OPERATION_NAME).set(ADD);
        stack.get(ModelDescriptionConstants.DESCRIPTION).set(resources.getString("jgroups.stack.add"));

        // optional TRANSPORT and PROTOCOLS parameters (to permit configuring a stack from add())
        TransportResourceDefinition.TRANSPORT.addOperationParameterDescription(resources,"jgroups.stack.add", stack);
        ProtocolResourceDefinition.PROTOCOLS.addOperationParameterDescription(resources, "jgroups.stack.add" , stack);

        return stack ;
    }

    private static void populate(ModelNode source, ModelNode target) {
        // nothing to do for a basic add
    }

    @Override
    protected void populateModel(final ModelNode operation, final ModelNode model) {
        // this method is abstract in AbstractAddStepHandler
        // we want to use its more explicit version below, but have to override it anyway
    }

    @Override
    protected void populateModel(final OperationContext context, final ModelNode operation, final Resource resource) {

        final ModelNode model = resource.getModel();

        // handle the basic add() operation parameters
        populate(operation, model);

        // add a step to initialize an *optional* TRANSPORT parameter
        if (operation.hasDefined(ModelKeys.TRANSPORT)) {
            // create an ADD operation to add the transport=TRANSPORT child
            ModelNode addTransport = operation.get(ModelKeys.TRANSPORT).clone();

            addTransport.get(OPERATION_NAME).set(ModelDescriptionConstants.ADD);
            ModelNode transportAddress = operation.get(OP_ADDR).clone();
            transportAddress.add(ModelKeys.TRANSPORT, ModelKeys.TRANSPORT_NAME);
            transportAddress.protect();
            addTransport.get(OP_ADDR).set(transportAddress);

            // execute the operation using the transport handler
            context.addStep(addTransport, TransportResourceDefinition.TRANSPORT_ADD, OperationContext.Stage.IMMEDIATE);
        }

        // add steps to initialize *optional* PROTOCOL parameters
        if (operation.hasDefined(ModelKeys.PROTOCOLS)) {

            List<ModelNode> protocols = operation.get(ModelKeys.PROTOCOLS).asList();
            // because we use stage IMMEDIATE when creating protocols, unless we reverse the order
            // of the elements in the LIST, they will get applied in reverse order - the last step
            // added gets executed first

            for (int i = protocols.size()-1; i >= 0; i--) {
                ModelNode protocol = protocols.get(i);

                // create an ADD operation to add the protocol=* child
                ModelNode addProtocol = protocol.clone();
                addProtocol.get(OPERATION_NAME).set(ModelKeys.ADD_PROTOCOL);
                // add-protocol is a stack operation
                ModelNode protocolAddress = operation.get(OP_ADDR).clone();
                protocolAddress.protect();
                addProtocol.get(OP_ADDR).set(protocolAddress);

                // execute the operation using the transport handler
                context.addStep(addProtocol, ProtocolResourceDefinition.PROTOCOL_ADD_HANDLER, OperationContext.Stage.IMMEDIATE);
            }
        }
    }

    @Override
    protected void performRuntime(OperationContext context, ModelNode operation, ModelNode model, ServiceVerificationHandler verificationHandler, List<ServiceController<?>> newControllers)
            throws OperationFailedException {

        // Because we use child resources in a read-only manner to configure the protocol stack, replace the local model with the full model
        model = Resource.Tools.readModel(context.readResource(PathAddress.EMPTY_ADDRESS));

        installRuntimeServices(context, operation, model, verificationHandler, newControllers);
    }

    protected void installRuntimeServices(OperationContext context, ModelNode operation, ModelNode model, ServiceVerificationHandler verificationHandler, List<ServiceController<?>> newControllers) throws OperationFailedException {

        final PathAddress address = PathAddress.pathAddress(operation.get(OP_ADDR));
        final String name = address.getLastElement().getValue();

        // check that we have enough information to create a stack
        protocolStackSanityCheck(name, model);

        // we need to preserve the order of the protocols as maintained by PROTOCOLS
        // pick up the ordered protocols here as a List<Property> where property is <name, ModelNode>
        List<Property> orderedProtocols = getOrderedProtocolPropertyList(model);

        // pick up the transport here and its values
        ModelNode resolvedValue = null;
        ModelNode transport = model.get(ModelKeys.TRANSPORT, ModelKeys.TRANSPORT_NAME);
        final String type = (resolvedValue = TransportResourceDefinition.TYPE.resolveModelAttribute(context, transport)).isDefined() ? resolvedValue.asString() : null;
        final boolean  shared = TransportResourceDefinition.SHARED.resolveModelAttribute(context, transport).asBoolean();
        final String machine = (resolvedValue = TransportResourceDefinition.MACHINE.resolveModelAttribute(context, transport)).isDefined() ? resolvedValue.asString() : null;
        final String rack = (resolvedValue = TransportResourceDefinition.RACK.resolveModelAttribute(context, transport)).isDefined() ? resolvedValue.asString() : null;
        final String site = (resolvedValue = TransportResourceDefinition.SITE.resolveModelAttribute(context, transport)).isDefined() ? resolvedValue.asString() : null;
        final String timerExecutor = (resolvedValue = TransportResourceDefinition.TIMER_EXECUTOR.resolveModelAttribute(context, transport)).isDefined() ? resolvedValue.asString() : null;
        final String threadFactory = (resolvedValue = TransportResourceDefinition.THREAD_FACTORY.resolveModelAttribute(context, transport)).isDefined() ? resolvedValue.asString() : null;
        final String diagnosticsSocketBinding = (resolvedValue = TransportResourceDefinition.DIAGNOSTICS_SOCKET_BINDING.resolveModelAttribute(context, transport)).isDefined() ? resolvedValue.asString() : null;
        final String defaultExecutor = (resolvedValue = TransportResourceDefinition.DEFAULT_EXECUTOR.resolveModelAttribute(context, transport)).isDefined() ? resolvedValue.asString() : null;
        final String oobExecutor = (resolvedValue = TransportResourceDefinition.OOB_EXECUTOR.resolveModelAttribute(context, transport)).isDefined() ? resolvedValue.asString() : null;
        final String transportSocketBinding = (resolvedValue = TransportResourceDefinition.SOCKET_BINDING.resolveModelAttribute(context, transport)).isDefined() ? resolvedValue.asString() : null;

        // set up the transport
        Transport transportConfig = new Transport(type);
        transportConfig.setShared(shared);
        transportConfig.setTopology(site, rack, machine);
        initProtocolProperties(context, transport, transportConfig);

        Relay relayConfig = null;
        List<Map.Entry<String, Injector<ChannelFactory>>> stacks = new LinkedList<Map.Entry<String, Injector<ChannelFactory>>>();
        if (model.hasDefined(ModelKeys.RELAY)) {
            final ModelNode relay = model.get(ModelKeys.RELAY, ModelKeys.RELAY_NAME);
            final String siteName = RelayResourceDefinition.SITE.resolveModelAttribute(context, relay).asString();
            relayConfig = new Relay(siteName);
            initProtocolProperties(context, relay, relayConfig);
            if (relay.hasDefined(ModelKeys.REMOTE_SITE)) {
                List<RemoteSiteConfiguration> remoteSites = relayConfig.getRemoteSites();
                for (Property remoteSiteProperty : relay.get(ModelKeys.REMOTE_SITE).asPropertyList()) {
                    final String remoteSiteName = remoteSiteProperty.getName();
                    final ModelNode remoteSite = remoteSiteProperty.getValue();
                    final String clusterName = RemoteSiteResourceDefinition.CLUSTER.resolveModelAttribute(context, remoteSite)
                            .asString();
                    final String stack = RemoteSiteResourceDefinition.STACK.resolveModelAttribute(context, remoteSite).asString();
                    final InjectedValue<ChannelFactory> channelFactory = new InjectedValue<ChannelFactory>();
                    remoteSites.add(new RemoteSite(remoteSiteName, clusterName, channelFactory));
                    stacks.add(new AbstractMap.SimpleImmutableEntry<String, Injector<ChannelFactory>>(stack, channelFactory));
                }
            }
        }

        Sasl saslConfig = null;
        if (model.hasDefined(ModelKeys.SASL)) {
            final ModelNode sasl = model.get(ModelKeys.SASL, ModelKeys.SASL_NAME);
            final String clusterRole = (resolvedValue = SaslResourceDefinition.CLUSTER_ROLE.resolveModelAttribute(context, sasl)).isDefined() ? resolvedValue.asString() : null;
            final String securityRealm = SaslResourceDefinition.SECURITY_REALM.resolveModelAttribute(context, sasl).asString();
            final String mech = SaslResourceDefinition.MECH.resolveModelAttribute(context, sasl).asString();
            saslConfig = new Sasl(securityRealm, mech, clusterRole);
            initProtocolProperties(context, sasl, saslConfig);
        }

        // set up the protocol stack Protocol objects
        ProtocolStack stackConfig = new ProtocolStack(name, transportConfig, relayConfig, saslConfig);
        List<Map.Entry<Protocol, String>> protocolSocketBindings = new ArrayList<Map.Entry<Protocol, String>>(orderedProtocols.size());
        for (Property protocolProperty : orderedProtocols) {
            ModelNode protocol = protocolProperty.getValue();
            final String protocolType = (resolvedValue = ProtocolResourceDefinition.TYPE.resolveModelAttribute(context, protocol)).isDefined() ? resolvedValue.asString() : null;
            Protocol protocolConfig = new Protocol(protocolType);
            initProtocolProperties(context, protocol, protocolConfig);
            stackConfig.getProtocols().add(protocolConfig);
            final String protocolSocketBinding = (resolvedValue = ProtocolResourceDefinition.SOCKET_BINDING.resolveModelAttribute(context, protocol)).isDefined() ? resolvedValue.asString() : null;
            protocolSocketBindings.add(new AbstractMap.SimpleImmutableEntry<Protocol, String>(protocolConfig, protocolSocketBinding));
        }

        // install the default channel factory service
        ServiceController<ChannelFactory> cfsController = installChannelFactoryService(context.getServiceTarget(),
                        name, diagnosticsSocketBinding, defaultExecutor, oobExecutor, timerExecutor, threadFactory,
                        transportSocketBinding, protocolSocketBindings, transportConfig, stackConfig, stacks, saslConfig, verificationHandler);
        if (newControllers != null) {
            newControllers.add(cfsController);
        }
    }

    protected void removeRuntimeServices(OperationContext context, ModelNode operation, ModelNode model)
            throws OperationFailedException {

        final PathAddress address = PathAddress.pathAddress(operation.get(ModelDescriptionConstants.OP_ADDR));
        final String name = address.getLastElement().getValue();

        // remove the ChannelFactoryServiceService
        context.removeService(ChannelFactoryService.getServiceName(name));
    }


    protected ServiceController<ChannelFactory> installChannelFactoryService(ServiceTarget target,
                                                                             String name,
                                                                             String diagnosticsSocketBinding,
                                                                             String defaultExecutor,
                                                                             String oobExecutor,
                                                                             String timerExecutor,
                                                                             String threadFactory,
                                                                             String transportSocketBinding,
                                                                             List<Map.Entry<Protocol, String>> protocolSocketBindings,
                                                                             Transport transportConfig,
                                                                             ProtocolStack stackConfig,
                                                                             List<Map.Entry<String, Injector<ChannelFactory>>> stacks,
                                                                             Sasl sasl,
                                                                             ServiceVerificationHandler verificationHandler) {

        // create the channel factory service builder
        ServiceBuilder<ChannelFactory> builder = target
                .addService(ChannelFactoryService.getServiceName(name), new ChannelFactoryService(stackConfig))
                .addDependency(ProtocolDefaultsService.SERVICE_NAME, ProtocolDefaults.class, stackConfig.getDefaultsInjector())
                .addDependency(MBeanServerService.SERVICE_NAME, MBeanServer.class, stackConfig.getMBeanServerInjector())
                .addDependency(ServerEnvironmentService.SERVICE_NAME, ServerEnvironment.class, stackConfig.getEnvironmentInjector())
                .setInitialMode(ServiceController.Mode.ON_DEMAND)
        ;
        // add transport dependencies
        addSocketBindingDependency(builder, transportSocketBinding, transportConfig.getSocketBindingInjector());

        for (Map.Entry<Protocol, String> entry: protocolSocketBindings) {
            addSocketBindingDependency(builder, entry.getValue(), entry.getKey().getSocketBindingInjector());
        }

        // add remaining dependencies
        addSocketBindingDependency(builder, diagnosticsSocketBinding, transportConfig.getDiagnosticsSocketBindingInjector());
        addExecutorDependency(builder, defaultExecutor, transportConfig.getDefaultExecutorInjector());
        addExecutorDependency(builder, oobExecutor, transportConfig.getOOBExecutorInjector());
        if (timerExecutor != null) {
            builder.addDependency(ThreadsServices.executorName(timerExecutor), ScheduledExecutorService.class, transportConfig.getTimerExecutorInjector());
        }
        if (threadFactory != null) {
            builder.addDependency(ThreadsServices.threadFactoryName(threadFactory), ThreadFactory.class, transportConfig.getThreadFactoryInjector());
        }
        if (sasl != null) {
            builder.addDependency(SecurityRealm.ServiceUtil.createServiceName(sasl.getRealm()), SecurityRealm.class, sasl.getSecurityRealmInjector());
        }
        for (Map.Entry<String, Injector<ChannelFactory>> entry: stacks) {
            builder.addDependency(ChannelFactoryService.getServiceName(entry.getKey()), ChannelFactory.class, entry.getValue());
        }
        return builder.install();
    }

    private void initProtocolProperties(OperationContext context, ModelNode protocol, Protocol protocolConfig) throws OperationFailedException {

        Map<String, String> properties = protocolConfig.getProperties();
        // properties are a child resource of protocol
        if (protocol.hasDefined(ModelKeys.PROPERTY)) {
            for (Property property : protocol.get(ModelKeys.PROPERTY).asPropertyList()) {
                // the format of the property elements
                //  "property" => {
                //       "relative-to" => {"value" => "fred"},
                //   }
                String propertyName = property.getName();
                String propertyValue = PropertyResourceDefinition.VALUE.resolveModelAttribute(context, property.getValue()).asString();
                if (propertyValue != null && !propertyValue.isEmpty()) {
                   properties.put(propertyName, propertyValue);
                }
            }
       }
    }

    public static List<Property> getOrderedProtocolPropertyList(ModelNode stack) {
        ModelNode orderedProtocols = new ModelNode();

        // check for the empty ordering list
        if  (!stack.hasDefined(ModelKeys.PROTOCOLS)) {
            return null;
        }
        // PROTOCOLS is a list of protocol names only, reflecting the order in which protocols were added to the stack
        List<ModelNode> protocolOrdering = stack.get(ModelKeys.PROTOCOLS).asList();

        // now construct an ordered list of the full protocol model nodes
        ModelNode unorderedProtocols = stack.get(ModelKeys.PROTOCOL);
        for (ModelNode protocolName : protocolOrdering) {
            ModelNode protocolModel = unorderedProtocols.get(protocolName.asString());
            orderedProtocols.add(protocolName.asString(), protocolModel);
        }
        return orderedProtocols.asPropertyList();
    }

    private void addSocketBindingDependency(ServiceBuilder<ChannelFactory> builder, String socketBinding, Injector<SocketBinding> injector) {
        if (socketBinding != null) {
            builder.addDependency(SocketBinding.JBOSS_BINDING_NAME.append(socketBinding), SocketBinding.class, injector);
        }
    }

    private void addExecutorDependency(ServiceBuilder<ChannelFactory> builder, String executor, Injector<Executor> injector) {
        if (executor != null) {
            builder.addDependency(ThreadsServices.executorName(executor), Executor.class, injector);
        }
    }

    /*
     * A check that we have the minimal configuration required to create a protocol stack.
     */
    private void protocolStackSanityCheck(String stackName, ModelNode model) throws OperationFailedException {

         ModelNode transport = model.get(ModelKeys.TRANSPORT, ModelKeys.TRANSPORT_NAME);
         if (!transport.isDefined()) {
            throw JGroupsMessages.MESSAGES.transportNotDefined(stackName);
         }

         List<Property> protocols = getOrderedProtocolPropertyList(model);
         if ( protocols == null || !(protocols.size() > 0)) {
             throw JGroupsMessages.MESSAGES.protocolListNotDefined(stackName);
         }
    }

    @Override
    protected boolean requiresRuntimeVerification() {
        return false;
    }

    static class ProtocolStack implements ProtocolStackConfiguration {
        private final InjectedValue<ProtocolDefaults> defaults = new InjectedValue<ProtocolDefaults>();
        private final InjectedValue<MBeanServer> mbeanServer = new InjectedValue<MBeanServer>();
        private final InjectedValue<ServerEnvironment> environment = new InjectedValue<ServerEnvironment>();
        private final InjectedValue<SecurityRealm> securityRealm = new InjectedValue<SecurityRealm>();

        private final String name;
        private final TransportConfiguration transport;
        private final RelayConfiguration relay;
        private final SaslConfiguration sasl;
        private final List<ProtocolConfiguration> protocols = new LinkedList<ProtocolConfiguration>();


        ProtocolStack(String name, TransportConfiguration transport, RelayConfiguration relay, SaslConfiguration sasl) {
            this.name = name;
            this.transport = transport;
            this.relay = relay;
            this.sasl = sasl;
        }

        public Injector<SecurityRealm> getSecurityRealmInjector() {
            return this.securityRealm;
        }

        Injector<ProtocolDefaults> getDefaultsInjector() {
            return this.defaults;
        }

        Injector<MBeanServer> getMBeanServerInjector() {
            return this.mbeanServer;
        }

        Injector<ServerEnvironment> getEnvironmentInjector() {
            return this.environment;
        }

        @Override
        public TransportConfiguration getTransport() {
            return this.transport;
        }

        @Override
        public List<ProtocolConfiguration> getProtocols() {
            return this.protocols;
        }

        @Override
        public String getName() {
            return this.name;
        }

        @Override
        public ProtocolDefaults getDefaults() {
            return this.defaults.getValue();
        }

        @Override
        public MBeanServer getMBeanServer() {
            return this.mbeanServer.getOptionalValue();
        }

        @Override
        public ServerEnvironment getEnvironment() {
            return this.environment.getValue();
        }

        @Override
        public RelayConfiguration getRelay() {
            return this.relay;
        }

        @Override
        public SaslConfiguration getSasl() {
            return this.sasl;
        }
    }

    static class Transport extends Protocol implements TransportConfiguration {
        private final InjectedValue<SocketBinding> diagnosticsSocketBinding = new InjectedValue<SocketBinding>();
        private final InjectedValue<Executor> defaultExecutor = new InjectedValue<Executor>();
        private final InjectedValue<Executor> oobExecutor = new InjectedValue<Executor>();
        private final InjectedValue<ScheduledExecutorService> timerExecutor = new InjectedValue<ScheduledExecutorService>();
        private final InjectedValue<ThreadFactory> threadFactory = new InjectedValue<ThreadFactory>();
        private boolean shared = true;
        private Topology topology;

        Transport(String name) {
            super(name);
        }

        Injector<SocketBinding> getDiagnosticsSocketBindingInjector() {
            return this.diagnosticsSocketBinding;
        }

        Injector<Executor> getDefaultExecutorInjector() {
            return this.defaultExecutor;
        }

        Injector<Executor> getOOBExecutorInjector() {
            return this.oobExecutor;
        }

        Injector<ScheduledExecutorService> getTimerExecutorInjector() {
            return this.timerExecutor;
        }

        Injector<ThreadFactory> getThreadFactoryInjector() {
            return this.threadFactory;
        }

        void setShared(boolean shared) {
            this.shared = shared;
        }

        @Override
        public boolean isShared() {
            return this.shared;
        }

        @Override
        public Topology getTopology() {
            return this.topology;
        }

        public void setTopology(String site, String rack, String machine) {
            if ((site != null) || (rack != null) || (machine != null)) {
                this.topology = new TopologyImpl(site, rack, machine);
            }
        }

        @Override
        public SocketBinding getDiagnosticsSocketBinding() {
            return this.diagnosticsSocketBinding.getOptionalValue();
        }

        @Override
        public ExecutorService getDefaultExecutor() {
            Executor executor = this.defaultExecutor.getOptionalValue();
            return (executor != null) ? JBossExecutors.protectedExecutorService(executor) : null;
        }

        @Override
        public ExecutorService getOOBExecutor() {
            Executor executor = this.oobExecutor.getOptionalValue();
            return (executor != null) ? JBossExecutors.protectedExecutorService(executor) : null;
        }

        @Override
        public ScheduledExecutorService getTimerExecutor() {
            return this.timerExecutor.getOptionalValue();
        }

        @Override
        public ThreadFactory getThreadFactory() {
            return this.threadFactory.getOptionalValue();
        }

        private class TopologyImpl implements Topology {
            private final String site;
            private final String rack;
            private final String machine;

            TopologyImpl(String site, String rack, String machine) {
                this.site = site;
                this.rack = rack;
                this.machine = machine;
            }

            @Override
            public String getMachine() {
                return this.machine;
            }

            @Override
            public String getRack() {
                return this.rack;
            }

            @Override
            public String getSite() {
                return this.site;
            }
        }
    }

    static class Relay extends Protocol implements RelayConfiguration {
        private final List<RemoteSiteConfiguration> remoteSites = new LinkedList<RemoteSiteConfiguration>();
        private final String siteName;

        Relay(String siteName) {
            super("relay.RELAY2");
            this.siteName = siteName;
        }

        @Override
        public String getSiteName() {
            return this.siteName;
        }

        @Override
        public List<RemoteSiteConfiguration> getRemoteSites() {
            return this.remoteSites;
        }
    }

    static class RemoteSite implements RemoteSiteConfiguration {
        private final Value<ChannelFactory> channelFactory;
        private final String name;
        private final String clusterName;

        RemoteSite(String name, String clusterName, Value<ChannelFactory> channelFactory) {
            this.name = name;
            this.clusterName = clusterName;
            this.channelFactory = channelFactory;
        }

        @Override
        public String getName() {
            return this.name;
        }

        @Override
        public ChannelFactory getChannelFactory() {
            return this.channelFactory.getValue();
        }

        @Override
        public String getCluster() {
            return this.clusterName;
        }
    }

    static class Sasl extends Protocol implements SaslConfiguration {
       private final InjectedValue<SecurityRealm> securityRealmInjector = new InjectedValue<SecurityRealm>();
       private final String clusterRole;
       private final String realm;
       private final String mech;

       Sasl(String realm, String mech, String clusterRole) {
          super("SASL");
          this.realm = realm;
          this.mech = mech;
          this.clusterRole = clusterRole;
       }

       @Override
       public String getClusterRole() {
           return clusterRole;
       }

       @Override
       public String getMech() {
           return mech;
       }

       public String getRealm() {
           return realm;
       }

       @Override
       public SecurityRealm getSecurityRealm() {
         return securityRealmInjector.getValue();
       }

       public Injector<SecurityRealm> getSecurityRealmInjector() {
           return securityRealmInjector;
       }
    }

    static class Protocol implements ProtocolConfiguration {
        private final String name;
        private final InjectedValue<SocketBinding> socketBinding = new InjectedValue<SocketBinding>();
        private final Map<String, String> properties = new HashMap<String, String>();
        final Class<?> protocolClass;

        Protocol(final String name) {
            this.name = name;
            PrivilegedAction<Class<?>> action = new PrivilegedAction<Class<?>>() {
                @Override
                public Class<?> run() {
                    try {
                        return Protocol.this.getClass().getClassLoader().loadClass(org.jgroups.conf.ProtocolConfiguration.protocol_prefix + '.' + name);
                    } catch (ClassNotFoundException e) {
                        throw new IllegalStateException(e);
                    }
                }
            };
            this.protocolClass = AccessController.doPrivileged(action);
        }

        @Override
        public String getName() {
            return this.name;
        }

        @Override
        public boolean hasProperty(final String property) {
            PrivilegedAction<Field> action = new PrivilegedAction<Field>() {
                @Override
                public Field run() {
                    return getField(Protocol.this.protocolClass, property);
                }
            };
            return AccessController.doPrivileged(action) != null;
        }

        static Field getField(Class<?> targetClass, String property) {
            try {
                return targetClass.getDeclaredField(property);
            } catch (NoSuchFieldException e) {
                Class<?> superClass = targetClass.getSuperclass();
                return (superClass != null) && org.jgroups.stack.Protocol.class.isAssignableFrom(superClass) ? getField(superClass, property) : null;
            }
        }

        @Override
        public Map<String, String> getProperties() {
            return this.properties;
        }

        Injector<SocketBinding> getSocketBindingInjector() {
            return this.socketBinding;
        }

        @Override
        public SocketBinding getSocketBinding() {
            return this.socketBinding.getOptionalValue();
        }
    }
}
