package org.infinispan.client.hotrod.impl.iteration;

import org.infinispan.client.hotrod.RemoteCache;
import org.infinispan.client.hotrod.exceptions.HotRodClientException;
import org.infinispan.client.hotrod.test.SingleHotRodServerTest;
import org.infinispan.commons.util.CloseableIterator;
import org.infinispan.filter.AbstractKeyValueFilterConverter;
import org.infinispan.filter.KeyValueFilterConverter;
import org.infinispan.filter.KeyValueFilterConverterFactory;
import org.infinispan.metadata.Metadata;
import org.infinispan.query.dsl.embedded.testdomain.hsearch.AccountHS;
import org.testng.annotations.Test;

import java.util.Collections;
import java.util.HashSet;
import java.util.Map;
import java.util.Map.Entry;
import java.util.NoSuchElementException;
import java.util.Set;

import static org.infinispan.client.hotrod.impl.iteration.RemoteIteratorTestUtils.*;
import static org.infinispan.commons.util.InfinispanCollections.emptySet;
import static org.testng.Assert.assertFalse;
import static org.testng.AssertJUnit.assertEquals;

/**
 * @author gustavonalle
 * @since 8.0
 */
@Test(groups = "functional", testName = "client.hotrod.iteration.RemoteIteratorTest")
public class SingleServerRemoteIteratorTest extends SingleHotRodServerTest {

   public static final String FILTER_CONVERTER_FACTORY_NAME = "even-accounts-descriptions";

   @Test
   public void testEmptyCache() {
      CloseableIterator<Entry<Object, Object>> iterator = null;
      try {
         iterator = remoteCacheManager.getCache().retrieveEntries(null, null, 100);
         assertFalse(iterator.hasNext());
         assertFalse(iterator.hasNext());
      } finally {
         if (iterator != null) {
            iterator.close();
         }
      }
   }

   @Test
   public void testEmptySegments() {
      RemoteCache<Integer, String> cache = remoteCacheManager.getCache();
      populateCache(1, new Function<Integer, String>() {
         @Override
         public String apply(Integer integer) {
            return "value" + integer;

         }
      }, cache);
      CloseableIterator<Entry<Object, Object>> iterator = null;
      try {
         iterator = remoteCacheManager.getCache().retrieveEntries(null, Collections.<Integer>emptySet(), 100);
         assertFalse(iterator.hasNext());
      } finally {
         if (iterator != null) {
            iterator.close();
         }
      }
   }

   @Test(expectedExceptions = HotRodClientException.class, expectedExceptionsMessageRegExp = "java.lang.IllegalStateException: ISPN006019: Factory '' not found in server")
   public void testEmptyFilter() {
      CloseableIterator<Entry<Object, Object>> iterator = null;
      try {
         iterator = remoteCacheManager.getCache().retrieveEntries("", null, 100);
         assertFalse(iterator.hasNext());
      } finally {
         if (iterator != null) {
            iterator.close();
         }
      }
   }

   @Test(expectedExceptions = NoSuchElementException.class)
   public void testException1() {
      CloseableIterator<Entry<Object, Object>> iterator = null;
      try {
         iterator = remoteCacheManager.getCache().retrieveEntries(null, null, 100);
         iterator.next();
      } finally {
         if (iterator != null) {
            iterator.close();
         }
      }
   }

   @Test(expectedExceptions = NoSuchElementException.class)
   public void testException2() {
      RemoteCache<Integer, String> cache = remoteCacheManager.getCache();

      populateCache(3, new Function<Integer, String>() {
         @Override
         public String apply(Integer integer) {
            return "value " + integer;
         }
      }, cache);

      CloseableIterator<Entry<Object, Object>> iterator = remoteCacheManager.getCache().retrieveEntries(null, null, 100);
      iterator.close();
      iterator.next();
   }

   public void testResultsSingleFetch() {
      RemoteCache<Integer, String> cache = remoteCacheManager.getCache();

      populateCache(3, new Function<Integer, String>() {
         @Override
         public String apply(Integer integer) {
            return "value " + integer;
         }
      }, cache);

      Set<Entry<Object, Object>> entries = new HashSet<Entry<Object, Object>>();

      CloseableIterator<Entry<Object, Object>> iterator = null;
      try {
         iterator = remoteCacheManager.getCache().retrieveEntries(null, null, 5);
         entries.add(iterator.next());
         entries.add(iterator.next());
         entries.add(iterator.next());
      } finally {
         if (iterator != null) {
            iterator.close();
         }
      }

      Set<Integer> keys = extractKeys(entries);
      Set<String> values = extractValues(entries);

      assertEquals(keys, setOf(0, 1, 2));
      assertEquals(values, setOf("value 0", "value 1", "value 2"));
   }

   public void testResultsMultipleFetches() {
      RemoteCache<Integer, String> cache = remoteCacheManager.getCache();

      int cacheSize = 100;
      populateCache(cacheSize, new Function<Integer, String>() {
         @Override
         public String apply(Integer integer) {
            return "value " + integer;
         }
      }, cache);

      Set<Map.Entry<Object, Object>> entries = new HashSet<Entry<Object, Object>>();

      CloseableIterator<Entry<Object, Object>> iterator = null;
      try {
         iterator = remoteCacheManager.getCache().retrieveEntries(null, null, 5);
         while (iterator.hasNext()) {
            entries.add(iterator.next());
         }
      } finally {
         if (iterator != null) {
            iterator.close();
         }
      }

      Set<Integer> keys = extractKeys(entries);
      assertEquals(range(0, cacheSize), keys);
   }

   public void testEntities() {
      RemoteCache<Integer, AccountHS> cache = remoteCacheManager.getCache();

      final int cacheSize = 50;
      populateCache(cacheSize, new Function<Integer, AccountHS>() {
         @Override
         public AccountHS apply(Integer integer) {
            return newAccount(integer);
         }
      }, cache);

      Set<Entry<Object, Object>> entries = new HashSet<Entry<Object, Object>>();

      CloseableIterator<Entry<Object, Object>> iterator = null;
      try {
         iterator = cache.retrieveEntries(null, null, 5);
         while (iterator.hasNext()) {
            entries.add(iterator.next());
         }
      } finally {
         if (iterator != null) {
            iterator.close();
         }
      }

      assertEquals(cacheSize, entries.size());

      Set<AccountHS> values = extractValues(entries);

      assertForAll(values, new org.infinispan.client.hotrod.impl.iteration.Condition<AccountHS>() {
         @Override
         public boolean test(AccountHS accountHS) {
            return accountHS != null;
         }
      });
      assertForAll(values, new org.infinispan.client.hotrod.impl.iteration.Condition<AccountHS>() {
         @Override
         public boolean test(AccountHS accountHS) {
            return accountHS.getId() < cacheSize;
         }
      });
   }

   public void testFilterConverter() {
      hotrodServer.addKeyValueFilterConverterFactory(FILTER_CONVERTER_FACTORY_NAME, new KeyValueFilterConverterFactory<Integer, AccountHS, String>() {
         @Override
         public KeyValueFilterConverter<Integer, AccountHS, String> getFilterConverter() {
            return new AbstractKeyValueFilterConverter<Integer, AccountHS, String>() {
               @Override
               public String filterAndConvert(Integer key, AccountHS value, Metadata metadata) {
                  if (!(key % 2 == 0)) {
                     return null;
                  }
                  return value.getDescription();
               }
            };
         }
      });

      RemoteCache<Integer, AccountHS> cache = remoteCacheManager.getCache();

      int cacheSize = 50;
      populateCache(cacheSize, new Function<Integer, AccountHS>() {
         @Override
         public AccountHS apply(Integer integer) {
            return newAccount(integer);
         }
      }, cache);

      Set<Entry<Object, Object>> entries = new HashSet<Entry<Object, Object>>();


      CloseableIterator<Entry<Object, Object>> iterator = null;
      try {
         iterator = remoteCacheManager.getCache().retrieveEntries(FILTER_CONVERTER_FACTORY_NAME, null, 5);
         while (iterator.hasNext()) {
            entries.add(iterator.next());
         }
      } finally {
         if (iterator != null) {
            iterator.close();
         }
      }

      assertEquals(cacheSize / 2, entries.size());

      Set<Integer> keys = extractKeys(entries);
      assertForAll(keys, new org.infinispan.client.hotrod.impl.iteration.Condition<Integer>() {
         @Override
         public boolean test(Integer integer) {
            return integer % 2 == 0;
         }
      });
   }


}