package com.redhat.installer.action;

import java.io.File;
import java.io.FilenameFilter;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashSet;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

import com.izforge.izpack.installer.AutomatedInstallData;
import com.izforge.izpack.installer.PanelAction;
import com.izforge.izpack.installer.PanelActionConfiguration;
import com.izforge.izpack.util.AbstractUIHandler;
import com.redhat.installer.util.JBossJDBCConstants;
import com.redhat.installer.util.JDBCConnectionUtils;

public class SetPreExistingDefaults implements PanelAction {

	private final String[] domainDescriptors = new String[] { "host.xml", "domain.xml" };
	private final String[] standaloneDescriptors = new String[] { "standalone.xml", "standalone-ha.xml", "standalone-osgi.xml", "standalone-full-ha.xml", "standalone-full.xml" };
	private final HashSet<String> collidingExtensions= new HashSet<String>();
	{
		collidingExtensions.add("org.modeshape");
		collidingExtensions.add("org.jboss.as.messaging");
	}
	private final HashSet<String> collidingSubsystems = new HashSet<String>();
	{
		collidingSubsystems.add("urn:jboss:domain:modeshape:1.0");
	}
	private final HashSet<String> collidingInfinispan = new HashSet<String>();
	{
		collidingInfinispan.add("modeshape");
	}
	
	private AutomatedInstallData idata;

	@Override
	public void executeAction(AutomatedInstallData idata, AbstractUIHandler handler) {
		// only if validation is passed will we reach here. Thus, we know the
		// path is valid.
		this.idata = idata;
		boolean needInstall = Boolean.parseBoolean(idata.getVariable("eap.needs.install"));

        resetDefaults(); // we need to do this so that, if the user first selects a directory with pre-existing defaults,
                         // and then selects another without, that the vault.preexisting and *.xml.vault.preexisting variables
                         // are cleared.
		if (needInstall) {
			// nothing to do, thankfully
			return;
		} else {
			// for every xml descriptor, we need to find out the following
			// information:
			// Is there a Vault declared?
			// Is there a security domain with the name "mySecurityDomain"
			// declared
			// Is there a pre-existing JDBC driver? DB url? etc
			for (String descriptor : standaloneDescriptors) {
				String path = idata.getVariable("INSTALL_PATH") + "/" + idata.getVariable("INSTALL_SUBPATH") + "/standalone/configuration/" + descriptor;
				File descFile = new File(path);
				if (descFile.exists()) {
					findDefaultInfo(descriptor, descFile);
				}
			}
			for (String descriptor : domainDescriptors){
				String path = idata.getVariable("INSTALL_PATH") + "/" + idata.getVariable("INSTALL_SUBPATH") + "/domain/configuration/" + descriptor;
				File descFile = new File(path);
				if (descFile.exists()){
					findDefaultInfo(descriptor, descFile);
				}
			}
		}

	}

    private void resetDefaults() {
        idata.setVariable("vault.preexisting", "false");
        for (String xml : standaloneDescriptors){
            idata.setVariable(xml+".vault.preexisting", "false");
            for (String extension : collidingExtensions){
                idata.setVariable(xml+"."+extension+".extension.exists","false");
            }
            for (String subsystem : collidingSubsystems){
                idata.setVariable(xml+"."+subsystem+".subsystem.exists","false");
            }
            for (String infinispan : collidingInfinispan){
                idata.setVariable(xml+"."+infinispan+".infinispan.exists","false");
            }
        }
    }


    private void setVaultDefaults(String xml, Document doc){
		// Load vault defaults into idata
		NodeList vault = doc.getElementsByTagName("vault-option");

		if (vault.getLength() != 0) {
			// there's a vault with options!
			idata.setVariable("vault.preexisting", "true");
			idata.setVariable(xml+".vault.preexisting", "true");
			for (int i = 0; i < vault.getLength(); i++) {
				Element e = (Element) vault.item(i);
				// set new values
				String name = e.getAttribute("name");
				String value = e.getAttribute("value");
				String varName = null;
				if (name.equals("KEYSTORE_URL")) {
					varName = "vault.keystoreloc";
				} else if (name.equals("KEYSTORE_ALIAS")) {
					varName = "vault.alias";
				} else if (name.equals("SALT")) {
					varName = "vault.salt";
				} else if (name.equals("ITERATION_COUNT")) {
					varName = "vault.itercount";
				} else if (name.equals("ENC_FILE_DIR")) {
					varName = "vault.encrdir";
				}
				// if the vault-option is has a value, and the varName is
				// one we care about
				if (varName != null && value != "") {
					idata.setVariable(varName, value);
				}
			}
		}
	}
	
	private void setSecurityDomainDefaults(Document doc){
		NodeList securityDomains = doc.getElementsByTagName("security-domain");

		if (securityDomains.getLength() != 0) {
			for (int i = 0; i < securityDomains.getLength(); i++) {
				Element e = (Element) securityDomains.item(i);
				String name = e.getAttribute("name");
				String existingName = idata.getVariable("securitydomain.name");
				// dv allows security domain installation
				if (name.equals(existingName)) {
					idata.setVariable("securitydomain.name", existingName + System.currentTimeMillis());
				}
			}
		}
	}
	
	private void setJDBCDefaults(String xml, Document doc){
		NodeList drivers = doc.getElementsByTagName("driver");
		ArrayList<String> modules = new ArrayList<String>(3);

		if (drivers.getLength() != 0) {
			for (int i = 0; i < drivers.getLength(); i++) {
				Element e = (Element) drivers.item(i);

				// get all of the listed drivers within a config

				String name = e.getAttribute("name");
				String module = e.getAttribute("module");
				// ignore h2 driver, since it isn't important for this
				// default setter
				if (!module.isEmpty() && !name.isEmpty()) {
					if (!name.equals("h2")) {
						modules.add(e.getAttribute("module"));
					}
				}
			}
		}

		URL[] preExistingJDBCJars = JDBCConnectionUtils.convertToUrlArray(parseJarLocationsFromModules(modules));
		ArrayList<String> existingDrivers = findAllExistingDrivers(preExistingJDBCJars);

	}
	
	private void ensureNoCliCollision(String xml, Document doc){
		NodeList extensions = doc.getElementsByTagName("extension");
		
		if (extensions.getLength() > 0){
			for (int i = 0; i < extensions.getLength(); i++){
				// try to find potentially conflicting extensions
				Element e = (Element) extensions.item(i);
				
				String module = e.getAttribute("module");
				if (collidingExtensions.contains(module)){
					idata.setVariable(xml+"."+module+".extension.exists","true");
				}
			}
		}
		
		NodeList subsystems = doc.getElementsByTagName("subsystem");
		
		if (subsystems.getLength() > 0){
			for (int i = 0; i < subsystems.getLength(); i++){
				Element e = (Element) subsystems.item(i);
				
				String xmlns = e.getAttribute("xmlns");
				if (collidingSubsystems.contains(xmlns)){
					idata.setVariable(xml+"."+xmlns+".subsystem.exists","true");
				}
			}
		}
		
		NodeList infinispan = doc.getElementsByTagName("cache-container");
		
		if (infinispan.getLength() > 0){
			for (int i = 0; i < infinispan.getLength(); i++){
				Element e = (Element) infinispan.item(i);
				
				String name = e.getAttribute("name");
				if (collidingInfinispan.contains(name)){
					idata.setVariable(xml+"."+name+".infinispan.exists","true");
				}
			}
		}
				
	}

	private void findDefaultInfo(String xml, File descriptor) {
		try {
			DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
			DocumentBuilder db = dbf.newDocumentBuilder();
			Document doc = db.parse(descriptor);
			ensureNoCliCollision(xml, doc);
			setVaultDefaults(xml, doc);
			setSecurityDomainDefaults(doc);
			setJDBCDefaults(xml, doc);
			
		} catch (Exception e) {
			e.printStackTrace();
		}

	}

	/**
	 * Attempts to load every JDBCDriver class, to check if it is pre-installed
	 * or not.
	 * 
	 * @param preExistingJDBCJars
	 * @return
	 */
	private ArrayList<String> findAllExistingDrivers(URL[] preExistingJDBCJars) {
		ArrayList<String> resultList = new ArrayList<String>(1);
		for (String driverClassname : JBossJDBCConstants.classnameList) {
			Class<?> driver = JDBCConnectionUtils.findDriverClass(driverClassname, preExistingJDBCJars);
			if (driver != null) {
				resultList.add(JBossJDBCConstants.classnameToJDBCMap.get(driverClassname));
			}
		}
		return resultList;
	}

	/**
	 * Relatively annoyingly complex method that takes a list of module names
	 * declared within the <drivers> element(s) of the parsed descriptor.
	 * Returns an array of type Object[] for conversion in the
	 * JDBCConnectionUtils class to use a classloader to search for real,
	 * existing JDBC drivers.
	 * 
	 * @param modules
	 * @return
	 */
	private Object[] parseJarLocationsFromModules(ArrayList<String> modules) {
		ArrayList<Object> resultList = new ArrayList<Object>(1);
		File parent = new File(idata.getInstallPath() + "/" + idata.getVariable("INSTALL_SUBPATH") + "/modules/system/layers");
		ArrayList<File> moduleSubdirs = new ArrayList<File>(3);
		for (String subPath : parent.list()) {
			// we don't much care about non-directories here
			if (new File(subPath).isDirectory()) {
				moduleSubdirs.add(new File(parent, subPath));
			}
		}
		for (String module : modules) {
			// the above code should leave moduleSubdirs filled with paths like
			// the following
			// INSTALL_PATH/jboss-eap-6.1/modules/system/layers/base
			// INSTALL_PATH/jboss-eap-6.1/modules/system/layers/soa
			// etc. now, we iterate through all of the module="blah" that we got
			// from the config files, making sure to look in each subdir
			for (File subdir : moduleSubdirs) {
				File check = new File(subdir, module);
				if (check.exists()) {
					// found it
					// manipulate the result into a usable form for our utility
					// classes
					String[] fileNames = check.list(new FilenameFilter() {

						@Override
						public boolean accept(File dir, String name) {
							if (name.endsWith("jar")) {
								return true;
							} else {
								return false;
							}
						}
					});

					// we now have all filenames. prepend them with the full
					// path.
					for (String file : fileNames) {
						resultList.add(check.getPath() + file);
					}
				}
			}
		}
		// done. return the result
		return resultList.toArray();
	}

	@Override
	public void initialize(PanelActionConfiguration configuration) {
		// unneeded
	}

}
