package com.redhat.installer.validator;

import com.izforge.izpack.installer.DataValidator;
import com.izforge.izpack.installer.AutomatedInstallData;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.util.HashMap;
import java.util.HashSet;

/**
 * Checks the user-mgmnt-properties.xml files for both domain and standalone modes
 * to ensure that the installer is not overwriting an existing username in either
 * file.
 * 
 *  Will warn if a duplicate username is found, so the user can decide to
 *  overwrite or not.
 * @author fcanas
 *
 */
public class DuplicateUserValidator implements DataValidator {
    private static final String MGMT_FILE = "mgmt-users.properties";
    private static final String CONFIG_PATH = "configuration";
    private static final String [] realms = new String [] {"standalone", "domain"};
    private static final String USERVAR = "adminUser";
    private static final String INSTALL_SUBPATH = "INSTALL_SUBPATH";
    
    /**
     * Error and warning messages:
     */
    private static String warning;
    private static String error;
    
    /**
     * We use a hashmap of hashsets to keep track of existing usernames in their
     * respective realm (domain or standalone).
     */
    private static HashMap<String, HashSet<String>> usersMap = new HashMap<String, HashSet<String>>(); 
    static {
        for (String realm : realms) {
            usersMap.put(realm, new HashSet<String>());
        }
    };

    public Status validateData(AutomatedInstallData idata) {
        String add_user_cond = (String) idata.getVariable("addUser");
        String user = (String) idata.getVariable(USERVAR);
        
        /**
         * Hacky way of skipping this validator entirely.
         * TODO: We must properly add the ability to skip validators
         * based on some xml-defined condition into izpack!
         */
        if ((add_user_cond != null) && add_user_cond.equals("false")) {
            // Set this false so we don't run the AddProperties spec job 
            // and write a new user in anyways during post install processes.
            idata.setVariable("add.new.user", "false");
            return Status.OK;
        }

        for (String realm : realms) {
            String filepath = getPath(realm, idata, MGMT_FILE);

            // Fill the user map:
            if (readExistingUsers(realm, filepath)) {
                // Check that we are not duplicating user:
                if (usersMap.get(realm).contains(user)) {
                    error = String.format(idata.langpack.getString("DuplicateUsername.errorMsg"),realm, user);
                    warning = String.format(idata.langpack.getString("DuplicateUsername.warningMsg"), user);
                    return Status.SKIP;
                }
            }
        }
        return Status.OK;
    }

    /**
     * Opens the mgmt-properties.xml file and reads existing users
     * into iData.  This occurs for both management realms: domain and standalone.
     * @return
     */
    public boolean readExistingUsers(String realm, String filepath) {
        String line;

            try {
                FileReader reader = new FileReader(filepath);
                BufferedReader in = new BufferedReader(reader);
                
                // Loop through the entire file.
                while((line = in.readLine()) != null) {
                    // Skip comments:
                    if (line.startsWith("#")) {
                        continue;
                    }
                    
                    // Upon finding a username entry, map the username.
                    if (line.contains("=")) {
                        String username = line.substring(0, line.indexOf('='));
                        ((HashSet<String>)usersMap.get(realm)).add(username);
                    }
                }
                in.close();
                reader.close();
            } catch (Exception e) {
                // No mgmt-properties files found, so it's a new EAP install.
                return false;
            }
            return true;
    }
    
    /**
     * Constructs the full path to the mgmt-properties file.
     * @param realm
     * @param idata
     * @return
     */
    private String getPath(String realm, AutomatedInstallData idata, String filename) {
        String path = idata.getInstallPath();
        String eapname = (String) idata.getVariable(INSTALL_SUBPATH);
        String filepath = path + File.separator + eapname + File.separator + 
                realm + File.separator + CONFIG_PATH + File.separator + filename;
        
        return filepath;
        
    }

    public String getErrorMessageId() {
        return error;
    }

    public String getWarningMessageId() {
        return warning;
    }

    public boolean getDefaultAnswer() {
        return true;
    }
}
