"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const Listr = require("listr");
const che_1 = require("../api/che");
const kube_1 = require("../api/kube");
const openshift_1 = require("../api/openshift");
const constants_1 = require("../constants");
const kube_2 = require("./kube");
/**
 * Holds tasks to work with CodeReady Workspaces component.
 */
class CheTasks {
    constructor(flags) {
        this.oc = new openshift_1.OpenShiftHelper();
        this.cheSelector = 'app=codeready,component=codeready';
        this.keycloakDeploymentName = 'keycloak';
        this.keycloakSelector = 'app=codeready,component=keycloak';
        this.postgresDeploymentName = 'postgres';
        this.postgresSelector = 'app=codeready,component=postgres';
        this.devfileRegistryDeploymentName = 'devfile-registry';
        this.devfileRegistrySelector = 'app=codeready,component=devfile-registry';
        this.pluginRegistryDeploymentName = 'plugin-registry';
        this.pluginRegistrySelector = 'app=codeready,component=plugin-registry';
        this.cheOperatorSelector = 'app=codeready-operator';
        this.kube = new kube_1.KubeHelper(flags);
        this.kubeTasks = new kube_2.KubeTasks(flags);
        this.che = new che_1.CheHelper(flags);
        this.cheAccessToken = flags['access-token'];
        this.cheNamespace = flags.chenamespace;
        this.cheDeploymentName = flags['deployment-name'];
    }
    /**
     * Returns tasks list that waits until every CodeReady Workspaces component will be started.
     *
     * Note that CodeReady Workspaces components statuses should be already set in context.
     *
     * @see che.checkIfCheIsInstalledTasks
     */
    waitDeployedChe(flags, command) {
        return [
            {
                title: 'PostgreSQL pod bootstrap',
                skip: () => !flags.multiuser,
                enabled: ctx => ctx.isPostgresDeployed && !ctx.isPostgresReady,
                task: () => this.kubeTasks.podStartTasks(command, this.postgresSelector, this.cheNamespace)
            },
            {
                title: 'Keycloak pod bootstrap',
                skip: () => !flags.multiuser,
                enabled: ctx => ctx.isKeycloakDeployed && !ctx.isKeycloakReady,
                task: () => this.kubeTasks.podStartTasks(command, this.keycloakSelector, this.cheNamespace)
            },
            {
                title: 'Devfile registry pod bootstrap',
                enabled: ctx => ctx.isDevfileRegistryDeployed && !ctx.isDevfileRegistryReady,
                task: () => this.kubeTasks.podStartTasks(command, this.devfileRegistrySelector, this.cheNamespace)
            },
            {
                title: 'Plugin registry pod bootstrap',
                enabled: ctx => ctx.isPluginRegistryDeployed && !ctx.isPluginRegistryReady,
                task: () => this.kubeTasks.podStartTasks(command, this.pluginRegistrySelector, this.cheNamespace)
            },
            {
                title: 'CodeReady Workspaces pod bootstrap',
                enabled: ctx => !ctx.isCheReady,
                task: () => this.kubeTasks.podStartTasks(command, this.cheSelector, this.cheNamespace)
            },
            ...this.retrieveEclipseCheUrl(flags),
            ...this.checkEclipseCheStatus()
        ];
    }
    /**
     * Returns list of tasks that checks if CodeReady Workspaces is already installed.
     *
     * After executing the following properties are set in context:
     * is[Component]Deployed, is[Component]Stopped, is[Component]Ready
     * where component is one the: Che, Keycloak, Postgres, PluginRegistry, DevfileRegistry
     */
    checkIfCheIsInstalledTasks(_flags, command) {
        return [
            {
                title: `Verify if CodeReady Workspaces is deployed into namespace \"${this.cheNamespace}\"`,
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    if (yield this.kube.deploymentExist(this.cheDeploymentName, this.cheNamespace)) {
                        // helm chart and CodeReady Workspaces operator use a deployment
                        ctx.isCheDeployed = true;
                        ctx.isCheReady = yield this.kube.deploymentReady(this.cheDeploymentName, this.cheNamespace);
                        if (!ctx.isCheReady) {
                            ctx.isCheStopped = yield this.kube.deploymentStopped(this.cheDeploymentName, this.cheNamespace);
                        }
                        ctx.isKeycloakDeployed = yield this.kube.deploymentExist(this.keycloakDeploymentName, this.cheNamespace);
                        if (ctx.isKeycloakDeployed) {
                            ctx.isKeycloakReady = yield this.kube.deploymentReady(this.keycloakDeploymentName, this.cheNamespace);
                            if (!ctx.isKeycloakReady) {
                                ctx.isKeycloakStopped = yield this.kube.deploymentStopped(this.keycloakDeploymentName, this.cheNamespace);
                            }
                        }
                        ctx.isPostgresDeployed = yield this.kube.deploymentExist(this.postgresDeploymentName, this.cheNamespace);
                        if (ctx.isPostgresDeployed) {
                            ctx.isPostgresReady = yield this.kube.deploymentReady(this.postgresDeploymentName, this.cheNamespace);
                            if (!ctx.isPostgresReady) {
                                ctx.isPostgresStopped = yield this.kube.deploymentStopped(this.postgresDeploymentName, this.cheNamespace);
                            }
                        }
                        ctx.isDevfileRegistryDeployed = yield this.kube.deploymentExist(this.devfileRegistryDeploymentName, this.cheNamespace);
                        if (ctx.isDevfileRegistryDeployed) {
                            ctx.isDevfileRegistryReady = yield this.kube.deploymentReady(this.devfileRegistryDeploymentName, this.cheNamespace);
                            if (!ctx.isDevfileRegistryReady) {
                                ctx.isDevfileRegistryStopped = yield this.kube.deploymentStopped(this.devfileRegistryDeploymentName, this.cheNamespace);
                            }
                        }
                        ctx.isPluginRegistryDeployed = yield this.kube.deploymentExist(this.pluginRegistryDeploymentName, this.cheNamespace);
                        if (ctx.isPluginRegistryDeployed) {
                            ctx.isPluginRegistryReady = yield this.kube.deploymentReady(this.pluginRegistryDeploymentName, this.cheNamespace);
                            if (!ctx.isPluginRegistryReady) {
                                ctx.isPluginRegistryStopped = yield this.kube.deploymentStopped(this.pluginRegistryDeploymentName, this.cheNamespace);
                            }
                        }
                    }
                    if (!ctx.isCheDeployed) {
                        task.title = `${task.title}...it is not`;
                    }
                    else {
                        return new Listr([
                            {
                                enabled: () => ctx.isCheDeployed,
                                title: `Found ${ctx.isCheStopped ? 'stopped' : 'running'} CodeReady Workspaces deployment`,
                                task: () => { }
                            },
                            {
                                enabled: () => ctx.isPostgresDeployed,
                                title: `Found ${ctx.isPostgresStopped ? 'stopped' : 'running'} postgres deployment`,
                                task: () => { }
                            },
                            {
                                enabled: () => ctx.isKeycloakDeployed,
                                title: `Found ${ctx.isKeycloakStopped ? 'stopped' : 'running'} keycloak deployment`,
                                task: () => { }
                            },
                            {
                                enabled: () => ctx.isPluginRegistryDeployed,
                                title: `Found ${ctx.isPluginRegistryStopped ? 'stopped' : 'running'} plugin registry deployment`,
                                task: () => { }
                            },
                            {
                                enabled: () => ctx.isDevfileRegistryDeployed,
                                title: `Found ${ctx.isDevfileRegistryStopped ? 'stopped' : 'running'} devfile registry deployment`,
                                task: () => { }
                            }
                        ]);
                    }
                })
            },
            {
                title: 'Check CodeReady Workspaces server status',
                enabled: (ctx) => ctx.isCheDeployed && ctx.isCheReady,
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    let cheURL = '';
                    try {
                        cheURL = yield this.che.cheURL(this.cheNamespace);
                        const status = yield this.che.getCheServerStatus(cheURL);
                        ctx.isAuthEnabled = yield this.che.isAuthenticationEnabled(cheURL);
                        const auth = ctx.isAuthEnabled ? '(auth enabled)' : '(auth disabled)';
                        task.title = `${task.title}...${status} ${auth}`;
                    }
                    catch (error) {
                        command.error(`E_CHECK_CHE_STATUS_FAIL - Failed to check CodeReady Workspaces status (URL: ${cheURL}). ${error.message}`);
                    }
                })
            }
        ];
    }
    /**
     * Returns tasks list which scale down all CodeReady Workspaces components which are deployed.
     * It requires {@link this#checkIfCheIsInstalledTasks} to be executed before.
     *
     * @see [CheTasks](#checkIfCheIsInstalledTasks)
     */
    scaleCheUpTasks(_command) {
        return [
            {
                title: 'Scaling up CodeReady Workspaces Deployments',
                enabled: (ctx) => ctx.isCheDeployed,
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    if (ctx.isPostgresDeployed) {
                        yield this.kube.scaleDeployment(this.postgresDeploymentName, this.cheNamespace, 1);
                    }
                    if (ctx.isKeycloakDeployed) {
                        yield this.kube.scaleDeployment(this.keycloakDeploymentName, this.cheNamespace, 1);
                    }
                    if (ctx.isPluginRegistryDeployed) {
                        yield this.kube.scaleDeployment(this.pluginRegistryDeploymentName, this.cheNamespace, 1);
                    }
                    if (ctx.isDevfileRegistryDeployed) {
                        yield this.kube.scaleDeployment(this.devfileRegistryDeploymentName, this.cheNamespace, 1);
                    }
                    yield this.kube.scaleDeployment(this.cheDeploymentName, this.cheNamespace, 1);
                    task.title = `${task.title}...done.`;
                })
            },
            {
                title: `CodeReady Workspaces is already running in namespace \"${this.cheNamespace}\".`,
                enabled: (ctx) => (ctx.isCheDeployed && ctx.isCheAvailable),
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    ctx.cheDeploymentExist = true;
                    ctx.cheIsAlreadyRunning = true;
                    ctx.cheURL = yield this.che.cheURL(this.cheNamespace);
                    task.title = yield `${task.title}...it's URL is ${ctx.cheURL}`;
                })
            }
        ];
    }
    /**
     * Returns tasks list which scale down all CodeReady Workspaces components which are deployed.
     * It requires {@link this#checkIfCheIsInstalledTasks} to be executed before.
     *
     * @see [CheTasks](#checkIfCheIsInstalledTasks)
     */
    scaleCheDownTasks(command) {
        return [{
                title: 'Stop CodeReady Workspaces server and wait until it\'s ready to shutdown',
                enabled: (ctx) => !ctx.isCheStopped,
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    if (ctx.isAuthEnabled && !this.cheAccessToken) {
                        command.error('E_AUTH_REQUIRED - CodeReady Workspaces authentication is enabled and an access token need to be provided (flag --access-token). ' +
                            `For instructions to retrieve a valid access token refer to ${constants_1.DOCS_LINK_AUTH_TO_CHE_SERVER_VIA_OPENID}`);
                    }
                    try {
                        const cheURL = yield this.che.cheURL(this.cheNamespace);
                        yield this.che.startShutdown(cheURL, this.cheAccessToken);
                        yield this.che.waitUntilReadyToShutdown(cheURL);
                        task.title = yield `${task.title}...done`;
                    }
                    catch (error) {
                        command.error(`E_SHUTDOWN_CHE_SERVER_FAIL - Failed to shutdown CodeReady Workspaces server. ${error.message}`);
                    }
                })
            },
            {
                title: `Scale \"${this.cheDeploymentName}\" deployment to zero`,
                enabled: (ctx) => !ctx.isCheStopped,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    try {
                        yield this.kube.scaleDeployment(this.cheDeploymentName, this.cheNamespace, 0);
                        task.title = yield `${task.title}...done`;
                    }
                    catch (error) {
                        command.error(`E_SCALE_DEPLOY_FAIL - Failed to scale deployment. ${error.message}`);
                    }
                })
            },
            {
                title: 'Scale \"keycloak\" deployment to zero',
                enabled: (ctx) => ctx.isKeycloakDeployed && !ctx.isKeycloakStopped,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    try {
                        yield this.kube.scaleDeployment(this.keycloakDeploymentName, this.cheNamespace, 0);
                        task.title = yield `${task.title}...done`;
                    }
                    catch (error) {
                        command.error(`E_SCALE_DEPLOY_FAIL - Failed to scale keycloak deployment. ${error.message}`);
                    }
                })
            },
            {
                title: 'Scale \"postgres\" deployment to zero',
                enabled: (ctx) => ctx.isPostgresDeployed && !ctx.isPostgresStopped,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    try {
                        yield this.kube.scaleDeployment(this.postgresDeploymentName, this.cheNamespace, 0);
                        task.title = yield `${task.title}...done`;
                    }
                    catch (error) {
                        command.error(`E_SCALE_DEPLOY_FAIL - Failed to scale postgres deployment. ${error.message}`);
                    }
                })
            },
            {
                title: 'Scale \"devfile registry\" deployment to zero',
                enabled: (ctx) => ctx.isDevfileRegistryDeployed && !ctx.isDevfileRegistryStopped,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    try {
                        yield this.kube.scaleDeployment(this.devfileRegistryDeploymentName, this.cheNamespace, 0);
                        task.title = yield `${task.title}...done`;
                    }
                    catch (error) {
                        command.error(`E_SCALE_DEPLOY_FAIL - Failed to scale devfile-registry deployment. ${error.message}`);
                    }
                })
            },
            {
                title: 'Scale \"plugin registry\" deployment to zero',
                enabled: (ctx) => ctx.isPluginRegistryDeployed && !ctx.isPluginRegistryStopped,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    try {
                        yield this.kube.scaleDeployment(this.pluginRegistryDeploymentName, this.cheNamespace, 0);
                        task.title = yield `${task.title}...done`;
                    }
                    catch (error) {
                        command.error(`E_SCALE_DEPLOY_FAIL - Failed to scale plugin-registry deployment. ${error.message}`);
                    }
                })
            }];
    }
    /**
     * Returns tasks which remove all CodeReady Workspaces related resources.
     */
    deleteTasks(flags) {
        return [
            {
                title: 'Delete all deployments',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.kube.deleteAllDeployments(flags.chenamespace);
                    task.title = yield `${task.title}...OK`;
                })
            },
            {
                title: 'Delete all services',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.kube.deleteAllServices(flags.chenamespace);
                    task.title = yield `${task.title}...OK`;
                })
            },
            {
                title: 'Delete all ingresses',
                enabled: (ctx) => !ctx.isOpenShift,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.kube.deleteAllIngresses(flags.chenamespace);
                    task.title = yield `${task.title}...OK`;
                })
            },
            {
                title: 'Delete all routes',
                enabled: (ctx) => ctx.isOpenShift,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.oc.deleteAllRoutes(flags.chenamespace);
                    task.title = yield `${task.title}...OK`;
                })
            },
            {
                title: 'Delete configmaps for CodeReady Workspaces server and operator',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    if (yield this.kube.getConfigMap('che', flags.chenamespace)) {
                        yield this.kube.deleteConfigMap('che', flags.chenamespace);
                    }
                    if (yield this.kube.getConfigMap('codeready-operator', flags.chenamespace)) {
                        yield this.kube.deleteConfigMap('codeready-operator', flags.chenamespace);
                    }
                    task.title = yield `${task.title}...OK`;
                })
            },
            {
                title: 'Delete rolebindings che, che-workspace-exec and che-workspace-view',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    if (yield this.kube.roleBindingExist('che', flags.chenamespace)) {
                        yield this.kube.deleteRoleBinding('che', flags.chenamespace);
                    }
                    if (yield this.kube.roleBindingExist('codeready-operator', flags.chenamespace)) {
                        yield this.kube.deleteRoleBinding('codeready-operator', flags.chenamespace);
                    }
                    if (yield this.kube.roleBindingExist('che-workspace-exec', flags.chenamespace)) {
                        yield this.kube.deleteRoleBinding('che-workspace-exec', flags.chenamespace);
                    }
                    if (yield this.kube.roleBindingExist('che-workspace-view', flags.chenamespace)) {
                        yield this.kube.deleteRoleBinding('che-workspace-view', flags.chenamespace);
                    }
                    task.title = yield `${task.title}...OK`;
                })
            },
            {
                title: 'Delete service accounts che, che-workspace',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    if (yield this.kube.serviceAccountExist('che', flags.chenamespace)) {
                        yield this.kube.deleteServiceAccount('che', flags.chenamespace);
                    }
                    if (yield this.kube.roleBindingExist('che-workspace', flags.chenamespace)) {
                        yield this.kube.deleteServiceAccount('che-workspace', flags.chenamespace);
                    }
                    task.title = yield `${task.title}...OK`;
                })
            },
            {
                title: 'Delete PVC postgres-data and che-data-volume',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    if (yield this.kube.persistentVolumeClaimExist('postgres-data', flags.chenamespace)) {
                        yield this.kube.deletePersistentVolumeClaim('postgres-data', flags.chenamespace);
                    }
                    if (yield this.kube.persistentVolumeClaimExist('che-data-volume', flags.chenamespace)) {
                        yield this.kube.deletePersistentVolumeClaim('che-data-volume', flags.chenamespace);
                    }
                    task.title = yield `${task.title}...OK`;
                })
            }
        ];
    }
    /**
     * Returns tasks which wait until pods are deleted.
     */
    waitPodsDeletedTasks() {
        return [
            {
                title: 'Wait until CodeReady Workspaces pod is deleted',
                enabled: (ctx) => !ctx.isCheStopped,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.kube.waitUntilPodIsDeleted(this.cheSelector, this.cheNamespace);
                    task.title = `${task.title}...done.`;
                })
            },
            {
                title: 'Wait until Keycloak pod is deleted',
                enabled: (ctx) => ctx.isKeycloakDeployed && !ctx.isKeycloakStopped,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.kube.waitUntilPodIsDeleted(this.keycloakSelector, this.cheNamespace);
                    task.title = `${task.title}...done.`;
                })
            },
            {
                title: 'Wait until Postgres pod is deleted',
                enabled: (ctx) => ctx.isPostgresDeployed && !ctx.isPostgresStopped,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.kube.waitUntilPodIsDeleted(this.postgresSelector, this.cheNamespace);
                    task.title = `${task.title}...done.`;
                })
            },
            {
                title: 'Wait until Devfile registry pod is deleted',
                enabled: (ctx) => ctx.isDevfileRegistryDeployed && !ctx.isDevfileRegistryStopped,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.kube.waitUntilPodIsDeleted(this.devfileRegistrySelector, this.cheNamespace);
                    task.title = `${task.title}...done.`;
                })
            },
            {
                title: 'Wait until Plugin registry pod is deleted',
                enabled: (ctx) => ctx.isPluginRegistryDeployed && !ctx.isPluginRegistryStopped,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.kube.waitUntilPodIsDeleted(this.pluginRegistrySelector, this.cheNamespace);
                    task.title = `${task.title}...done.`;
                })
            }
        ];
    }
    verifyCheNamespaceExistsTask(flags, command) {
        return [{
                title: `Verify if namespace '${flags.chenamespace}' exists`,
                task: () => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    if (!(yield this.che.cheNamespaceExist(flags.chenamespace))) {
                        command.error(`E_BAD_NS - Namespace does not exist.\nThe Kubernetes Namespace "${flags.chenamespace}" doesn't exist. The configuration cannot be injected.\nFix with: verify the namespace where workspace is running (kubectl get --all-namespaces deployment | grep workspace)`, { code: 'EBADNS' });
                    }
                })
            }];
    }
    /**
     * Verifies if workspace running and puts #V1Pod into a context.
     */
    verifyWorkspaceRunTask(flags, command) {
        return [{
                title: 'Verify if the workspaces is running',
                task: (ctx) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    ctx.pod = yield this.che.getWorkspacePodName(flags.chenamespace, flags.workspace).catch(e => command.error(e.message));
                })
            }];
    }
    /**
     * Return tasks to collect CodeReady Workspaces logs.
     */
    serverLogsTasks(flags, follow) {
        return [
            {
                title: `${follow ? 'Start following' : 'Read'} Operator logs`,
                skip: () => flags.installer !== 'operator' && flags.installer !== 'olm',
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.che.readPodLog(flags.chenamespace, this.cheOperatorSelector, ctx.directory, follow);
                    task.title = `${task.title}...done`;
                })
            },
            {
                title: `${follow ? 'Start following' : 'Read'} CodeReady Workspaces logs`,
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.che.readPodLog(flags.chenamespace, this.cheSelector, ctx.directory, follow);
                    task.title = yield `${task.title}...done`;
                })
            },
            {
                title: `${follow ? 'Start following' : 'Read'} Postgres logs`,
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.che.readPodLog(flags.chenamespace, this.postgresSelector, ctx.directory, follow);
                    task.title = yield `${task.title}...done`;
                })
            },
            {
                title: `${follow ? 'Start following' : 'Read'} Keycloak logs`,
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.che.readPodLog(flags.chenamespace, this.keycloakSelector, ctx.directory, follow);
                    task.title = yield `${task.title}...done`;
                })
            },
            {
                title: `${follow ? 'Start following' : 'Read'} Plugin registry logs`,
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.che.readPodLog(flags.chenamespace, this.pluginRegistrySelector, ctx.directory, follow);
                    task.title = yield `${task.title}...done`;
                })
            },
            {
                title: `${follow ? 'Start following' : 'Read'} Devfile registry logs`,
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.che.readPodLog(flags.chenamespace, this.devfileRegistrySelector, ctx.directory, follow);
                    task.title = yield `${task.title}...done`;
                })
            }
        ];
    }
    workspaceLogsTasks(namespace, workspaceId) {
        return [
            {
                title: 'Read workspace logs',
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    ctx['workspace-run'] = yield this.che.readWorkspacePodLog(namespace, workspaceId, ctx.directory);
                    task.title = `${task.title}...done`;
                })
            }
        ];
    }
    namespaceEventsTask(namespace, command, follow) {
        return [
            {
                title: `${follow ? 'Start following' : 'Read'} namespace events`,
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.che.readNamespaceEvents(namespace, ctx.directory, follow).catch(e => command.error(e.message));
                    task.title = yield `${task.title}...done`;
                })
            }
        ];
    }
    debugTask(flags) {
        return [
            {
                title: 'Find CodeReady Workspaces server pod',
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const chePods = yield this.kube.listNamespacedPod(flags.chenamespace, undefined, this.cheSelector);
                    if (chePods.items.length === 0) {
                        throw new Error(`CodeReady Workspaces server pod not found in the namespace '${flags.chenamespace}'`);
                    }
                    ctx.podName = chePods.items[0].metadata.name;
                    task.title = `${task.title}...done`;
                })
            },
            {
                title: 'Check if debug mode is enabled',
                task: (task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const configMap = yield this.kube.getConfigMap('che', flags.chenamespace);
                    if (!configMap || configMap.data.CHE_DEBUG_SERVER !== 'true') {
                        throw new Error('CodeReady Workspaces server should be redeployed with \'--debug\' flag');
                    }
                    task.title = `${task.title}...done`;
                })
            },
            {
                title: `Forward port '${flags['debug-port']}'`,
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.kube.portForward(ctx.podName, flags.chenamespace, flags['debug-port']);
                    task.title = `${task.title}...done`;
                })
            }
        ];
    }
    retrieveEclipseCheUrl(flags) {
        return [
            {
                title: 'Retrieving CodeReady Workspaces server URL',
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    ctx.cheURL = yield this.che.cheURL(flags.chenamespace);
                    task.title = `${task.title}... ${ctx.cheURL}`;
                })
            }
        ];
    }
    checkEclipseCheStatus() {
        return [
            {
                title: 'CodeReady Workspaces status check',
                task: (ctx) => tslib_1.__awaiter(this, void 0, void 0, function* () { return this.che.isCheServerReady(ctx.cheURL); })
            }
        ];
    }
    checkIsAuthenticationEnabled() {
        return [
            {
                title: 'Checking authentication',
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    ctx.isAuthEnabled = yield this.che.isAuthenticationEnabled(ctx.cheURL);
                    if (ctx.isAuthEnabled && !this.cheAccessToken) {
                        throw new Error('E_AUTH_REQUIRED - CodeReady Workspaces authentication is enabled but access token is missed. Use --access-token to provide access token.');
                    }
                    task.title = `${task.title}... ${ctx.isAuthEnabled ? '(auth enabled)' : '(auth disabled)'}`;
                })
            }
        ];
    }
}
exports.CheTasks = CheTasks;
//# sourceMappingURL=che.js.map