"use strict";
/*********************************************************************
 * Copyright (c) 2020 Red Hat, Inc.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 **********************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const command_1 = require("@oclif/command");
const flags_1 = require("@oclif/parser/lib/flags");
const fs = require("fs");
const Listr = require("listr");
const os = require("os");
const path = require("path");
const che_1 = require("../../api/che");
const common_flags_1 = require("../../common-flags");
const constants_1 = require("../../constants");
const che_2 = require("../../tasks/che");
const api_1 = require("../../tasks/platforms/api");
class Export extends command_1.Command {
    run() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { flags } = this.parse(Export);
            const ctx = {};
            const cheHelper = new che_1.CheHelper(flags);
            const cheTasks = new che_2.CheTasks(flags);
            const apiTasks = new api_1.ApiTasks();
            const tasks = new Listr([], { renderer: 'silent' });
            tasks.add(apiTasks.testApiTasks(flags, this));
            tasks.add(cheTasks.verifyCheNamespaceExistsTask(flags, this));
            try {
                yield tasks.run(ctx);
                const cheCaCert = yield cheHelper.retrieveCheCaCert(flags.chenamespace);
                if (cheCaCert) {
                    const targetFile = yield cheHelper.saveCheCaCert(cheCaCert, this.getTargetFile(flags.destination));
                    this.log(`CodeReady Workspaces self-signed CA certificate is exported to ${targetFile}`);
                }
                else {
                    this.log('Seems commonly trusted certificate is used.');
                }
            }
            catch (error) {
                this.error(error);
            }
        });
    }
    /**
     * Handles certificate target location and returns string which points to the target file.
     */
    getTargetFile(destinaton) {
        if (!destinaton) {
            return path.join(os.homedir(), constants_1.DEFAULT_CA_CERT_FILE_NAME);
        }
        if (fs.existsSync(destinaton)) {
            return fs.lstatSync(destinaton).isDirectory() ? path.join(destinaton, constants_1.DEFAULT_CA_CERT_FILE_NAME) : destinaton;
        }
        this.error(`Given path "${destinaton}" doesn't exist.`);
    }
}
exports.default = Export;
Export.description = 'Retrieves CodeReady Workspaces self-signed certificate';
Export.flags = {
    help: command_1.flags.help({ char: 'h' }),
    chenamespace: common_flags_1.cheNamespace,
    destination: flags_1.string({
        char: 'd',
        description: `Destination where to store Che self-signed CA certificate.
                    If the destination is a file (might not exist), then the certificate will be saved there in PEM format.
                    If the destination is a directory, then ${constants_1.DEFAULT_CA_CERT_FILE_NAME} file will be created there with Che certificate in PEM format.
                    If this option is ommited, then Che certificate will be stored in user's home directory as ${constants_1.DEFAULT_CA_CERT_FILE_NAME}`,
        env: 'CHE_CA_CERT_LOCATION',
        default: ''
    }),
    'skip-kubernetes-health-check': common_flags_1.skipKubeHealthzCheck
};
//# sourceMappingURL=export.js.map