package org.jboss.eap.util.xp.patch.stream.manager;

import java.nio.file.Path;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * @author <a href="mailto:kabir.khan@jboss.com">Kabir Khan</a>
 */
public class ManagerStatus {
    private final Map<ManagerState, State> states;
    private final ManagerState managerState;
    private final State currentState;

    private ManagerStatus(Map<ManagerState, State> states, ManagerState managerState) {
        this.states = Collections.unmodifiableMap(states);
        this.managerState = managerState;
        this.currentState = states.get(managerState);
    }

    ManagerState getManagerState() {
        return managerState;
    }

    static ManagerStatus determineStatus(Path jbossHome, Path modulesDir) throws Exception {
        List<String> moduleLayerNames = ManagerActionUtils.getManagedModuleLayerNames();
        List<String> patchStreamNames = ManagerActionUtils.getManagedPatchStreamNames();

        FileSet installedFileSet = FileSet.createInstalled(jbossHome, modulesDir, moduleLayerNames, patchStreamNames);

        Map<ManagerState, State> states = new HashMap<>();
        states.put(ManagerState.INSTALLED, new InstalledState(installedFileSet, moduleLayerNames, patchStreamNames));
        states.put(ManagerState.CLEAN, new CleanState(moduleLayerNames, patchStreamNames));

        ManagerState currentManagerState;
        if (installedFileSet.getStatus() == FileSet.Status.NONE_EXIST) {
            currentManagerState = ManagerState.CLEAN;
        } else if (installedFileSet.getStatus() == FileSet.Status.ALL_EXIST) {
            currentManagerState = ManagerState.INSTALLED;
        } else {
            // inconsistent
            FileSet inconsistent = FileSet.createInconsistent(installedFileSet);
            states.put(ManagerState.INCONSISTENT, new InconsistentState(inconsistent, moduleLayerNames, patchStreamNames));
            currentManagerState = ManagerState.INCONSISTENT;
        }

        return new ManagerStatus(states, currentManagerState);
    }

    Set<ManagerCommand> getAvailableCommands() {
        return currentState.getAvailableCommands();
    }

    public FileSet getFileSet() {
        return currentState.fileSet;
    }

    public FileSet getToFileSet(ManagerState toManagerState) {
        return states.get(toManagerState).fileSet;
    }

    private static abstract class State {
        private final ManagerState managerState;
        private final FileSet fileSet;
        private final List<String> moduleLayerNames;
        private final List<String> patchStreamNames;

        public State(ManagerState managerState, FileSet fileSet, List<String> moduleLayerNames, List<String> patchStreamNames) {
            this.managerState = managerState;
            this.fileSet = fileSet;
            this.moduleLayerNames = moduleLayerNames;
            this.patchStreamNames = patchStreamNames;
        }

        Set<ManagerCommand> getAvailableCommands() {
            return managerState.getAvailableCommands();
        }
    }

    private static class CleanState extends State {
        CleanState(List<String> moduleLayerNames, List<String> patchStreamNames) {
            super(ManagerState.CLEAN, null, moduleLayerNames, patchStreamNames);
        }
    }

    private static class InstalledState extends State {
        public InstalledState(FileSet fileSet, List<String> moduleLayerNames, List<String> patchStreamNames) {
            super(ManagerState.INSTALLED, fileSet, moduleLayerNames, patchStreamNames);
        }
    }

    private static class InconsistentState extends State {
        public InconsistentState(FileSet fileSet, List<String> moduleLayerNames, List<String> patchStreamNames) {
            super(ManagerState.INCONSISTENT, fileSet, moduleLayerNames, patchStreamNames);
        }
    }
}
