/*
 * JBoss, Home of Professional Open Source
 * Copyright 2020, JBoss Inc., and individual contributors as indicated
 * by the @authors tag.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.jboss.eap.util.xp.patch.stream.manager;

import java.io.IOException;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * @author <a href="mailto:kabir.khan@jboss.com">Kabir Khan</a>
 */
public class UpgradeManagerAction extends ManagerAction {

    private static final String PATCH_STREAM_FILE_CONTENTS =
            "current-version=0.0.0\npatches=\ncumulative-patch-id=base\ninstalled-patches=";

    private final FileSet fromFileSet;
    private final FileSet toFileSet;
    private final URL jarUrl;
    private final Path jbossHome;
    private final Path modulesDir;
    private final Path serverConfigDir;
    private final Path basePatch;
    private final Path xpPatch;
    private final ServerWrapper serverWrapper;
    private final List<Path> createdLayers = new ArrayList<>();
    private final List<Path> createdPatchStreams = new ArrayList<>();

    FileSet backupSet;


    private UpgradeManagerAction(ManagerStatus status, ServerWrapper serverWrapper, boolean supportPolicyAccepted, URL jarUrl, Path jbossHome,
                                 Path modulesDir, Path serverConfigDir, Path basePatch, Path xpPatch) {
        super(ManagerCommand.UPGRADE, status, ManagerState.INSTALLED, supportPolicyAccepted);
        this.fromFileSet = status.getInstalledFileSet();
        this.toFileSet = status.getToFileSet(ManagerState.INSTALLED);
        this.jarUrl = jarUrl;
        this.jbossHome = jbossHome;
        this.modulesDir = modulesDir;
        this.serverConfigDir = serverConfigDir;
        this.basePatch = basePatch;
        this.xpPatch = xpPatch;
        this.serverWrapper = serverWrapper;
    }

    static UpgradeManagerAction create(ManagerStatus status, ServerWrapper serverWrapper, boolean supportPolicyAccepted, URL jarUrl, Path jbossHome,
                                       Path modulesDir, Path serverConfigDir, Path basePatch, Path xpPatch) throws Exception {
        return new UpgradeManagerAction(status, serverWrapper, supportPolicyAccepted, jarUrl, jbossHome,
                modulesDir, serverConfigDir, basePatch, xpPatch);
    }

    @Override
    public void doExecute() throws Exception {
        System.out.println(ManagerLogger.LOGGER.startingUpgrade());
        RemoveUtils removeUtils = new RemoveUtils(serverWrapper, jbossHome);
        try {
            try {
                backupExistingInstalledFiles();

                try {
                    removeUtils.rollbackPatchStreamPatches(Collections.singletonList(fromFileSet));
                } catch (Exception e) {
                    System.err.println(ManagerLogger.LOGGER.errorRemovingPatches());
                    e.printStackTrace();
                }
                removeUtils.removeFiles(Collections.singletonList(fromFileSet));

                SetupManagerAction setup = SetupManagerAction.createForUpgrade(status, serverWrapper, supportPolicyAccepted, jarUrl, jbossHome, modulesDir, serverConfigDir, basePatch, xpPatch);
                setup.doExecute();

                System.err.println(ManagerLogger.LOGGER.successfullyUpgradedExpansionPack(ManagerManifestConfig.INSTANCE.getCurrentXpConfig().getVersion()));
            } catch (Exception e) {
                System.err.println(ManagerLogger.LOGGER.errorUpgrading(e));

                boolean success = true;
                success = success && restoreFiles(backupSet.getPatchStreamFiles(), fromFileSet.getPatchStreamFiles());
                success = success && restoreFiles(backupSet.getModuleLayerDirs(), fromFileSet.getModuleLayerDirs());
                success = success && restoreFile(backupSet.getLayersConf(), fromFileSet.getLayersConf());
                if (!success) {
                    System.err.println(ManagerLogger.LOGGER.errorRestoringFilesInUpgradeInconsistent());
                }
            }
        } finally {
            if (backupSet != null) {
                removeUtils.removeFiles(backupSet);
            }
        }

    }


    private void backupExistingInstalledFiles() throws Exception {
        System.out.println(ManagerLogger.LOGGER.backingUpCurrentPatchStreamFiles());

        backupSet = fromFileSet.createBackupClone();
        Files.copy(fromFileSet.getLayersConf(), backupSet.getLayersConf());

        backupPaths(fromFileSet.getPatchStreamFiles(), backupSet.getPatchStreamFiles());
        backupPaths(fromFileSet.getModuleLayerDirs(), backupSet.getModuleLayerDirs());

        System.out.println(ManagerLogger.LOGGER.successfullyBackedUpCurrentPatchStreamFiles());
    }

    private void backupPaths(List<Path> paths, List<Path> backups) throws IOException {
        for (int i = 0; i < paths.size(); i++) {
            Files.copy(paths.get(i), backups.get(i));
        }
    }

    private boolean restoreFiles(List<Path> backups, List<Path> toRestore) {
        boolean success = true;
        for (int i = 0; i < backups.size(); i++) {
            success = success && restoreFile(backups.get(i), toRestore.get(i));
        }
        return success;
    }

    private boolean restoreFile(Path backup, Path toRestore) {
        if (Files.exists(backup)) {
            try {
                if (Files.exists(toRestore)) {
                    RemoveUtils.remove(toRestore);
                }
                Files.move(backup, toRestore);
            } catch (IOException e) {
                System.out.println(ManagerLogger.LOGGER.errorHappenedRestoringBackup(e, backup, toRestore));
                return false;
            }
        }
        return true;
    }
}
