package org.jboss.eap.util.xp.patch.stream.manager;

import java.io.BufferedInputStream;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.List;
import java.util.Properties;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * @author <a href="mailto:kabir.khan@jboss.com">Kabir Khan</a>
 */
public class StatusManagerAction extends ManagerAction {
    private static final String JBOSS_EAP_PATCH_STREAM_NAME = "JBoss EAP";
    private static final String CONF_EXTENSION = ".conf";
    private static final String JBOSS_EAP_CONF = JBOSS_EAP_PATCH_STREAM_NAME + CONF_EXTENSION;
    private static final String BASE_CUMULATIVE_PATCH_ID = "base";
    private static final String CUMULATIVE_PATCH_ID = "cumulative-patch-id";

    private final Path jbossHome;

    public StatusManagerAction(ManagerStatus status, Path jbossHome) {
        super(ManagerCommand.STATUS, status, null,false);
        this.jbossHome = jbossHome;
    }

    @Override
    public Status doExecute() throws Exception {
        System.out.println();
        switch (status.getManagerState()) {
            case CLEAN:
                System.out.println(ManagerLogger.LOGGER.installationStatusClean());
                break;
            case INCONSISTENT:
                System.out.println(ManagerLogger.LOGGER.installationStatusInconsistent());
                if (status.hasSeveralXPPatchStreams()) {
                    System.out.println(ManagerLogger.LOGGER.installationStatusInconsistentSeveralXPPatchStreams());
                }
                break;
            case INSTALLED:
            case OLD_INSTALLED:
                ManagerManifestConfig config = ManagerManifestConfig.INSTANCE;
                String installedVersion = status.getInstalledXPConfig().getVersion();
                String upgradeMessage = "";
                if (status.getManagerState() == ManagerState.OLD_INSTALLED) {
                    String currentVersion = config.getCurrentXpConfig().getVersion();
                    upgradeMessage = ManagerLogger.LOGGER.oldVersionInstalledPleaseUpgrade(currentVersion);
                }

                System.out.println(
                        ManagerLogger.LOGGER.installationStatusInstalled(
                                ManagerLogger.LOGGER.xpSupportPolicy(),
                                installedVersion,
                                upgradeMessage));

                break;
        }


        outputStreamsAndCumultivePatchIds();

        System.out.println(ManagerLogger.LOGGER.installationStatusAvailableCommands(status.getAvailableCommands()));
        return Status.OK;
    }

    private void outputStreamsAndCumultivePatchIds() throws Exception {

        // Read the information from the installation/*.conf files so we don't pollute the status output
        // with output from starting the embedded server

        Path installationDir = jbossHome.resolve(".installation");
        System.out.println(ManagerLogger.LOGGER.installedPatchStreamsAndCumulativePatchIdsHeader());
        if (Files.exists(installationDir)) {
            Path jbossEapConf = installationDir.resolve(JBOSS_EAP_CONF);
            if (!Files.exists(jbossEapConf)) {
                System.out.println(
                        ManagerLogger.LOGGER.patchStreamAndCumulativePatchId(JBOSS_EAP_PATCH_STREAM_NAME, BASE_CUMULATIVE_PATCH_ID));
            } else {
                outputStreamAndCumulativePatchId(jbossEapConf);
            }

            try (Stream<Path> stream =
                         Files
                                 .list(installationDir)
                                 .filter(p -> {
                                     String fileName = p.getFileName().toString();
                                     return !fileName.equals(JBOSS_EAP_CONF) && fileName.endsWith(CONF_EXTENSION);
                                 })) {
                List<Path> paths = stream.collect(Collectors.toList());
                for (Path confFile : paths) {
                    outputStreamAndCumulativePatchId(confFile);
                }
            }
        }
        System.out.println();
    }

    private void outputStreamAndCumulativePatchId(Path confFile) {
        Properties properties = new Properties();
        try (InputStream in = new BufferedInputStream(new FileInputStream(confFile.toFile()))) {
            properties.load(in);
        } catch (IOException ignore) {
            //Should not happen, and is not a critical error
            ignore.printStackTrace();
        }

        String confFileName = confFile.getFileName().toString();
        String streamName = confFileName.substring(0, confFileName.indexOf(CONF_EXTENSION));
        String cumulativePatchId = properties.getProperty(CUMULATIVE_PATCH_ID);
        if (cumulativePatchId == null || cumulativePatchId.trim().length() == 0) {
            cumulativePatchId = BASE_CUMULATIVE_PATCH_ID;
        }

        System.out.println(ManagerLogger.LOGGER.patchStreamAndCumulativePatchId(streamName, cumulativePatchId));
    }
}
