package org.jboss.eap.util.xp.patch.stream.manager;

import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.jboss.eap.util.xp.patch.stream.manager.ManagerManifestConfig.XPConfig;

/**
 * @author <a href="mailto:kabir.khan@jboss.com">Kabir Khan</a>
 */
class FileSet {
    private final XPConfig xpConfig;
    private final boolean layersConfExists;
    private final int modulesCount;
    private final int patchStreamCount;
    private final Path layersConf;
    private final List<Path> moduleLayerDirs;
    private final List<Path> patchStreamFiles;

    private FileSet(
            XPConfig xpConfig,
            boolean layersConfExists,
            int modulesCount,
            int patchStreamCount,
            Path layersConf,
            List<Path> moduleLayerDirs,
            List<Path> patchStreamFiles) {
        this.xpConfig = xpConfig;
        this.layersConfExists = layersConfExists;
        this.modulesCount = modulesCount;
        this.patchStreamCount = patchStreamCount;
        this.layersConf = layersConf;
        this.moduleLayerDirs = Collections.unmodifiableList(moduleLayerDirs);
        this.patchStreamFiles = Collections.unmodifiableList(patchStreamFiles);
    }

    FileSet createBackupClone() {
        return new FileSet(
                xpConfig,
                layersConfExists,
                modulesCount,
                patchStreamCount,
                createBackupPath(layersConf),
                createBackupPaths(moduleLayerDirs),
                createBackupPaths(patchStreamFiles));
    }

    private List<Path> createBackupPaths(List<Path> paths) {
        List<Path> backups = new ArrayList<>();
        for (Path path : paths) {
            backups.add(createBackupPath(path));
        }
        return backups;
    }

    private Path createBackupPath(Path path) {
        return path.getParent().resolve("." + path.getFileName().toString() + ".bak");
    }

    public XPConfig getXpConfig() {
        return xpConfig;
    }

    public boolean isLayersConfExists() {
        return layersConfExists;
    }

    public FileStatus getModuleLayersStatus() {
        return createStatus(moduleLayerDirs.size(), modulesCount);
    }

    public FileStatus getPatchStreamDirStatus() {
        return createStatus(patchStreamFiles.size(), patchStreamCount);
    }

    private FileStatus createStatus(int expected, int actual) {
        if (expected == actual) {
            return FileStatus.ALL;
        } else if (actual > 0) {
            return FileStatus.SOME;
        }
        return FileStatus.NONE;
    }

    public boolean isInstalled() {
        return layersConfExists && getModuleLayersStatus() == FileStatus.ALL && getPatchStreamDirStatus() == FileStatus.ALL;
    }

    public Path getLayersConf() {
        return layersConf;
    }

    public List<Path> getModuleLayerDirs() {
        return moduleLayerDirs;
    }

    public List<Path> getPatchStreamFiles() {
        return patchStreamFiles;
    }

    static FileSet create(
            XPConfig xpConfig, Path jbossHome, Path modulesDir) throws Exception {
        return create(true, xpConfig, jbossHome, modulesDir);
    }

    private static FileSet create(boolean enabled, XPConfig xpConfig,
                                  Path jbossHome, Path modulesDir) throws Exception {
        Path layersConf = getLayersConf(modulesDir, enabled);
        List<Path> moduleLayerDirs = getManagedModuleLayerDirs(modulesDir, xpConfig.getModuleLayerNames(), enabled);
        List<Path> patchStreamFiles = getManagedPatchStreams(jbossHome, xpConfig.getPatchStreamNames(), enabled);

        // Determine status
        boolean layersConfExists = false;
        if (Files.exists(layersConf)) {
            layersConfExists = true;
        }

        int modulesCount = 0;
        for (Path p : moduleLayerDirs) {
            if (Files.exists(p)) {
                modulesCount++;
            }
        }

        int patchStreamCount = 0;
        for (Path p : patchStreamFiles) {
            if (Files.exists(p)) {
                patchStreamCount++;
            }
        }

        return new FileSet(
                xpConfig,
                layersConfExists,
                modulesCount,
                patchStreamCount,
                layersConf,
                moduleLayerDirs,
                patchStreamFiles);
    }

    private static Path getLayersConf(Path modulesDir, boolean enabled) {
        String fileName = "layers.conf";
        if (!enabled) {
            fileName = "." + fileName;
        }
        return modulesDir.resolve(fileName);
    }

    private static List<Path> getManagedModuleLayerDirs(Path modulesDir, List<String> moduleLayerNames, boolean enabled) throws Exception {
        List<Path> layers = new ArrayList<>();
        for (String name : moduleLayerNames) {

            String dirName = "system/layers/";
            if (!enabled) {
                dirName += ".";
            }
            dirName += name;

            Path path = modulesDir.resolve(dirName);
            layers.add(path);
        }
        return layers;
    }

    private static List<Path> getManagedPatchStreams(Path jbossHome, List<String> patchStreamNames, boolean enabled) throws Exception {
        List<Path> streams = new ArrayList<>();
        for (String name : patchStreamNames) {

            String confFileName = ".installation/";
            if (!enabled) {
                confFileName += ".";
            }
            confFileName += name + (enabled ? ".conf" : ".bak");

            Path path = jbossHome.resolve(confFileName);
            streams.add(path);
        }
        return streams;
    }

    enum FileStatus {
        ALL,
        SOME,
        NONE;
    }
}
