/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2024, Red Hat, Inc., and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.wildfly.test.integration.microprofile.health;

import org.jboss.arquillian.container.test.api.Deployment;
import org.jboss.arquillian.container.test.api.RunAsClient;
import org.jboss.arquillian.junit.Arquillian;
import org.jboss.as.arquillian.api.ContainerResource;
import org.jboss.as.arquillian.container.ManagementClient;
import org.jboss.shrinkwrap.api.ShrinkWrap;
import org.jboss.shrinkwrap.api.asset.StringAsset;
import org.jboss.shrinkwrap.api.spec.WebArchive;
import org.junit.Test;
import org.junit.runner.RunWith;

import java.io.IOException;

@RunWith(Arquillian.class)
@RunAsClient
public abstract class MicroProfileHealthDisabledDefaultProceduresMultiWarTestBase {
    static final String SERVICE_ONE = "service-one";
    static final String SERVICE_TWO = "service-two";

    abstract void checkGlobalOutcome(final ManagementClient managementClient, final String operation, final boolean mustBeUP,
                                     final String probeName, final Integer expectedChecksCount) throws IOException;

    @Deployment(name = SERVICE_ONE, order = 1)
    public static WebArchive createDeployment1() {
        return ShrinkWrap.create(WebArchive.class, SERVICE_ONE + ".war")
                // no deployment health checks provided, we want for WildFly to automatically add readiness + startup checks here
                // see MicroProfile_Health.adoc
                .addAsWebInfResource(new StringAsset("<beans bean-discovery-mode=\"all\"></beans>"), "beans.xml");
    }

    @Deployment(name = SERVICE_TWO, order = 2)
    public static WebArchive createDeployment2() {
        return ShrinkWrap.create(WebArchive.class, SERVICE_TWO + ".war")
                // we want the deployment to provide ready and startup checks here
                .addClasses(MyReadyProbe.class, SuccessfulStartupProbe.class)
                // but we disable all the server default procedures too
                .addAsManifestResource(new StringAsset("mp.health.disable-default-procedures=true"),
                        "microprofile-config.properties")
                .addAsWebInfResource(new StringAsset("<beans bean-discovery-mode=\"all\"></beans>"), "beans.xml");
    }

    @ContainerResource
    ManagementClient managementClient;

    @Test
    public void testDisabledDefaultProcedures() throws IOException {
        // 1. ready-service-one.war (automatically added)
        // 2. started-service-one.war (automatically added)
        // 3. MyReadyProbe (added by the service-two deployment)
        // 4. SuccessfulStartupCheck (added by the service-two deployment)
        checkGlobalOutcome(managementClient, "check", true, "myReadyProbe", 4);
    }
}
