/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2007, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.test.cluster.invokerha;

import java.rmi.server.UID;

import javax.transaction.Transaction;

import org.jboss.ha.framework.interfaces.LoadBalancePolicy;
import org.jboss.invocation.Invocation;
import org.jboss.invocation.Invoker;
import org.jboss.logging.Logger;
import org.jboss.test.cluster.invokerha.InvokerHaInfrastructure.InvokerHaFactory;
import org.jboss.test.cluster.invokerha.InvokerHaInfrastructure.TraceFirstAvailable;
import org.jboss.test.cluster.invokerha.InvokerHaInfrastructure.TraceFirstAvailableIdenticalAllProxies;
import org.jboss.test.cluster.invokerha.InvokerHaInfrastructure.TraceRandomRobin;
import org.jboss.test.cluster.invokerha.InvokerHaInfrastructure.TraceRoundRobin;
import org.jboss.test.cluster.invokerha.InvokerHaTransactionalMockUtils.MockTransaction;

import junit.framework.TestCase;

/**
 * Base class for invoker related tests that do not run within AS.
 * 
 * @author <a href="mailto:galder.zamarreno@jboss.com">Galder Zamarreno</a>
 */
public abstract class AbstractInvokerHa extends TestCase
{
   private static final Logger log = Logger.getLogger(AbstractInvokerHa.class);
   
   InvokerHaInfrastructure infrastructure;
   
   InvokerHaTransactionalMockUtils transactionalMockUtils;
   
   InvokerHaFactory invokerHaFactory;
   
   Invoker timeTellerProxy;
   
   Invoker systemTimeProxy;
   
   Object prevChosenTargetDateTimeTeller;
   
   Object prevChosenTargetSystemTimeTeller;

   protected void setUp(int serverCount, InvokerHaFactory factory) throws Exception
   {
      super.setUp();
      
      invokerHaFactory = factory; 
      
      transactionalMockUtils = new InvokerHaTransactionalMockUtils();
      
      infrastructure = invokerHaFactory.getInvokerHaInfrastructure(2);
      
      infrastructure.registerManagedObjects();
      
      infrastructure.deployServers();
      
      infrastructure.createDateTimeTeller();
      infrastructure.createSystemTimeTeller();

      infrastructure.deployDateTimeTeller();
      infrastructure.deploySystemTimeTeller();
   }

   @Override
   protected void tearDown() throws Exception
   {
      super.tearDown();
      
      infrastructure.unregisterManagedObjects();
      
      infrastructure.undeployDateTimeTeller();
      infrastructure.undeploySystemTimeTeller();
      
      infrastructure.undeployServers();
   }

   public void testTransactionalSuccessfulCallsRoundRobin() throws Exception
   {
      /* same proxies used in simulated transactions */
      transactionalSuccessfulCalls(TraceRoundRobin.class, false);
      /* different proxies in simulated transactions */
      transactionalSuccessfulCalls(TraceRoundRobin.class, true);
   }
   
   public void testTransactionalSuccessfulCallsFirstAvailable() throws Exception
   {
      /* same proxies used in simulated transactions */
      transactionalSuccessfulCalls(TraceFirstAvailable.class, false);
      /* same proxies used in simulated transactions */
      transactionalSuccessfulCalls(TraceFirstAvailable.class, true);            
   }
   
   public void testTransactionalSuccessfulCallsFirstAvailableIndenticalAllProxies() throws Exception
   {
      /* same proxies used in simulated transactions */
      transactionalSuccessfulCalls(TraceFirstAvailableIdenticalAllProxies.class, false);
      /* same proxies used in simulated transactions */
      transactionalSuccessfulCalls(TraceFirstAvailableIdenticalAllProxies.class, true);                  
   }
   
   public void testTransactionalSuccessfulCallsRandomRobin() throws Exception
   {
      /* same proxies used in simulated transactions */
      transactionalSuccessfulCalls(TraceRandomRobin.class, false);
      /* same proxies used in simulated transactions */
      transactionalSuccessfulCalls(TraceRandomRobin.class, true);                  
   }
   
   protected void transactionalSuccessfulCalls(Class<? extends LoadBalancePolicy> policyClass, boolean newProxiesInBetweenTransactions)
   {
      log.debug("transactional successfull calls [policy=" + policyClass + ",newProxiesInBetweenTransactions=" + newProxiesInBetweenTransactions + "]");
      
      try
      {
         UID uid;
         
         createNewProxies(0, policyClass, true);

         /* Simulate client user transaction */
         uid = new UID();         
         transactionalMockUtils.getTpcf().setUid(uid);
         performCalls(3, null, policyClass);
         
         if (newProxiesInBetweenTransactions)
         {
            createNewProxies(0, policyClass, false);
         }
         
         /* Simulate transaction interceptor */
         uid = new UID();
         Transaction tx = new MockTransaction();
         transactionalMockUtils.getTpcf().setUid(uid);
         transactionalMockUtils.getTpci().setTransaction(tx);
         performCalls(3, tx, policyClass);                  
      }
      catch(Exception e)
      {
         /* catching to log the error properly (JUnit in eclipse does not show 
          * correctly exceptions from invokers) and fail */
         log.error("error", e);
         fail();
      }
   }

   protected void performCalls(int numberPairCalls, 
         Transaction tx, Class<? extends LoadBalancePolicy> policyClass) throws Exception
   {
      Invocation inv;
      
      for (int i = 0; i < numberPairCalls; i++)
      {
         /* create invocation to date time teller */
         inv = infrastructure.createDateTimeTellerInvocation(tx, null);
         /* invoke on proxy passing the invocation */
         log.debug(timeTellerProxy.invoke(inv));
         /* assert post conditions after invocation */
         prevChosenTargetDateTimeTeller = assertSuccessfulPostConditions(inv, prevChosenTargetDateTimeTeller, tx, policyClass);
         
         /* create invocation to system time teller */
         inv = infrastructure.createSystemTimeTellerInvocation(tx, null);
         /* invoke on proxy passing the invocation */
         log.debug(systemTimeProxy.invoke(inv));
         /* assert post conditions after invocation */
         prevChosenTargetSystemTimeTeller = assertSuccessfulPostConditions(inv, prevChosenTargetSystemTimeTeller, tx, policyClass);
      }
   }
   
   protected Object assertSuccessfulPostConditions(Invocation inv, Object prevChosenTarget, Transaction tx, Class<? extends LoadBalancePolicy> policyClass)
   {
      assertEquals(0, inv.getAsIsValue("FAILOVER_COUNTER"));
      Object chosenTarget = inv.getTransientValue(invokerHaFactory.getChosenTargetKey());
      assertNotNull(chosenTarget);
      /* if tx was null, invocation's tx should be null after invocation. */
      assertEquals(tx, inv.getTransaction());
      if (transactionalMockUtils.getTpcf().getUid() != null)
      {
         /* check tx failover authorisations */
         assertTrue("transaction should have reached the server", invokerHaFactory.getTxFailoverAuthorizationsMap().containsKey(transactionalMockUtils.getTpcf().getUid()));         
      }
      /* check chosen target with previously chosen target, if there's any */
      return assertChosenTarget(policyClass, chosenTarget, prevChosenTarget);
   }
   
   protected void createNewProxies(int serverIndex, Class<? extends LoadBalancePolicy> policyClass, boolean isVery1st) throws Exception
   {
      /* Create a proxy instances retrieved from the first server */
      timeTellerProxy = infrastructure.createDateTimeTellerProxy(serverIndex, policyClass);
      systemTimeProxy = infrastructure.createSystemTimeTellerProxy(serverIndex, policyClass);
      
      /* Initialise previous chosen targets. If not new proxies elected 
       * between transactions, this allows to carry on checking chosen 
       * targets in between transactions. */ 
      if (!isVery1st && policyClass.equals(TraceFirstAvailableIdenticalAllProxies.class))
      {
         /* In the particular case of first availble indentical proxies, if we're 
         * not creating the proxies for the first time, do not initialise the 
         * proxies because we need them to check them with next chosen ones. */
      }
      else
      {
         prevChosenTargetDateTimeTeller = null;
         prevChosenTargetSystemTimeTeller = null;                  
      }
   }
   
   protected Object assertChosenTarget(Class<? extends LoadBalancePolicy> policyClass, Object chosenTarget, Object prevChosenTarget)
   {
      if (policyClass.equals(TraceRoundRobin.class))
      {
         prevChosenTarget = checkRoundRobin(chosenTarget, prevChosenTarget);
      }
      else if (policyClass.equals(TraceFirstAvailable.class))
      {
         prevChosenTarget = checkFirstAvailable(chosenTarget, prevChosenTarget);
      }
      else if (policyClass.equals(TraceFirstAvailableIdenticalAllProxies.class))
      {
         prevChosenTarget = checkFirstAvailableIndenticalAllProxies(chosenTarget, prevChosenTarget);
      }
      
      return prevChosenTarget;
   }
   
   protected Object checkRoundRobin(Object chosenTarget, Object prevChosenTarget)
   {
      if (prevChosenTarget != null)
      {
         /* In round robin, previous chosen target must be different to the 
          * current one, unless there's only one node in the cluster, but we're 
          * not testing that here. */
         assertNotSame(prevChosenTarget, chosenTarget);
      }      
      
      return chosenTarget;
   }
   
   protected Object checkFirstAvailable(Object chosenTarget, Object prevChosenTarget)
   {
      if (prevChosenTarget != null)
      {
         /* In first available robin, previous chosen target must be the same to the 
          * current one, unless there's only one node in the cluster, but we're 
          * not testing that here. */
         assertEquals(prevChosenTarget, chosenTarget);
      }
      
      return chosenTarget;      
   }
   
   protected Object checkFirstAvailableIndenticalAllProxies(Object chosenTarget, Object prevChosenTarget)
   {
      return checkFirstAvailable(chosenTarget, prevChosenTarget);
   }   
}
