/*
* JBoss, Home of Professional Open Source
* Copyright 2005, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.ant.taskdefs.server;

import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.Project;
import org.apache.tools.ant.Task;

import org.jboss.jbossas.servermanager.Server;
import org.jboss.jbossas.servermanager.ServerManager;

/**
 * A ConfigManagerTask.  Delegates to the Server manager.
 * 
 * @author <a href="ryan.campbell@jboss.com">Ryan Campbell</a>
 * @version $Revision: 79974 $
 */
public class ConfigManagerTask extends Task
{
   /** The key for the project reference **/
   protected static final String MANAGER_REF = "serverManager";

   static ServerManager manager = null;

   /**
    * Create a new ConfigManagerTask.
    * 
    * 
    */
   public ConfigManagerTask()
   {
      if (ConfigManagerTask.manager == null)
      {
         // set TCL and save the original TCL to restore it later
         Thread curThread = Thread.currentThread();
         ClassLoader orgCl = curThread.getContextClassLoader();
         curThread.setContextClassLoader(getClassLoader());

        /*
         Create ServerManager with the appropriate TCL, otherwise
         the clean-up thread wouldn't be able to shutdown servers
         cleanly. Then reset the context classloader.
        */
         try
         {
            ConfigManagerTask.manager = new ServerManager();
         }
         finally
         {
            curThread.setContextClassLoader(orgCl);
         }
      }
   }
   
   /**
    * Create a server manager using the specified configuration.
    */
   public void execute() throws BuildException
   {
      if (getProject().getReference(ConfigManagerTask.MANAGER_REF)==null)
      {
         getProject().addReference(MANAGER_REF, ConfigManagerTask.manager);
      }
   }

   /**
    * Get Project's server manager.
    *
    * @return the server manager
    */
   protected static ServerManager getServerManager(Project project)
   {
      ServerManager manager = (ServerManager) project.getReference(MANAGER_REF);
      if ( manager == null ) throw new BuildException("Can't find reference to server manager. Consider context you execute server tasks.");

      return manager;
   }

   /**
    * Get Server Manager's ClassLoader.
    *
    * @return the class loader aware of server manager dependencies
    */
   protected static ClassLoader getClassLoader()
   {
      return ServerManager.class.getClassLoader();
   }

   /**
    * Add a server.
    * 
    * @param server the server
    */
   public void addServer(Server server)
   {
      ConfigManagerTask.manager.addServer(server);
   }

   /** 
    * JAVA_HOME to start jboss with.
    * @param javaHome
    */
   public void setJavaHome(String javaHome)
   {
      ConfigManagerTask.manager.setJavaHome(javaHome);
   }

   /**
    * JBoss dist to start.
    * @param jbossHome
    */
   public void setJbossHome(String jbossHome)
   {
      ConfigManagerTask.manager.setJbossHome(jbossHome);
   }

   /**
    * JVM command to use default is "java"
    * @param jvm
    */
   public void setJvm(String jvm)
   {
      ConfigManagerTask.manager.setJvm(jvm);
   }
   
   /**
    * The UDP group to pass to org.jboss.Main using
    * the -u option.
    * 
    * @param udpGroup the udp group
    */
   public void setUdpGroup(String udpGroup)
   {
      ConfigManagerTask.manager.setUdpGroup(udpGroup);
   }
}
