/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2012, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.test.security.test.auth;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.security.auth.Subject;
import javax.security.auth.callback.Callback;
import javax.security.auth.callback.CallbackHandler;
import javax.security.auth.callback.UnsupportedCallbackException;
import javax.security.auth.spi.LoginModule;

import junit.framework.TestCase;

import org.apache.log4j.AppenderSkeleton;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.apache.log4j.spi.LoggingEvent;
import org.jboss.security.ClientLoginModule;
import org.jboss.security.SimplePrincipal;
import org.jboss.security.auth.spi.BaseCertLoginModule;
import org.jboss.security.auth.spi.DatabaseServerLoginModule;
import org.jboss.security.auth.spi.IdentityLoginModule;
import org.jboss.security.auth.spi.LdapExtLoginModule;
import org.jboss.security.auth.spi.LdapLoginModule;
import org.jboss.security.auth.spi.RunAsLoginModule;
import org.jboss.security.auth.spi.UsersRolesLoginModule;
import org.jboss.security.negotiation.AdvancedADLoginModule;
import org.jboss.security.negotiation.AdvancedLdapLoginModule;
import org.jboss.security.negotiation.spnego.SPNEGOLoginModule;

/**
 * A LogWrongLoginModuleOptionsTest for JBPAPP-7794.
 * 
 * @author <a href="olukas@redhat.com">Ondrej Lukas</a>
 * @author Josef Cacek
 */
public class LogWrongLoginModuleOptionsTest extends TestCase
{

   private static final Set<String> NUMBER_OPTIONS = new HashSet<String>();

   private static final String[] ADVANCED_LDAP_LM_OPTIONS =
   {"bindAuthentication", "jaasSecurityDomain", "bindDN", "bindCredential", "baseCtxDN", "baseFilter",
         "searchTimeLimit", "allowEmptyPassword", "rolesCtxDN", "roleFilter", "roleAttributeID", "roleAttributeIsDN",
         "roleNameAttributeID", "recurseRoles", "searchScope", "searchTimeLimit", "java.naming.provider.url"};

   private LogWarnAppender appender = new LogWarnAppender();

   Logger logger;

   static
   {
      NUMBER_OPTIONS.add("searchTimeLimit");
   }

   /**
    * Set up method
    */
   @Override
   public void setUp()
   {
      logger = Logger.getRootLogger();
      logger.addAppender(appender);
   }

   /**
    * Tear down method
    */
   @Override
   public void tearDown()
   {
      logger.removeAppender(appender);
   }

   /**
    * Test that LdapLoginModule logs invalid property name as WARN and doesn't log correct property name as invalid
    */
   public void testLdapLoginModuleNotWrongWarnLogging()
   {
      LdapLoginModule lm = new LdapLoginModule();
      String[] strOptions =
      {"java.naming.factory.initial", "java.naming.provider.url", "java.naming.security.authentication",
            "java.naming.security.protocol", "java.naming.security.principal", "java.naming.security.credentials",
            "principalDNPrefix", "principalDNSuffix", "rolesCtxDN", "userRolesCtxDNAttributeName", "roleAttributeID",
            "roleAttributeIsDN", "roleNameAttributeID", "uidAttributeID", "matchOnUserDN", "unauthenticatedIdentity",
            "allowEmptyPasswords", "password-stacking", "hashAlgorithm", "hashEncoding", "hashCharset",
            "hashUserPassword", "hashStorePassword", "unauthenticatedIdentity"};
      createOptions(lm, strOptions);
   }

   /**
    * Test that LdapExtLoginModule logs invalid property name as WARN and doesn't log correct property name as invalid
    */
   public void testLdapExtLoginModuleNotWrongWarnLogging()
   {
      LdapExtLoginModule lm = new LdapExtLoginModule();
      String[] strOptions =
      {"baseCtxDN", "bindDN", "bindCredential", "jaasSecurityDomain", "baseFilter", "rolesCtxDN", "roleFilter",
            "roleAttributeIsDN", "roleAttributeID", "roleNameAttributeID", "roleRecursion", "searchTimeLimit",
            "searchScope", "allowEmptyPasswords", "defaultRole", "parseRoleNameFromDN", "parseUsername",
            "usernameBeginString", "usernameEndString", "password-stacking", "java.naming.factory.initial",
            "java.naming.security.protocol", "java.naming.provider.url", "java.naming.security.authentication",
            "java.naming.referral", "hashAlgorithm", "hashEncoding", "hashCharset", "hashUserPassword",
            "hashStorePassword", "unauthenticatedIdentity"};
      createOptions(lm, strOptions);
   }

   /**
    * Test that UsersRolesLoginModule logs invalid property name as WARN and doesn't log correct property name as invalid
    */
   public void testUsersRolesLoginModuleNotWrongWarnLogging()
   {
      UsersRolesLoginModule lm = new UsersRolesLoginModule();
      String[] strOptions =
      {"usersProperties", "rolesProperties", "hashAlgorithm", "hashEncoding", "hashCharset", "hashUserPassword",
            "hashStorePassword", "unauthenticatedIdentity"};
      createOptions(lm, strOptions);
   }

   /**
    * Test that DatabaseServerLoginModule logs invalid property name as WARN and doesn't log correct property name as invalid
    */
   public void testDatabaseServerLoginModuleNotWrongWarnLogging()
   {
      DatabaseServerLoginModule lm = new DatabaseServerLoginModule();
      String[] strOptions =
      {"dsJndiName", "principalsQuery", "rolesQuery", "ignorePasswordCase", "password-stacking", "hashAlgorithm",
            "hashEncoding", "hashCharset", "hashUserPassword", "hashStorePassword", "unauthenticatedIdentity"};
      createOptions(lm, strOptions);
   }

   /**
    * Test that BaseCertLoginModule logs invalid property name as WARN and doesn't log correct property name as invalid
    */
   public void testBaseCertLoginModuleNotWrongWarnLogging()
   {
      BaseCertLoginModule lm = new BaseCertLoginModule();
      String[] strOptions =
      {"securityDomain", "password-stacking"};
      createOptions(lm, strOptions);
   }

   /**
    * Test that IdentityLoginModule logs invalid property name as WARN and doesn't log correct property name as invalid
    */
   public void testIdentityLoginModuleNotWrongWarnLogging()
   {
      IdentityLoginModule lm = new IdentityLoginModule();
      String[] strOptions =
      {"principal", "roles", "password-stacking"};
      createOptions(lm, strOptions);
   }

   /**
    * Test that RunAsLoginModule logs invalid property name as WARN and doesn't log correct property name as invalid
    */
   public void testRunAsLoginModuleNotWrongWarnLogging()
   {
      RunAsLoginModule lm = new RunAsLoginModule();
      String[] strOptions =
      {"roleName"};
      createOptions(lm, strOptions);
   }

   /**
    * Test that ClientLoginModule logs invalid property name as WARN and doesn't log correct property name as invalid
    */
   public void testClientLoginModuleNotWrongWarnLogging()
   {
      ClientLoginModule lm = new ClientLoginModule();
      String[] strOptions =
      {"multi-threaded", "password-stacking", "restore-login-identity"};
      createOptions(lm, strOptions);
   }

   /**
    * Test that RoleMappingLoginModule logs invalid property name as WARN and doesn't log correct property name as invalid
    */
   public void testRoleMappingLoginModuleNotWrongWarnLogging()
   {
      org.jboss.security.auth.spi.RoleMappingLoginModule lm = new org.jboss.security.auth.spi.RoleMappingLoginModule();
      String[] strOptions =
      {"rolesProperties", "replaceRole"};
      createOptions(lm, strOptions);
   }

   /**
    * Test that SPNEGOLoginModule logs invalid property name as WARN and doesn't log correct property name as invalid
    */
   public void testSPNEGOLoginModuleNotWrongWarnLogging()
   {
      SPNEGOLoginModule lm = new SPNEGOLoginModule();
      String[] strOptions =
      {"password-stacking", "serverSecurityDomain"};
      createOptions(lm, strOptions);
   }

   /**
    * Test that AdvancedADLoginModule logs invalid property name as WARN and doesn't log correct property name as invalid
    */
   public void testAdvancedADLoginModuleNotWrongWarnLogging()
   {
      //extends AdvancedLdapLoginModule
      AdvancedADLoginModule lm = new AdvancedADLoginModule();
      createOptions(lm, ADVANCED_LDAP_LM_OPTIONS, new String[]
      {"primaryGroupID", "objectSid"});
   }

   /**
    * Test that RoleMappingLoginModule logs invalid property name as WARN and doesn't log correct property name as invalid
    */
   public void testAdvancedLdapLoginModuleNotWrongWarnLogging()
   {
      AdvancedLdapLoginModule lm = new AdvancedLdapLoginModule();
      createOptions(lm, ADVANCED_LDAP_LM_OPTIONS);
   }

   /**
    * Test that Login Modules log invalid property name as WARN and don't log correct property name as invalid
    */
   private void createOptions(LoginModule lm, String[] options)
   {
      createOptions(lm, options, null);
   }

   private void createOptions(LoginModule lm, String[] options, String[] invalidOptions)
   {
      Set<String> invalidOptionSet = new HashSet<String>();
      invalidOptionSet.add("wrongPropertyName");
      if (invalidOptions != null)
      {
         invalidOptionSet.addAll(Arrays.asList(invalidOptions));
      }

      Map<String, String> lmOptionMap = new HashMap<String, String>();
      for (String option : options)
      {
         lmOptionMap.put(option, NUMBER_OPTIONS.contains(option) ? "500" : "xxx");
      }
      for (String option : invalidOptionSet)
      {
         lmOptionMap.put(option, NUMBER_OPTIONS.contains(option) ? "500" : "xxx");
      }
      lmOptionMap.put("principalClass", SimplePrincipal.class.getName());
      lm.initialize(new Subject(), new MyCallbackHandler(), new HashMap<String, String>(), lmOptionMap);
      boolean wrongProperty = false;
      List<String> list = new ArrayList<String>();
      Iterator<LoggingEvent> it = appender.getLogs().iterator();
      while (it.hasNext())
      {
         LoggingEvent e = it.next();
         for (int i = 0; i < options.length; i++)
         {
            if (e.getMessage().toString().contains(options[i]))
            {
               list.add(options[i]);
            }
         }
         Iterator<String> invalidIterator = invalidOptionSet.iterator();
         while (invalidIterator.hasNext())
         {
            final String option = invalidIterator.next();
            if (String.valueOf(e.getMessage()).contains(option))
            {
               invalidIterator.remove();
            }
         }
      }
      assertTrue("Correct options were reported as invalid - " + list, list.isEmpty());
      assertTrue("Invalid property names weren't reported - " + invalidOptionSet, invalidOptionSet.isEmpty());

   }

   /**
    * A LogWarnAppender.
    * 
    * @author <a href="olukas@redhat.com">Ondrej Lukas</a>
    */
   private class LogWarnAppender extends AppenderSkeleton
   {

      private List<LoggingEvent> logs = new ArrayList<LoggingEvent>();

      @Override
      public void close()
      {
      }

      @Override
      public boolean requiresLayout()
      {
         return false;
      }

      @Override
      protected void append(LoggingEvent event)
      {
         if (event.getLevel().equals(Level.WARN))
         {
            logs.add(event);
         }
      }

      public List<LoggingEvent> getLogs()
      {
         return logs;
      }

   }

   private class MyCallbackHandler implements CallbackHandler
   {

      @Override
      public void handle(Callback[] callbacks) throws IOException, UnsupportedCallbackException
      {
      }

   }
}