/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2015, Red Hat, Inc., and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.wildfly.clustering.web.infinispan.session.fine;

import java.io.IOException;
import java.util.EnumSet;
import java.util.List;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Map;
import java.util.TreeMap;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.BiConsumer;
import java.util.function.Function;

import org.infinispan.Cache;
import org.infinispan.context.Flag;
import org.wildfly.clustering.ee.Immutability;
import org.wildfly.clustering.ee.Mutator;
import org.wildfly.clustering.ee.MutatorFactory;
import org.wildfly.clustering.ee.cache.CacheProperties;
import org.wildfly.clustering.ee.cache.function.MapComputeFunction;
import org.wildfly.clustering.ee.infinispan.CacheComputeMutatorFactory;
import org.wildfly.clustering.infinispan.spi.PredicateKeyFilter;
import org.wildfly.clustering.infinispan.spi.listener.PostActivateListener;
import org.wildfly.clustering.infinispan.spi.listener.PrePassivateListener;
import org.wildfly.clustering.marshalling.spi.Marshaller;
import org.wildfly.clustering.web.cache.session.CompositeImmutableSession;
import org.wildfly.clustering.web.cache.session.ImmutableSessionAttributeActivationNotifier;
import org.wildfly.clustering.web.cache.session.SessionAttributeActivationNotifier;
import org.wildfly.clustering.web.cache.session.SessionAttributes;
import org.wildfly.clustering.web.cache.session.SessionAttributesFactory;
import org.wildfly.clustering.web.cache.session.fine.FineImmutableSessionAttributes;
import org.wildfly.clustering.web.cache.session.fine.FineSessionAttributes;
import org.wildfly.clustering.web.infinispan.logging.InfinispanWebLogger;
import org.wildfly.clustering.web.infinispan.session.InfinispanSessionAttributesFactoryConfiguration;
import org.wildfly.clustering.web.infinispan.session.SessionCreationMetaDataKey;
import org.wildfly.clustering.web.infinispan.session.SessionCreationMetaDataKeyFilter;
import org.wildfly.clustering.web.session.HttpSessionActivationListenerProvider;
import org.wildfly.clustering.web.session.ImmutableSessionAttributes;
import org.wildfly.clustering.web.session.ImmutableSessionMetaData;

/**
 * {@link SessionAttributesFactory} for fine granularity sessions.
 * A given session's attributes are mapped to N+1 co-located cache entries, where N is the number of session attributes.
 * A separate cache entry stores the activate attribute names for the session.
 * @author Paul Ferraro
 */
public class FineSessionAttributesFactory<S, C, L, V> implements SessionAttributesFactory<C, Map.Entry<Map<String, UUID>, Map<UUID, Object>>> {

    private final Cache<SessionAttributeNamesKey, Map<String, UUID>> namesCache;
    private final Cache<SessionAttributeKey, V> attributeCache;
    private final MutatorFactory<SessionAttributeNamesKey, Map.Entry<Map<String, UUID>, Map<UUID, Object>>> mutatorFactory;
    private final Marshaller<Object, V> marshaller;
    private final Immutability immutability;
    private final CacheProperties properties;
    private final HttpSessionActivationListenerProvider<S, C, L> provider;
    private final Function<String, SessionAttributeActivationNotifier> notifierFactory;
    private final Object evictListener;
    private final Object evictAttributesListener;
    private final Object prePassivateListener;
    private final Object postActivateListener;

    public FineSessionAttributesFactory(InfinispanSessionAttributesFactoryConfiguration<S, C, L, Object, V> configuration) {
        this.properties = configuration.getCacheProperties();
        this.namesCache = configuration.getCache();
        this.attributeCache = configuration.getCache();
        this.marshaller = configuration.getMarshaller();
        this.immutability = configuration.getImmutability();
        this.provider = configuration.getHttpSessionActivationListenerProvider();
        this.notifierFactory = configuration.getActivationNotifierFactory();
        this.evictListener = new PrePassivateListener<>(this::cascadeEvict, configuration.getExecutor());
        this.evictAttributesListener = new PrePassivateListener<>(this::cascadeEvictAttributes, configuration.getExecutor());
        this.prePassivateListener = !this.properties.isPersistent() ? new PrePassivateListener<>(this::prePassivate, configuration.getExecutor()) : null;
        this.postActivateListener = !this.properties.isPersistent() ? new PostActivateListener<>(this::postActivate, configuration.getExecutor()) : null;
        if (this.prePassivateListener != null) {
            this.attributeCache.addListener(this.prePassivateListener, new PredicateKeyFilter<>(SessionAttributeKeyFilter.INSTANCE), null);
        }
        if (this.postActivateListener != null) {
            this.attributeCache.addListener(this.postActivateListener, new PredicateKeyFilter<>(SessionAttributeKeyFilter.INSTANCE), null);
        }
        this.namesCache.addListener(this.evictAttributesListener, new PredicateKeyFilter<>(SessionAttributeNamesKeyFilter.INSTANCE), null);
        this.namesCache.addListener(this.evictListener, new PredicateKeyFilter<>(SessionCreationMetaDataKeyFilter.INSTANCE), null);
        MutatorFactory<SessionAttributeNamesKey, Map<String, UUID>> namesMutatorFactory = new CacheComputeMutatorFactory<>(this.namesCache, MapComputeFunction::new);
        this.mutatorFactory = new MutatorFactory<SessionAttributeNamesKey, Map.Entry<Map<String, UUID>, Map<UUID, Object>>>() {
            @Override
            public Mutator createMutator(SessionAttributeNamesKey namesKey, Map.Entry<Map<String, UUID>, Map<UUID, Object>> entry) {
                String id = namesKey.getId();
                Map<String, UUID> names = entry.getKey();
                Map<UUID, Object> attributes = entry.getValue();
                List<Map.Entry<SessionAttributeKey, V>> updates = new ArrayList<>(attributes.size());
                List<SessionAttributeKey> removes = new ArrayList<>(attributes.size());
                Cache<SessionAttributeKey, V> cache = configuration.<SessionAttributeKey, V>getCache().getAdvancedCache().withFlags(Flag.IGNORE_RETURN_VALUES);
                Marshaller<Object, V> marshaller = configuration.getMarshaller();
                try {
                    for (Map.Entry<UUID, Object> attribute : attributes.entrySet()) {
                        SessionAttributeKey key = new SessionAttributeKey(id, attribute.getKey());
                        Object value = attribute.getValue();
                        if (value != null) {
                            updates.add(new AbstractMap.SimpleImmutableEntry<>(key, marshaller.write(value)));
                        } else {
                            removes.add(key);
                        }
                    }
                } catch (IOException e) {
                    throw new IllegalStateException(e);
                }
                return new Mutator() {
                    @Override
                    public void mutate() {
                        for (Map.Entry<SessionAttributeKey, V> entry : updates) {
                            cache.put(entry.getKey(), entry.getValue());
                        }
                        if (!names.isEmpty()) {
                            namesMutatorFactory.createMutator(namesKey, names).mutate();
                        }
                        for (SessionAttributeKey key : removes) {
                            cache.remove(key);
                        }
                    }
                };
            }
        };
    }

    @Override
    public void close() {
        this.namesCache.removeListener(this.evictListener);
        this.namesCache.removeListener(this.evictAttributesListener);
        if (this.prePassivateListener != null) {
            this.attributeCache.removeListener(this.prePassivateListener);
        }
        if (this.postActivateListener != null) {
            this.attributeCache.removeListener(this.postActivateListener);
        }
    }

    @Override
    public Map.Entry<Map<String, UUID>, Map<UUID, Object>> createValue(String id, Void context) {
        return new AbstractMap.SimpleImmutableEntry<>(new ConcurrentHashMap<>(), new ConcurrentHashMap<>());
    }

    @Override
    public Map.Entry<Map<String, UUID>, Map<UUID, Object>> findValue(String id) {
        return this.getValue(id, true);
    }

    @Override
    public Map.Entry<Map<String, UUID>, Map<UUID, Object>> tryValue(String id) {
        return this.getValue(id, false);
    }

    private Map.Entry<Map<String, UUID>, Map<UUID, Object>> getValue(String id, boolean purgeIfInvalid) {
        Map<String, UUID> names = this.namesCache.get(new SessionAttributeNamesKey(id));
        if (names == null) {
            return this.createValue(id, null);
        }
        // Validate all attributes
        Map<SessionAttributeKey, String> keys = new TreeMap<>();
        for (Map.Entry<String, UUID> entry : names.entrySet()) {
            keys.put(new SessionAttributeKey(id, entry.getValue()), entry.getKey());
        }
        Map<SessionAttributeKey, V> values = this.attributeCache.getAdvancedCache().getAll(keys.keySet());
        // Validate attributes
        Map<UUID, Object> attributes = new ConcurrentHashMap<>();
        for (Map.Entry<SessionAttributeKey, String> entry : keys.entrySet()) {
            SessionAttributeKey key = entry.getKey();
            V value = values.get(key);
            if (value != null) {
                try {
                    attributes.put(key.getAttributeId(), this.marshaller.read(value));
                    continue;
                } catch (IOException e) {
                    InfinispanWebLogger.ROOT_LOGGER.failedToActivateSessionAttribute(e, id, entry.getValue());
                }
            } else {
                InfinispanWebLogger.ROOT_LOGGER.missingSessionAttributeCacheEntry(id, entry.getValue());
            }
            if (purgeIfInvalid) {
                this.purge(id);
            }
            return null;
        }
        return new AbstractMap.SimpleImmutableEntry<>(new ConcurrentHashMap<>(names), attributes);
    }

    @Override
    public boolean remove(String id) {
        return this.delete(id);
    }

    @Override
    public boolean purge(String id) {
        return this.delete(id, Flag.SKIP_LISTENER_NOTIFICATION);
    }

    private boolean delete(String id, Flag... flags) {
        Map<String, UUID> names = this.namesCache.getAdvancedCache().withFlags(EnumSet.of(Flag.FORCE_SYNCHRONOUS, flags)).remove(new SessionAttributeNamesKey(id));
        if (names != null) {
            for (UUID attributeId : names.values()) {
                this.attributeCache.getAdvancedCache().withFlags(EnumSet.of(Flag.IGNORE_RETURN_VALUES, flags)).remove(new SessionAttributeKey(id, attributeId));
            }
        }
        return true;
    }

    @Override
    public SessionAttributes createSessionAttributes(String id, Map.Entry<Map<String, UUID>, Map<UUID, Object>> entry, ImmutableSessionMetaData metaData, C context) {
        SessionAttributeActivationNotifier notifier = this.properties.isPersistent() ? new ImmutableSessionAttributeActivationNotifier<>(this.provider, new CompositeImmutableSession(id, metaData, this.createImmutableSessionAttributes(id, entry)), context) : null;
        return new FineSessionAttributes<>(new SessionAttributeNamesKey(id), entry.getKey(), entry.getValue(), this.mutatorFactory, this.marshaller, this.immutability, this.properties, notifier);
    }

    @Override
    public ImmutableSessionAttributes createImmutableSessionAttributes(String id, Map.Entry<Map<String, UUID>, Map<UUID, Object>> entry) {
        return new FineImmutableSessionAttributes(entry.getKey(), entry.getValue());
    }

    private void cascadeEvict(SessionCreationMetaDataKey key, Object value) {
        this.namesCache.evict(new SessionAttributeNamesKey(key.getId()));
    }

    private void cascadeEvictAttributes(SessionAttributeNamesKey key, Map<String, UUID> value) {
        String sessionId = key.getId();
        for (UUID attributeId : value.values()) {
            this.attributeCache.evict(new SessionAttributeKey(sessionId, attributeId));
        }
    }

    private void prePassivate(SessionAttributeKey key, V value) {
        this.notify(SessionAttributeActivationNotifier.PRE_PASSIVATE, key, value);
    }

    private void postActivate(SessionAttributeKey key, V value) {
        this.notify(SessionAttributeActivationNotifier.POST_ACTIVATE, key, value);
    }

    private void notify(BiConsumer<SessionAttributeActivationNotifier, Object> notification, SessionAttributeKey key, V value) {
        String sessionId = key.getId();
        try (SessionAttributeActivationNotifier notifier = this.notifierFactory.apply(key.getId())) {
            notification.accept(notifier, this.marshaller.read(value));
        } catch (IOException e) {
            InfinispanWebLogger.ROOT_LOGGER.failedToActivateSessionAttribute(e, sessionId, key.getAttributeId().toString());
        }
    }
}
