/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2013, Red Hat, Inc., and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.wildfly.clustering.web.cache.session.fine;

import java.io.NotSerializableException;
import java.util.AbstractMap;
import java.util.Map;
import java.util.TreeMap;
import java.util.UUID;
import java.util.concurrent.ThreadLocalRandom;

import org.wildfly.clustering.ee.Immutability;
import org.wildfly.clustering.ee.MutatorFactory;
import org.wildfly.clustering.ee.cache.CacheProperties;
import org.wildfly.clustering.marshalling.spi.Marshallability;
import org.wildfly.clustering.web.cache.session.SessionAttributeActivationNotifier;
import org.wildfly.clustering.web.cache.session.SessionAttributes;

/**
 * Exposes session attributes for fine granularity sessions.
 * @author Paul Ferraro
 */
public class FineSessionAttributes<K> extends FineImmutableSessionAttributes implements SessionAttributes {

    private final MutatorFactory<K, Map.Entry<Map<String, UUID>, Map<UUID, Object>>> mutatorFactory;
    private final K key;
    private final Map<String, UUID> names;
    private final Map<UUID, Object> attributes;
    private final Marshallability marshaller;
    private final Immutability immutability;
    private final CacheProperties properties;
    private final SessionAttributeActivationNotifier notifier;
    private final Map<String, UUID> nameUpdates = new TreeMap<>();
    private final Map<UUID, Object> updates = new TreeMap<>();

    public FineSessionAttributes(K key, Map<String, UUID> names, Map<UUID, Object> attributes, MutatorFactory<K, Map.Entry<Map<String, UUID>, Map<UUID, Object>>> mutatorFactory, Marshallability marshaller, Immutability immutability, CacheProperties properties, SessionAttributeActivationNotifier notifier) {
        super(names, attributes);
        this.key = key;
        this.names = names;
        this.attributes = attributes;
        this.mutatorFactory = mutatorFactory;
        this.marshaller = marshaller;
        this.immutability = immutability;
        this.properties = properties;
        this.notifier = notifier;

        if (this.notifier != null) {
            for (Object value : this.attributes.values()) {
                this.notifier.postActivate(value);
            }
        }
    }

    @Override
    public Object getAttribute(String name) {
        UUID id = this.names.get(name);
        Object value = (id != null) ? this.attributes.get(id) : null;

        if (value != null) {
            // If the object is mutable, we need to mutate this value on close
            if (!this.immutability.test(value)) {
                synchronized (this.updates) {
                    this.updates.put(id, value);
                }
            }
        }

        return value;
    }

    @Override
    public Object removeAttribute(String name) {
        UUID id = this.names.remove(name);

        if (id != null) {
            synchronized (this.nameUpdates) {
                this.nameUpdates.put(name, null);
            }
        }

        Object result = (id != null) ? this.attributes.remove(id) : null;

        if (result != null) {
            synchronized (this.updates) {
                this.updates.put(id, null);
            }
        }

        return result;
    }

    @Override
    public Object setAttribute(String name, Object value) {
        if (value == null) {
            return this.removeAttribute(name);
        }

        if (this.properties.isMarshalling() && !this.marshaller.isMarshallable(value)) {
            throw new IllegalArgumentException(new NotSerializableException(value.getClass().getName()));
        }

        UUID id = createUUID();
        UUID existing = this.names.putIfAbsent(name, id);
        if (existing == null) {
            synchronized (this.nameUpdates) {
                this.nameUpdates.put(name, id);
            }
        } else {
            id = existing;
        }

        Object result = this.attributes.put(id, value);

        // Always trigger attribute update, even if called with an existing reference
        synchronized (this.updates) {
            this.updates.put(id, value);
        }

        return result;
    }

    @Override
    public void close() {
        if (this.notifier != null) {
            for (Object value : this.attributes.values()) {
                this.notifier.prePassivate(value);
            }
        }
        synchronized (this.nameUpdates) {
            synchronized (this.updates) {
                if (!this.nameUpdates.isEmpty() || !this.updates.isEmpty()) {
                    this.mutatorFactory.createMutator(this.key, new AbstractMap.SimpleImmutableEntry<>(this.nameUpdates, this.updates)).mutate();
                }
            }
        }
    }

    private static UUID createUUID() {
        byte[] data = new byte[16];
        ThreadLocalRandom.current().nextBytes(data);
        data[6] &= 0x0f; /* clear version */
        data[6] |= 0x40; /* set to version 4 */
        data[8] &= 0x3f; /* clear variant */
        data[8] |= 0x80; /* set to IETF variant */
        long msb = 0;
        long lsb = 0;
        for (int i = 0; i < 8; i++) {
           msb = (msb << 8) | (data[i] & 0xff);
        }
        for (int i = 8; i < 16; i++) {
           lsb = (lsb << 8) | (data[i] & 0xff);
        }
        return new UUID(msb, lsb);
    }
}
