/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2011, Red Hat, Inc., and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.as.test.integration.domain;

import org.jboss.as.controller.PathAddress;
import org.jboss.as.controller.operations.common.Util;
import org.jboss.as.test.integration.domain.management.util.DomainLifecycleUtil;
import org.jboss.as.test.integration.domain.management.util.WildFlyManagedConfiguration;
import org.jboss.dmr.ModelNode;
import org.jboss.logging.Logger;
import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.BeforeClass;
import org.junit.Test;

import java.io.IOException;

import static org.jboss.as.controller.descriptions.ModelDescriptionConstants.EXTENSION;
import static org.jboss.as.controller.descriptions.ModelDescriptionConstants.HOST;
import static org.jboss.as.controller.descriptions.ModelDescriptionConstants.OUTCOME;
import static org.jboss.as.controller.descriptions.ModelDescriptionConstants.PROFILE;
import static org.jboss.as.controller.descriptions.ModelDescriptionConstants.READ_RESOURCE_OPERATION;
import static org.jboss.as.controller.descriptions.ModelDescriptionConstants.SUBSYSTEM;
import static org.jboss.as.controller.descriptions.ModelDescriptionConstants.SUCCESS;

/**
 * Ensures the default domain.xml and host.xml start.
 *
 * @author Brian Stansberry (c) 2013 Red Hat Inc.
 */
public class LegacyMicroprofileSubsystemDomainTestCase extends BuildConfigurationTestBase {
    private static final Logger LOGGER = Logger.getLogger(LegacyMicroprofileSubsystemDomainTestCase.class);

    private static DomainLifecycleUtil utils;

    private static final PathAddress FULL_PROFILE = PathAddress.pathAddress(PROFILE, "full");

    private static final LegacyInfo CONFIG =
            new LegacyInfo(
                    "org.wildfly.extension.microprofile.config-smallrye",
                    "microprofile-config-smallrye");
    // Migrating health results in replacement of the subsystem and extension
    private static final LegacyInfo HEALTH =
            new LegacyInfo(
                    "org.wildfly.extension.microprofile.health-smallrye",
                    "microprofile-health-smallrye",
                    new SubsystemInfo("org.wildfly.extension.health", "health"));
    // Migrating metrics results in replacement of the subsystem and extension
    private static final LegacyInfo METRICS =
            new LegacyInfo(
                    "org.wildfly.extension.microprofile.metrics-smallrye",
                    "microprofile-metrics-smallrye",
                    new SubsystemInfo("org.wildfly.extension.metrics", "metrics"));
    // Migrating OpenTracing just results in removal of the subsystem and extension
    private static final LegacyInfo OPEN_TRACING =
            new LegacyInfo(
                    "org.wildfly.extension.microprofile.opentracing-smallrye",
                    "microprofile-opentracing-smallrye",
                    null);

    private static final LegacyInfo[] LEGACY_INFOS = new LegacyInfo[]{CONFIG, HEALTH, METRICS, OPEN_TRACING};

    @BeforeClass
    public static void startDomain() throws Exception {
        final WildFlyManagedConfiguration config = createConfiguration("domain.xml", "host.xml", LegacyMicroprofileSubsystemDomainTestCase.class.getSimpleName());
        config.setAdminOnly(true);
        utils = new DomainLifecycleUtil(config);
        utils.start();
    }

    @AfterClass
    public static void stopDomain() {
        for (LegacyInfo info : LEGACY_INFOS) {
            if (info.migration != null) {
                removeIfExistsSafe(info.migration.getSubsystemAddress());
                removeIfExistsSafe(info.migration.getExtensionAddress());
            }
            removeIfExistsSafe(info.getSubsystemAddress());
            removeIfExistsSafe(info.getExtensionAddress());
        }
        utils.stop();
    }

    @Test
    public void testMigrateLegacyConfig() throws Exception {
        testMigrateLegacySubsystem(CONFIG);
    }

    @Test
    public void testMigrateLegacyHealth() throws Exception {
        testMigrateLegacySubsystem(HEALTH);
    }

    @Test
    public void testMigrateLegacyMetrics() throws Exception {
        testMigrateLegacySubsystem(METRICS);
    }

    @Test
    public void testMigrateLegacyOpenTracing() throws Exception {
        testMigrateLegacySubsystem(OPEN_TRACING);
    }

    public void testMigrateLegacySubsystem(LegacyInfo info) throws Exception {
        assertResourceDoesNotExist(info.getExtensionAddress());
        assertResourceDoesNotExist(info.getSubsystemAddress());
        if (info.migration != null) {
            assertResourceDoesNotExist(info.migration.getExtensionAddress());
            assertResourceDoesNotExist(info.migration.getSubsystemAddress());
        }

        add(info.getExtensionAddress());
        add(info.getSubsystemAddress());

        assertResourceExists(info.getExtensionAddress());
        assertResourceExists(info.getSubsystemAddress());

        migrate(info.getSubsystemAddress());

        assertResourceExists(info.getExtensionAddress());
        assertResourceDoesNotExist(info.getSubsystemAddress());

        if (info.migration != null) {
            assertResourceExists(info.migration.getExtensionAddress());
            assertResourceExists(info.migration.getSubsystemAddress());
        }
    }


    private void assertResourceExists(PathAddress pathAddress) throws Exception {
        Assert.assertTrue(resourceExists(pathAddress));
    }

    private void assertResourceDoesNotExist(PathAddress pathAddress) throws Exception {
        Assert.assertFalse(resourceExists(pathAddress));
    }

    private static boolean resourceExists(PathAddress pathAddress) throws Exception {
        ModelNode op = Util.getEmptyOperation(READ_RESOURCE_OPERATION, pathAddress.toModelNode());

        ModelNode response = utils.getDomainClient().execute(op);
        return checkOutcome(response);
    }

    private static boolean checkOutcome(ModelNode response) {
        return response.get(OUTCOME).asString().equals(SUCCESS);
    }

    private void add(PathAddress pathAddress) throws Exception {
        ModelNode add = Util.createAddOperation(pathAddress);
        ModelNode response = utils.getDomainClient().execute(add);
        checkOutcome(response);
    }

    private static void removeIfExistsSafe(PathAddress pathAddress) {
        final boolean resourceExists;
        try {
            resourceExists = resourceExists(pathAddress);
        } catch (Exception e) {
            LOGGER.errorf(e, "Error checking if exists %s", pathAddress);
            return;
        }
        if (resourceExists) {
            ModelNode add = Util.createAddOperation(pathAddress);
            try {
                utils.getDomainClient().execute(add);
            } catch (IOException e) {
                LOGGER.errorf(e, "Could not remove %s", pathAddress);
            }
        }
    }

    private void migrate(PathAddress pathAddress) throws Exception {
        ModelNode migrate = Util.createEmptyOperation("migrate", pathAddress);
        ModelNode response = utils.getDomainClient().execute(migrate);
        checkOutcome(response);
    }

    private void reload(boolean adminOnly) throws Exception{
        ModelNode restartAdminOnly = Util.createEmptyOperation("reload", PathAddress.pathAddress(HOST, "master"));
        restartAdminOnly.get("admin-only").set(adminOnly);
        utils.executeAwaitConnectionClosed(restartAdminOnly);
        utils.connect();
        utils.awaitHostController(System.currentTimeMillis());
    }

    private static class SubsystemInfo {
        private final String extensionName;
        private final String subsystemName;

        SubsystemInfo(String extensionName, String subsystemName) {
            this.extensionName = extensionName;
            this.subsystemName = subsystemName;
        }

        PathAddress getExtensionAddress() {
            return PathAddress.pathAddress(EXTENSION, extensionName);
        }

        PathAddress getSubsystemAddress() {
            return FULL_PROFILE.append(SUBSYSTEM, subsystemName);
        }
    }

    private static class LegacyInfo extends SubsystemInfo {
        private final SubsystemInfo migration;

        LegacyInfo(String extensionName, String subsystemName) {
            this(extensionName, subsystemName, null);
        }

        LegacyInfo(String extensionName, String subsystemName, SubsystemInfo migration) {
            super(extensionName, subsystemName);
            this.migration = migration;
        }
    }
}
