/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2021 Red Hat, Inc., and individual contributors
 * as indicated by the @author tags.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.jboss.installer.common;

import org.jboss.installer.core.InstallerRuntimeException;

import javax.swing.BorderFactory;
import javax.swing.JPanel;
import javax.swing.JTextPane;
import javax.swing.KeyStroke;
import javax.swing.event.HyperlinkEvent;
import javax.swing.text.AttributeSet;
import javax.swing.text.BadLocationException;
import javax.swing.text.Element;
import javax.swing.text.TextAction;
import javax.swing.text.html.HTML;
import javax.swing.text.html.HTMLDocument;
import java.awt.Color;
import java.awt.Component;
import java.awt.Desktop;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.Rectangle;
import java.awt.event.ActionEvent;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import java.awt.event.InputEvent;
import java.awt.event.MouseEvent;
import java.io.IOException;
import java.net.URISyntaxException;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import static org.jboss.installer.screens.DefaultScreen.CONTENT_PANEL_WIDTH;
import static org.jboss.installer.screens.DefaultScreen.NO_INSET;

public class HtmlTextPanel extends JPanel {

    private static final int TEXT_PANE_HGAP = 2;

    public HtmlTextPanel(String text) {
        this.setLayout(new FlowLayout(FlowLayout.LEADING, 2, 0));
        init(text);
    }

    private void init(String text) {
        // split the text around links
        final Pattern pattern = Pattern.compile("<a href=(.*?)</a>");
        final Matcher matcher = pattern.matcher(text);

        int index = 0;
        if (matcher.find()) {
            for  (int i=0; i<matcher.groupCount(); i++) {
                final String link = matcher.group(i);
                String partText = text.substring(index, text.indexOf(link));
                index += partText.length() + link.length();
                // remove all html formatting - we'll recreate it in next step
                partText = partText.replaceAll("<[^>]*>", "");
                this.add(createTextPane(partText, false));
                this.add(createTextPane(link, true));
            }

        }
        if (index < text.length()) {
            String partText = text.substring(index);
            partText = partText.replaceAll("<[^>]*>", "");
            this.add(createTextPane(partText, false));
        }
    }

    private JTextPane createTextPane(String text, boolean isLink) {
        JTextPane textArea = new JTextPane();
        textArea.setContentType("text/html");
        textArea.setMargin(NO_INSET);
        // margin-top is added to prevent TextPane from adding empty spaces on top of the component
        textArea.setText("<html><p style=\"margin-top:0\">" + text + "</p></<html>");
        textArea.setFont(FontResources.getOpenSansRegular());
        textArea.setEditable(false);
        textArea.setBackground(null);
        if (!isLink) {
            textArea.setFocusable(false);
        } else {
            textArea.setFocusable(true);
            // add empty border with non-zero thickness so that the component is not moved when focus border is printed
            textArea.setBorder(BorderFactory.createEmptyBorder(1,1,1,1));
            textArea.addFocusListener(new FocusListener() {
                @Override
                public void focusGained(FocusEvent e) {
                    // hack to display a focus around the text
                    ((JTextPane)e.getComponent()).setBorder(BorderFactory.createLineBorder(new Color(135,206,235), 1, true));
                    e.getComponent().repaint();
                }

                @Override
                public void focusLost(FocusEvent e) {
                    // add empty border with non-zero thickness so that the component is not moved when focus border is printed
                    ((JTextPane)e.getComponent()).setBorder(BorderFactory.createEmptyBorder(1,1,1,1));
                    e.getComponent().repaint();
                }
            });


            textArea.addHyperlinkListener(e -> {
                if (e.getEventType() == HyperlinkEvent.EventType.ACTIVATED) {
                    if (Desktop.isDesktopSupported()) {
                        try {
                            Desktop.getDesktop().browse(e.getURL().toURI());
                        } catch (IOException | URISyntaxException e1) {
                            throw new InstallerRuntimeException("Unable to open the requested URL", e1);
                        }
                    }
                }
            });

            textArea.getInputMap().put(KeyStroke.getKeyStroke("ENTER"), "ClickHyperlink");
            textArea.getActionMap().put("ClickHyperlink", new ClickHyperlinkAction());
        }
        return textArea;
    }

    static class ClickHyperlinkAction extends TextAction {
        public ClickHyperlinkAction() {
            super("ClickHyperlink");
        }

        public void actionPerformed(ActionEvent ae) {
            JTextPane component = (JTextPane) getFocusedComponent();
            HTMLDocument doc = (HTMLDocument) component.getDocument();
            int position = 1;
            Element e = doc.getCharacterElement(position);
            AttributeSet as = e.getAttributes();
            AttributeSet anchor = (AttributeSet) as.getAttribute(HTML.Tag.A);

            if (anchor != null) {
                try {
                    Rectangle r = component.modelToView(position);

                    MouseEvent me = new MouseEvent(
                            component,
                            MouseEvent.MOUSE_CLICKED,
                            System.currentTimeMillis(),
                            InputEvent.BUTTON1_MASK,
                            r.x,
                            r.y,
                            1,
                            false);

                    component.dispatchEvent(me);
                } catch (BadLocationException ble) {
                }
            }
        }
    }

    @Override
    public Dimension getMinimumSize() {
        int totalWidth = 0;
        int totalHeight = 0;
        int rowWidth = 0;
        int rowHeight = 0;
        for (Component comp : getComponents()) {
            Dimension compDim = comp.getPreferredSize();
            if (rowWidth + compDim.width > CONTENT_PANEL_WIDTH) {
                totalWidth = rowWidth;
                rowWidth = compDim.width;
                totalHeight += rowHeight;
                rowHeight = compDim.height + TEXT_PANE_HGAP*2; // make sure we add spacing between lines
            } else {
                rowWidth += compDim.width;
                rowHeight = (rowHeight > compDim.height)?rowHeight:compDim.height;
            }
        }

        totalWidth = (totalWidth > rowWidth)?totalWidth:rowWidth;
        totalHeight += rowHeight + TEXT_PANE_HGAP;
        return new Dimension(totalWidth, totalHeight);
    }
}
