/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * <p>
 * http://www.apache.org/licenses/LICENSE-2.0
 * <p>
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.jboss.installer.validators;

import org.apache.commons.lang3.StringUtils;
import org.jboss.installer.core.LanguageUtils;
import org.jboss.installer.core.ValidationResult;
import org.jboss.installer.postinstall.task.secdom.CertSecurity;
import org.wildfly.security.asn1.OidsUtil;

import static org.jboss.installer.screens.domain.CertificateSecurityDomainPanel.FILTER_VALUE_KEY;
import static org.jboss.installer.screens.domain.CertificateSecurityDomainPanel.MAX_SEGMENT_KEY;
import static org.jboss.installer.screens.domain.CertificateSecurityDomainPanel.ROLES_KEY;
import static org.jboss.installer.screens.domain.CertificateSecurityDomainPanel.START_SEGMENT_KEY;
import static org.jboss.installer.screens.domain.CertificateSecurityDomainPanel.TRUST_STORE_PASSWORD_KEY;
import static org.jboss.installer.screens.domain.CertificateSecurityDomainPanel.TRUST_STORE_PATH_KEY;

public class CertificateSecurityDomainValidator {

    public static final String GENERIC_VALIDATION_EMPTY_FIELD = "generic.validation.empty_field";
    public static final String UNSUPPORTED_X509_FIELD = "security_domain_screen.cert.unsupported.x509.field";
    private final LanguageUtils langUtils;


    public CertificateSecurityDomainValidator(LanguageUtils langUtils) {
        this.langUtils = langUtils;
    }

    public ValidationResult validate(CertSecurity.CertificateConfig config) {
        if (StringUtils.isEmpty(config.getTrustStorePassword())) {
            return ValidationResult.error(langUtils.getString(GENERIC_VALIDATION_EMPTY_FIELD, langUtils.getString(TRUST_STORE_PASSWORD_KEY)));
        }
        if (config.getTrustStorePath() == null) {
            return ValidationResult.error(langUtils.getString(GENERIC_VALIDATION_EMPTY_FIELD, langUtils.getString(TRUST_STORE_PATH_KEY)));
        } else {
            ValidationResult res =  new FileValidator(langUtils).validate(config.getTrustStorePath());
            if (res.getResult() != ValidationResult.Result.OK) {
                return res;
            }
        }
        if (StringUtils.isEmpty(config.getFilterExpression())) {
            return ValidationResult.error(langUtils.getString(GENERIC_VALIDATION_EMPTY_FIELD, langUtils.getString(FILTER_VALUE_KEY)));
        }
        if (!config.isUseOid() && OidsUtil.attributeNameToOid(OidsUtil.Category.RDN, config.getFilterExpression()) == null) {
            return ValidationResult.error(langUtils.getString(UNSUPPORTED_X509_FIELD, config.getFilterExpression()));
        }
        if (config.isUseOid() && OidsUtil.oidToAttributeName(OidsUtil.Category.RDN, config.getFilterExpression()) == null) {
            return ValidationResult.error(langUtils.getString(UNSUPPORTED_X509_FIELD, config.getFilterExpression()));
        }
        if (config.getRoles() == null || config.getRoles().length == 0) {
            return ValidationResult.error(langUtils.getString(GENERIC_VALIDATION_EMPTY_FIELD, langUtils.getString(ROLES_KEY)));
        }
        if (config.getMaximumSegments() == null) {
            return ValidationResult.error(langUtils.getString(GENERIC_VALIDATION_EMPTY_FIELD, langUtils.getString(MAX_SEGMENT_KEY)));
        }
        if (config.getStartSegments() == null) {
            return ValidationResult.error(langUtils.getString(GENERIC_VALIDATION_EMPTY_FIELD, langUtils.getString(START_SEGMENT_KEY)));
        }
        return ValidationResult.ok();
    }
}
