/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2021 Red Hat, Inc., and individual contributors
 * as indicated by the @author tags.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.jboss.installer.screens.domain;

import org.jboss.installer.core.ValidationResult;
import org.jboss.installer.test.utils.MockLanguageUtils;
import org.junit.Test;

import static org.jboss.installer.screens.domain.SecurityDomainScreen.AUTHENTICATION_TYPE_NONE_ERROR;
import static org.jboss.installer.screens.domain.SecurityDomainScreen.DOMAIN_NAME_LABEL;
import static org.jboss.installer.screens.domain.SecurityDomainScreen.SECURITY_DOMAIN_WHITESPACE_ERROR;
import static org.jboss.installer.validators.CertificateSecurityDomainValidator.GENERIC_VALIDATION_EMPTY_FIELD;
import static org.junit.Assert.assertEquals;

public class SecurityDomainScreenTest {

    @Test
    public void whitespacesOnlyDomainName_NotAllowed() throws Exception {
        final SecurityDomainScreen screen = new SecurityDomainScreen(null, new MockLanguageUtils(), false) {
            @Override
            String getDomainName() {
                return "  ";
            }
        };

        final ValidationResult result = screen.validate();

        assertEquals(result.toString(), ValidationResult.Result.ERROR, result.getResult());
        assertEquals(GENERIC_VALIDATION_EMPTY_FIELD + " [" + DOMAIN_NAME_LABEL + "]", result.getMessage());
    }

    @Test
    public void emptyDomainName_NotAllowed() throws Exception {
        final SecurityDomainScreen screen = new SecurityDomainScreen(null, new MockLanguageUtils(), false) {
            @Override
            String getDomainName() {
                return "";
            }
        };

        final ValidationResult result = screen.validate();

        assertEquals(result.toString(), ValidationResult.Result.ERROR, result.getResult());
        assertEquals(GENERIC_VALIDATION_EMPTY_FIELD + " [" + DOMAIN_NAME_LABEL + "]", result.getMessage());
    }

    @Test
    public void spaceInWord_NotAllowed() throws Exception {
        final SecurityDomainScreen screen = new SecurityDomainScreen(null, new MockLanguageUtils(), false) {
            @Override
            String getDomainName() {
                return "foo bar";
            }
        };

        final ValidationResult result = screen.validate();

        assertEquals(result.toString(), ValidationResult.Result.ERROR, result.getResult());
        assertEquals(SECURITY_DOMAIN_WHITESPACE_ERROR + " [" + DOMAIN_NAME_LABEL + "]", result.getMessage());
    }

    @Test
    public void authenticatorType_NotSelected() throws Exception {
        final SecurityDomainScreen screen = new SecurityDomainScreen(null, new MockLanguageUtils(), false) {
            @Override
            String getDomainName() {
                return "test";
            }

            @Override
            int getSelectedIndex() {
                return NONE_INDEX;
            }
        };

        final ValidationResult result = screen.validate();

        assertEquals(result.toString(), ValidationResult.Result.ERROR, result.getResult());
        assertEquals(AUTHENTICATION_TYPE_NONE_ERROR, result.getMessage());
    }

}