/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2021, Red Hat, Inc., and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.wildfly.clustering.web.cache.session.fine;

import java.io.IOException;

import org.infinispan.protostream.descriptors.WireType;
import org.wildfly.clustering.marshalling.protostream.ProtoStreamMarshaller;
import org.wildfly.clustering.marshalling.protostream.ProtoStreamReader;
import org.wildfly.clustering.marshalling.protostream.ProtoStreamWriter;
import org.wildfly.clustering.marshalling.spi.ByteBufferMarshalledValue;

/**
 * @author Paul Ferraro
 */
public class SessionAttributeMapEntryMarshaller implements ProtoStreamMarshaller<SessionAttributeMapEntry<ByteBufferMarshalledValue<Object>>> {

    private static final int NAME_INDEX = 1;
    private static final int VALUE_INDEX = 2;

    @SuppressWarnings("unchecked")
    @Override
    public Class<? extends SessionAttributeMapEntry<ByteBufferMarshalledValue<Object>>> getJavaClass() {
        return (Class<SessionAttributeMapEntry<ByteBufferMarshalledValue<Object>>>) (Class<?>) SessionAttributeMapEntry.class;
    }

    @Override
    public SessionAttributeMapEntry<ByteBufferMarshalledValue<Object>> readFrom(ProtoStreamReader reader) throws IOException {
        String name = null;
        ByteBufferMarshalledValue<Object> value = null;
        while (!reader.isAtEnd()) {
            int tag = reader.readTag();
            switch (WireType.getTagFieldNumber(tag)) {
                case NAME_INDEX:
                    name = reader.readString();
                    break;
                case VALUE_INDEX:
                    value = reader.readObject(ByteBufferMarshalledValue.class);
                    break;
                default:
                    reader.skipField(tag);
            }
        }
        return new SessionAttributeMapEntry<>(name, value);
    }

    @Override
    public void writeTo(ProtoStreamWriter writer, SessionAttributeMapEntry<ByteBufferMarshalledValue<Object>> entry) throws IOException {
        String name = entry.getKey();
        if (name != null) {
            writer.writeString(NAME_INDEX, name);
        }
        ByteBufferMarshalledValue<Object> value = entry.getValue();
        if (value != null) {
            writer.writeObject(VALUE_INDEX, value);
        }
    }
}
