/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2022 Red Hat, Inc., and individual contributors
 * as indicated by the @author tags.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.jboss.installer.core;

import org.jboss.installer.common.UiResources;
import org.jboss.installer.postinstall.task.PortBindingConfig;

import javax.swing.JFormattedTextField;
import javax.swing.JTextField;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

import static org.jboss.installer.screens.CustomPortConfigurationScreen.JGROUPS_MPING_ADDRESS_KEY;
import static org.jboss.installer.screens.CustomPortConfigurationScreen.JGROUPS_MPING_MCAST_PORT_KEY;
import static org.jboss.installer.screens.CustomPortConfigurationScreen.JGROUPS_UDP_ADDRESS_KEY;
import static org.jboss.installer.screens.CustomPortConfigurationScreen.JGROUPS_UDP_KEY;
import static org.jboss.installer.screens.CustomPortConfigurationScreen.JGROUPS_UDP_MCAST_PORT_KEY;
import static org.jboss.installer.screens.CustomPortConfigurationScreen.KEY_PREFIX;
import static org.jboss.installer.screens.CustomPortConfigurationScreen.MODCLUSTER_ADDRESS_KEY;
import static org.jboss.installer.screens.CustomPortConfigurationScreen.MODCLUSTER_MCAST_PORT_KEY;
import static org.jboss.installer.screens.CustomPortConfigurationScreen.portOnlyBindings;

public class SocketBindingUtils {

    private final HashMap<String, JTextField> systemPropertyFieldMap;
    private final HashMap<String, JTextField> ipAddressFieldMap;
    private final HashMap<String, JFormattedTextField> portFieldMap;
    public SocketBindingUtils(HashMap<String, JTextField> systemPropertyFieldMap,
                              HashMap<String, JTextField> ipAddressFieldMap,
                              HashMap<String, JFormattedTextField> portFieldMap) {
        this.ipAddressFieldMap = ipAddressFieldMap;
        this.portFieldMap = portFieldMap;
        this.systemPropertyFieldMap = systemPropertyFieldMap;
    }

    public List<PortBindingConfig.SocketBinding> createSocketBindings(List<String> properties) {
        List<PortBindingConfig.SocketBinding> socketBindings = new ArrayList<>();
        for (String key : properties) {
            if (isPort(key)) {
                socketBindings.add(createPortSocketBinding(key));
            }
            if (key.equals(JGROUPS_UDP_KEY)) {
                socketBindings.add(createUdpSocketBinding());
            }
            if (key.equals(JGROUPS_MPING_MCAST_PORT_KEY)) {
                socketBindings.add(createMpingSocketBinding());
            }
            if (key.equals(MODCLUSTER_MCAST_PORT_KEY)) {
                socketBindings.add(createModclusterSocketBinding());
            }
        }
        return socketBindings;
    }

    public PortBindingConfig.SocketBinding createPortSocketBinding(String key) {
        return new PortBindingConfig.SocketBinding(
                getSocketBindingName(key),
                UiResources.readTextValue(systemPropertyFieldMap.get(key)),
                Integer.parseInt(UiResources.readTextValue(portFieldMap.get(key))));
    }

    public PortBindingConfig.SocketBinding createModclusterSocketBinding() {
        return new PortBindingConfig.SocketBinding(
                getSocketBindingName(MODCLUSTER_MCAST_PORT_KEY),
                UiResources.readTextValue(systemPropertyFieldMap.get(MODCLUSTER_ADDRESS_KEY)),
                UiResources.readTextValue(ipAddressFieldMap.get(MODCLUSTER_ADDRESS_KEY)),
                Integer.parseInt(UiResources.readTextValue(portFieldMap.get(MODCLUSTER_MCAST_PORT_KEY))),
                null);
    }

    public PortBindingConfig.SocketBinding createMpingSocketBinding() {
        return new PortBindingConfig.SocketBinding(
                getSocketBindingName(JGROUPS_MPING_MCAST_PORT_KEY),
                UiResources.readTextValue(systemPropertyFieldMap.get(JGROUPS_MPING_ADDRESS_KEY)),
                UiResources.readTextValue(ipAddressFieldMap.get(JGROUPS_MPING_ADDRESS_KEY)),
                Integer.parseInt(UiResources.readTextValue(portFieldMap.get(JGROUPS_MPING_MCAST_PORT_KEY))),
                null);
    }

    public PortBindingConfig.SocketBinding createUdpSocketBinding() {
        return new PortBindingConfig.SocketBinding(
                getSocketBindingName(JGROUPS_UDP_KEY),
                UiResources.readTextValue(systemPropertyFieldMap.get(JGROUPS_UDP_ADDRESS_KEY)),
                UiResources.readTextValue(ipAddressFieldMap.get(JGROUPS_UDP_ADDRESS_KEY)),
                Integer.parseInt(UiResources.readTextValue(portFieldMap.get(JGROUPS_UDP_MCAST_PORT_KEY))),
                Integer.parseInt(UiResources.readTextValue(portFieldMap.get(JGROUPS_UDP_KEY))));
    }

    private String getSocketBindingName(String key) {
        return key.replace(KEY_PREFIX, "");
    }

    private boolean isPort(String key) {
        return portOnlyBindings.contains(key);
    }
}
