package org.jboss.installer.validators;

import org.jboss.installer.postinstall.task.JSFLibraryConfig;
import org.jboss.installer.test.utils.TestProperties;
import org.junit.Test;

import java.io.File;
import java.net.URL;
import java.util.jar.JarFile;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

public class JsfUtilsTest {

    private final JsfUtils jsfUtils = new JsfUtils();

    @Test
    public void testCleanedVerTrimmed() throws Exception{
        String result = jsfUtils.cleanedVer("4.0-SNAPSHOT");
        assertEquals("4.0", result);
    }

    @Test
    public void testCleanedVerUntrimmedShort() throws Exception{
        String result = jsfUtils.cleanedVer("4.0");
        assertEquals("4.0", result);
    }

    @Test
    public void testCleanedVerWhiteSpace() throws Exception{
        String result = jsfUtils.cleanedVer("   4.0   ");
        assertEquals("4.0", result);
    }

    @Test
    public void testCleanedVerUntrimmedLong() throws Exception{
        String result = jsfUtils.cleanedVer("4.0.0");
        assertEquals("4.0.0", result);
    }

    @Test
    public void testCleanedVerUntrimmedLong40() throws Exception{
        String result = jsfUtils.cleanedVer("4.0.0");
        assertEquals("4.0.0", result);
    }

    @Test
    public void testCleanedVerInvalidString() throws Exception{
        String result = jsfUtils.cleanedVer("SNAPSHOT");
        assertEquals(null, result);
    }

    @Test
    public void testCleanedVerInvalidVersion() throws Exception{
        String result = jsfUtils.cleanedVer("1.0");
        assertEquals(null, result);
    }

    @Test
    public void testCleanedVerTooLong() throws Exception{
        String result = jsfUtils.cleanedVer("4.0.222");
        assertEquals(null, result);
    }

    @Test
    public void testCleanedVerDecorator1() throws Exception{
        String result = jsfUtils.cleanedVer("4.0asdf");
        assertEquals("4.0", result);
    }

    @Test
    public void testCleanedVerDecorator2() throws Exception{
        String result = jsfUtils.cleanedVer("4.0 asdf");
        assertEquals("4.0", result);
    }

    @Test
    public void testCleanedVerDecorator3() throws Exception{
        String result = jsfUtils.cleanedVer("4.0-asdf");
        assertEquals("4.0", result);
    }

    @Test
    public void testCleanedVerDecorator4() throws Exception{
        String result = jsfUtils.cleanedVer("4.0_asdf");
        assertEquals("4.0", result);
    }

    @Test
    public void testCleanedVerEmpty() throws Exception{
        String result = jsfUtils.cleanedVer("");
        assertEquals(null, result);
    }

    @Test
    public void testCleanedVerDecoratorEndWithDelim() throws Exception{
        String result = jsfUtils.cleanedVer("4.0.snapshot.");
        assertEquals(null, result);
    }

    @Test
    public void testCleanedVerEndWithDelim1() throws Exception{
        String result = jsfUtils.cleanedVer("4.0.");
        assertEquals(null, result);
    }

    @Test
    public void testCleanedVerEndWithDelim2() throws Exception{
        String result = jsfUtils.cleanedVer("4.0_");
        assertEquals(null, result);
    }

    @Test
    public void testCleanedVerEndWithDelim3() throws Exception{
        String result = jsfUtils.cleanedVer("4.0-");
        assertEquals(null, result);
    }

    @Test
    public void testMyFacesApiIdentifiedAsApi() throws Exception {
        final URL apiJarUrl = JsfValidatorTest.class.getClassLoader().getResource("jars/myfaces-api.jar");

        final JarFile jarFile = new JarFile(new File(apiJarUrl.toURI()));
        assertTrue(jsfUtils.verifyIsApi(jarFile));
    }

    @Test
    public void testMojarraImplIdentifiedAsImpl() throws Exception {
        final URL apiJarUrl = JsfValidatorTest.class.getClassLoader().getResource("jars/mojarra-impl.jar");

        final JarFile jarFile = new JarFile(new File(apiJarUrl.toURI()));
        assertTrue(jsfUtils.verifyIsImpl(jarFile));
    }

    @Test
    public void testMyFacesImplIdentifiedAsImpl() throws Exception {
        final URL apiJarUrl = JsfValidatorTest.class.getClassLoader().getResource("jars/myfaces-impl.jar");

        final JarFile jarFile = new JarFile(new File(apiJarUrl.toURI()));
        assertTrue(jsfUtils.verifyIsImpl(jarFile));
    }

    @Test
    public void testReadVersionFromMojarraImpl() throws Exception {
        final URL apiJarUrl = JsfValidatorTest.class.getClassLoader().getResource("jars/mojarra-impl.jar");

        final JarFile jarFile = new JarFile(new File(apiJarUrl.toURI()));
        assertEquals("4.0.0.SP01", jsfUtils.getVersion(jarFile));
    }

    @Test
    public void testReadVersionFromMyFacesApi() throws Exception {
        final URL apiJarUrl = JsfValidatorTest.class.getClassLoader().getResource("jars/myfaces-api.jar");

        final JarFile jarFile = new JarFile(new File(apiJarUrl.toURI()));
        assertEquals(TestProperties.getMyFacesVersion(), jsfUtils.getVersion(jarFile));
    }

    @Test
    public void testReadVersionFromMyFacesImpl() throws Exception {
        final URL apiJarUrl = JsfValidatorTest.class.getClassLoader().getResource("jars/myfaces-impl.jar");

        final JarFile jarFile = new JarFile(new File(apiJarUrl.toURI()));
        assertEquals(TestProperties.getMyFacesVersion(), jsfUtils.getVersion(jarFile));
    }

    @Test
    public void testReadProjectFromMojarraImpl() throws Exception {
        final URL implJarUrl = JsfValidatorTest.class.getClassLoader().getResource("jars/mojarra-impl.jar");

        final JarFile jarFile = new JarFile(new File(implJarUrl.toURI()));
        assertEquals(JSFLibraryConfig.JsfProject.Mojarra, jsfUtils.getProject(jarFile));
    }
}