/*
 * Copyright The WildFly Authors
 * SPDX-License-Identifier: Apache-2.0
 */
package org.jboss.as.test.integration.domain.suites;

import static org.jboss.as.controller.descriptions.ModelDescriptionConstants.ADD;
import static org.jboss.as.controller.descriptions.ModelDescriptionConstants.NAME;
import static org.jboss.as.controller.descriptions.ModelDescriptionConstants.READ_OPERATION_DESCRIPTION_OPERATION;
import static org.jboss.as.controller.descriptions.ModelDescriptionConstants.VALIDATE_OPERATION;
import static org.jboss.as.controller.descriptions.ModelDescriptionConstants.VALUE;

import java.io.IOException;
import java.net.UnknownHostException;

import org.junit.Assert;

import org.jboss.as.controller.client.ModelControllerClient;
import org.jboss.as.test.integration.domain.management.util.DomainTestSupport;
import org.jboss.as.test.integration.management.base.AbstractMgmtTestBase;
import org.jboss.as.test.integration.management.util.MgmtOperationException;
import org.jboss.as.test.integration.management.util.ModelUtil;
import org.jboss.dmr.ModelNode;
import org.junit.AfterClass;
import org.junit.BeforeClass;
import org.junit.Test;

/**
 * Tests that the validate-operation operation works as it should
 *
 * @author <a href="kabir.khan@jboss.com">Kabir Khan</a>
 */
public class ValidateOperationOperationTestCase extends AbstractMgmtTestBase {


    private static final String PRIMARY = "primary";
    private static final String SECONDARY = "secondary";
    private static final String PRIMARY_SERVER = "main-one";
    private static final String SECONDARY_SERVER = "main-three";

    private static ModelControllerClient client;

    @BeforeClass
    public static void setup() throws UnknownHostException {
        DomainTestSupport testSupport = DomainTestSuite.createSupport(ValidateOperationOperationTestCase.class.getSimpleName());
        client = testSupport.getDomainPrimaryLifecycleUtil().getDomainClient();
    }

    @AfterClass
    public static void afterClass() throws IOException {
        DomainTestSuite.stopSupport();
        client.close();
        client = null;
    }

    @Override
    protected ModelControllerClient getModelControllerClient() {
        return client;
    }

    @Test
    public void testValidRootOperation() throws IOException, MgmtOperationException {
        ModelNode op = ModelUtil.createOpNode(null, READ_OPERATION_DESCRIPTION_OPERATION);
        op.get(NAME).set("Doesn't matter");
        executeOperation(createValidateOperation(op));
    }

    @Test
    public void testFailedRootOperation() throws IOException {
        ModelNode op = ModelUtil.createOpNode(null, READ_OPERATION_DESCRIPTION_OPERATION);
        executeInvalidOperation(op);
    }

    @Test
    public void testValidDcOperation() throws IOException, MgmtOperationException {
        ModelNode op = ModelUtil.createOpNode("profile=default/subsystem=jmx/remoting-connector=jmx", ADD);
        executeOperation(createValidateOperation(op));
    }

    @Test
    public void testInvalidDcOperation() throws IOException {
        ModelNode op = ModelUtil.createOpNode("profile=default/subsystem=jmx/remoting-connector=jmx", ADD);
        op.get("badata").set("junk");
        executeInvalidOperation(op);
    }

    @Test
    public void testValidPrimaryHcOperation() throws IOException, MgmtOperationException {
        testValidHcOperation(PRIMARY);
    }

    @Test
    public void testValidSecondaryHcOperation() throws IOException, MgmtOperationException {
        testValidHcOperation(SECONDARY);
    }

    public void testValidHcOperation(String host) throws IOException, MgmtOperationException {
        ModelNode op = ModelUtil.createOpNode("host=" + host + "/path=ff", ADD);
        op.get("path").set("/home");
        executeOperation(createValidateOperation(op));
    }

    @Test
    public void testInvalidPrimaryHcOperation() throws IOException {
        testInvalidHcOperation(PRIMARY);
    }

    @Test
    public void testInvalidSecondaryHcOperation() throws IOException {
        testInvalidHcOperation(SECONDARY);
    }

    private void testInvalidHcOperation(String host) throws IOException {
        ModelNode op = ModelUtil.createOpNode("host=" + host + "/path=ff", ADD);
        executeInvalidOperation(op);
    }

    @Test
    public void testValidPrimaryHcServerOperation() throws IOException, MgmtOperationException {
        testValidServerOperation(PRIMARY, PRIMARY_SERVER);
    }

    @Test
    public void testValidSecondaryHcServerOperation() throws IOException, MgmtOperationException {
        testValidServerOperation(SECONDARY, SECONDARY_SERVER);
    }

    private void testValidServerOperation(String host, String server) throws IOException, MgmtOperationException {
        ModelNode op = ModelUtil.createOpNode("host=" + host + "/server=" + server + "/subsystem=jmx/remoting-connector=jmx", ADD);
        executeOperation(createValidateOperation(op));
    }

    @Test
    public void testInvalidPrimaryHcServerOperation() throws IOException, MgmtOperationException {
        testInvalidServerOperation(PRIMARY, PRIMARY_SERVER);
    }

    @Test
    public void testInvalidSecondaryHcServerOperation() throws IOException, MgmtOperationException {
        testInvalidServerOperation(SECONDARY, SECONDARY_SERVER);
    }

    private void testInvalidServerOperation(String host, String server) throws IOException, MgmtOperationException {
        ModelNode op = ModelUtil.createOpNode("host=" + host + "/server=" + server + "/subsystem=jmx/remoting-connector=jmx", ADD);
        op.get("badata").set("junk");
        executeInvalidOperation(op);
    }

    private ModelNode createValidateOperation(ModelNode validatedOperation) throws IOException {
        ModelNode node = ModelUtil.createOpNode(null, VALIDATE_OPERATION);
        node.get(VALUE).set(validatedOperation);
        return node;
    }

    private void executeInvalidOperation(ModelNode operation) throws IOException {
        try {
            executeOperation(createValidateOperation(operation));
            Assert.fail("Should have failed on no required parameter included");
        } catch (MgmtOperationException expected) {
        }
    }
}
