/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2024 Red Hat, Inc., and individual contributors
 * as indicated by the @author tags.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.jboss.installer.postinstall.task;

import static org.jboss.installer.postinstall.ldap.LdapModel.LDAP_PASSWORD_KEY;
import static org.jboss.installer.postinstall.task.secdom.CertificateConfig.TRUST_STORE_PASSWORD_KEY;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.function.BiFunction;

import org.jboss.installer.auto.AutomaticInstallationParsingException;
import org.jboss.installer.core.FlatListPostInstallConfig;
import org.jboss.installer.postinstall.ldap.LdapModel;
import org.jboss.installer.postinstall.task.secdom.CertificateConfig;
import org.jboss.installer.postinstall.task.secdom.JdbcConfig;
import org.jboss.installer.postinstall.task.secdom.KerberosConfig;
import org.jboss.installer.postinstall.task.secdom.PropertiesFileConfig;

public class SecurityDomainConfig extends FlatListPostInstallConfig {
    public static final String REALM_SUFFIX = "-realm";
    private String domainName;
    private PropertiesFileConfig propertiesFileConfig;
    private JdbcConfig jdbcConfig;
    private ConfigType configType = ConfigType.None;
    private LdapModel ldapConfig;
    private CertificateConfig certConfig;
    private KerberosConfig kerberosConfig;

    public void setCertConfig(CertificateConfig certConfig) {
        this.certConfig = certConfig;
        this.configType = ConfigType.Cert;
    }

    public CertificateConfig getCertConfig() {
        return certConfig;
    }

    public enum ConfigType {
        None, Properties, Jdbc, Ldap, Cert, Kerberos
    }

    public ConfigType getConfigType() {
        return configType;
    }

    public String getDomainName() {
        return domainName;
    }

    public void setDomainName(String domainName) {
        this.domainName = domainName;
    }

    public PropertiesFileConfig getPropertiesFileConfig() {
        return propertiesFileConfig;
    }

    public void setPropertiesFileConfig(PropertiesFileConfig propertiesFileConfig) {
        this.propertiesFileConfig = propertiesFileConfig;
        this.configType = ConfigType.Properties;
    }

    public JdbcConfig getJdbcConfig() {
        return jdbcConfig;
    }

    public void setJdbcFileConfig(JdbcConfig jdbcConfig) {
        this.jdbcConfig = jdbcConfig;
        this.configType = ConfigType.Jdbc;
    }

    public void setLdapConfig(LdapModel config) {
        this.ldapConfig = config;
        this.configType = ConfigType.Ldap;
    }

    public LdapModel getLdapConfig() {
        return ldapConfig;
    }

    public KerberosConfig getKerberosConfig() {
        return kerberosConfig;
    }

    public void setKerberosConfig(KerberosConfig kerberosConfig) {
        this.kerberosConfig = kerberosConfig;
        this.configType = ConfigType.Kerberos;
    }

    @Override
    protected Map<String, String> listAttributes() {
        final HashMap<String, String> attrs = new HashMap<>();
        attrs.put("domainName", domainName);
        attrs.put("configType", configType.toString());
        switch (configType) {
            case Properties:
                attrs.putAll(propertiesFileConfig.toAttributes());
                break;
            case Jdbc:
                attrs.putAll(jdbcConfig.toAttributes());
                break;
            case Ldap:
                attrs.putAll(ldapConfig.toAttributes());
                break;
            case Cert:
                attrs.putAll(certConfig.toAttributes());
                break;
            case Kerberos:
                attrs.putAll(kerberosConfig.toAttributes());
                break;
            default:
                throw new IllegalStateException("Unknown config type: " + configType);
        }
        return attrs;
    }

    @Override
    protected Set<String> listVariables() {
        HashSet<String> vars = new HashSet<>();
        if (configType == ConfigType.Ldap) {
            vars.add(LDAP_PASSWORD_KEY);
        }
        if (configType == ConfigType.Cert) {
            vars.add(TRUST_STORE_PASSWORD_KEY);
        }
        return vars;
    }

    @Override
    protected void acceptAttributes(Map<String, String> attributes, BiFunction<String, String, String> variableResolver) throws AutomaticInstallationParsingException {
        if (!attributes.containsKey("domainName") || !attributes.containsKey("configType")) {
            throw new AutomaticInstallationParsingException("Stored configuration doesn't have required fields ['configType', 'domainName']");
        }

        this.domainName = attributes.get("domainName");
        this.configType = ConfigType.valueOf(attributes.get("configType"));

        switch (configType) {
            case Properties:
                this.propertiesFileConfig = new PropertiesFileConfig(attributes);
                break;
            case Jdbc:
                this.jdbcConfig = new JdbcConfig(attributes);
                break;
            case Ldap:
                attributes.put(LDAP_PASSWORD_KEY, variableResolver.apply(LDAP_PASSWORD_KEY, "LDAP server password:"));
                this.ldapConfig = new LdapModel(attributes);
                break;
            case Cert:
                attributes.put(TRUST_STORE_PASSWORD_KEY, variableResolver.apply(TRUST_STORE_PASSWORD_KEY, "Trust store password:"));
                this.certConfig = new CertificateConfig(attributes);
                break;
            case Kerberos:
                this.kerberosConfig = new KerberosConfig(attributes);
                break;
            default:
                throw new AutomaticInstallationParsingException("Unknown config type: " + configType);
        }
    }
}
