/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2021 Red Hat, Inc., and individual contributors
 * as indicated by the @author tags.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.jboss.installer.core;

import org.jboss.installer.actions.ActionException;
import org.jboss.installer.actions.impl.InstallEap;
import org.jboss.installer.test.utils.MockLanguageUtils;
import org.jboss.installer.test.utils.TestProperties;
import org.junit.Assert;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;

import java.beans.PropertyChangeSupport;
import java.io.IOException;
import java.nio.file.Path;
import java.util.Arrays;
import java.util.Collections;

import static org.jboss.installer.screens.ComponentSelectionScreen.APPCLIENT_PACKAGE;
import static org.jboss.installer.screens.ComponentSelectionScreen.DOCS_PACKAGE;

public class ComponentSelectionTest {

    public static final String EAP_FOLDER = "eap";
    @Rule
    public final TemporaryFolder tempDir = new TemporaryFolder();
    private LanguageUtils langUtils = new MockLanguageUtils();
    @Test
    public void docsPackageExcludedTest() throws IOException, ActionException {
        Path dirPath = tempDir.newFolder(EAP_FOLDER).toPath();
        dirPath.toFile().setWritable(true);
        final InstallEap installEap = new InstallEap(dirPath, evt -> {}, Collections.singletonList(DOCS_PACKAGE),
                TestProperties.getTestRepositories(), langUtils);
        installEap.perform(new PropertyChangeSupport(new Object()));

        Assert.assertTrue(isDocsPackageExcluded(dirPath));
        Assert.assertFalse(isExamplesConfigsIncluded(dirPath));
    }


    @Test
    public void appClientPackageExcludedTest() throws IOException, ActionException {
        Path dirPath = tempDir.newFolder(EAP_FOLDER).toPath();
        dirPath.toFile().setWritable(true);
        final InstallEap installEap = new InstallEap(dirPath, evt -> {}, Collections.singletonList(APPCLIENT_PACKAGE),
                TestProperties.getTestRepositories(), langUtils);
        installEap.perform(new PropertyChangeSupport(new Object()));

        Assert.assertTrue(isAppClientPackageExcluded(dirPath));
    }

    @Test
    public void allPackagesIncludedTest() throws IOException, ActionException {
        Path dirPath = tempDir.newFolder(EAP_FOLDER).toPath();
        dirPath.toFile().setWritable(true);
        final InstallEap installEap = new InstallEap(dirPath, evt -> {}, Collections.emptyList(),
                TestProperties.getTestRepositories(), langUtils);
        installEap.perform(new PropertyChangeSupport(new Object()));

        Assert.assertFalse(isAppClientPackageExcluded(dirPath));
        Assert.assertFalse(isDocsPackageExcluded(dirPath));
        Assert.assertTrue(isExamplesConfigsIncluded(dirPath));
    }

    @Test
    public void allPackagesExcludedTest() throws IOException, ActionException {
        Path dirPath = tempDir.newFolder(EAP_FOLDER).toPath();
        dirPath.toFile().setWritable(true);
        final InstallEap installEap = new InstallEap(dirPath, evt -> {},
                Arrays.asList(APPCLIENT_PACKAGE, DOCS_PACKAGE),
                TestProperties.getTestRepositories(), langUtils);
        installEap.perform(new PropertyChangeSupport(new Object()));

        Assert.assertTrue(isAppClientPackageExcluded(dirPath));
        Assert.assertTrue(isDocsPackageExcluded(dirPath));
        Assert.assertFalse(isExamplesConfigsIncluded(dirPath));
    }

    @Test
    public void docsExamplesConfigsIncludedTest() throws IOException, ActionException {
        Path dirPath = tempDir.newFolder(EAP_FOLDER).toPath();
        dirPath.toFile().setWritable(true);
        final InstallEap installEap = new InstallEap(dirPath, evt -> {}, Collections.emptyList(),
                TestProperties.getTestRepositories(), langUtils);
        installEap.perform(new PropertyChangeSupport(new Object()));

        Assert.assertTrue(isExamplesConfigsIncluded(dirPath));
    }

    private boolean isDocsPackageExcluded(Path dir) {
        return Arrays.stream(dir.resolve(DOCS_PACKAGE).toFile().list()).allMatch(i -> i.equals("licenses"));
    }

    private boolean isExamplesConfigsIncluded(Path dir) {
        return Arrays.asList(dir.resolve(DOCS_PACKAGE).toFile().list()).contains("examples");
    }

    private boolean isAppClientPackageExcluded(Path dir) {
        return !Arrays.asList(dir.toFile().list()).contains(APPCLIENT_PACKAGE);
    }
}
