/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2024 Red Hat, Inc., and individual contributors
 * as indicated by the @author tags.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.jboss.installer.postinstall.task;

import static org.jboss.installer.postinstall.task.AbstractHttpsEnableConfig.DEFAULT_CIPHER_SUITES_TLS12;
import static org.jboss.installer.postinstall.task.AbstractHttpsEnableConfig.KEYSTORE_PASSWORD_VARIABLE_KEY;
import static org.jboss.installer.postinstall.task.AbstractHttpsEnableConfig.PROTOCOL_TLSv13;
import static org.jboss.installer.postinstall.task.AbstractHttpsEnableConfig.TRUSTSTORE_PASSWORD_VARIABLE_KEY;
import static org.jboss.installer.postinstall.task.AbstractHttpsEnableConfig.DEFAULT_CIPHER_NAMES_TLS13;
import static org.junit.Assert.assertEquals;

import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.List;
import java.util.Optional;
import java.util.Set;

import org.apache.commons.io.FileUtils;
import org.jboss.installer.auto.InstallationDataSerializer;
import org.jboss.installer.core.InstallationData;
import org.jboss.installer.test.utils.MockLanguageUtils;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;

public class HttpsEnableDescriptorTest {
    private static final String KEYSTORE_PASSWORD = "keystore-secret";
    private static final String KEYSTORE_PATH = "test.keystore";
    private static final String TRUSTSTORE_PASSWORD = "truststore-secret";
    private static final String TRUSTSTORE_PATH = "test.truststore";
    private static final String TEST_DN = "CN=test";
    private static final String VALIDITY = "30";
    private static final String CLIENT_CERT_PATH = Paths.get("src", "test", "resources", "client.crt").toAbsolutePath().toString();

    private static final AdminHttpsConfig HTTP_ENABLE_CONFIG = new AdminHttpsConfig(
            Set.of(PROTOCOL_TLSv13), DEFAULT_CIPHER_SUITES_TLS12, DEFAULT_CIPHER_NAMES_TLS13, KEYSTORE_PATH, KEYSTORE_PASSWORD,
            true, TRUSTSTORE_PATH, TRUSTSTORE_PASSWORD,
            true, TEST_DN, VALIDITY, CLIENT_CERT_PATH, true);

    private static final ApplicationHttpsConfig APP_FULL_CONFIG = new ApplicationHttpsConfig(
            Set.of(PROTOCOL_TLSv13), DEFAULT_CIPHER_SUITES_TLS12, DEFAULT_CIPHER_NAMES_TLS13, KEYSTORE_PATH, KEYSTORE_PASSWORD,
            true, TRUSTSTORE_PATH, TRUSTSTORE_PASSWORD,
            true, TEST_DN, VALIDITY, CLIENT_CERT_PATH, true);

    private static final ApplicationHttpsConfig APP_MINIMAL_CONFIG = new ApplicationHttpsConfig(
            null, null, null, KEYSTORE_PATH, KEYSTORE_PASSWORD,
            false, null, null,
            true, null, null, null, false);

    @Rule
    public TemporaryFolder temp = new TemporaryFolder();

    private InstallationData iData = new InstallationData();

    @Test
    public void testSerializeDeserializeHttpsEnable() throws Exception {
        iData.putConfig(HTTP_ENABLE_CONFIG);
        final Path tempFile = temp.newFile("auto.xml").toPath();
        final InstallationDataSerializer serializer = new InstallationDataSerializer(new MockLanguageUtils());
        serializer.serialize(iData, tempFile);
        List<String> passwords = List.of(
                KEYSTORE_PASSWORD_VARIABLE_KEY + "=" + HTTP_ENABLE_CONFIG.getKeyStorePassword(),
                TRUSTSTORE_PASSWORD_VARIABLE_KEY + "=" + HTTP_ENABLE_CONFIG.getTrustStorePassword());
        FileUtils.writeLines(tempFile.getParent().resolve(tempFile.getFileName().toString() + ".variables").toFile(), "UTF-8", passwords);
        final InstallationData loadedData = serializer.deserialize(tempFile, Optional.empty());

        final AdminHttpsConfig config = loadedData.getConfig(AdminHttpsConfig.class);
        assertEquals(HTTP_ENABLE_CONFIG, config);
    }

    @Test
    public void testSerializeDeserializeApplicationConfig() throws Exception {
        iData.putConfig(APP_FULL_CONFIG);
        final Path tempFile = temp.newFile("auto.xml").toPath();
        final InstallationDataSerializer serializer = new InstallationDataSerializer(new MockLanguageUtils());
        serializer.serialize(iData, tempFile);
        List<String> passwords = List.of(
                KEYSTORE_PASSWORD_VARIABLE_KEY + "=" + APP_FULL_CONFIG.getKeyStorePassword(),
                TRUSTSTORE_PASSWORD_VARIABLE_KEY + "=" + APP_FULL_CONFIG.getTrustStorePassword());
        FileUtils.writeLines(tempFile.getParent().resolve(tempFile.getFileName().toString() + ".variables").toFile(), "UTF-8", passwords);
        final InstallationData loadedData = serializer.deserialize(tempFile, Optional.empty());

        final ApplicationHttpsConfig config = loadedData.getConfig(ApplicationHttpsConfig.class);
        assertEquals(APP_FULL_CONFIG, config);
    }

    @Test
    public void testSerializeDeserializeMinimalApplicationHttpConfig() throws Exception {
        iData.putConfig(APP_MINIMAL_CONFIG);
        final Path tempFile = temp.newFile("auto.xml").toPath();
        final InstallationDataSerializer serializer = new InstallationDataSerializer(new MockLanguageUtils());
        serializer.serialize(iData, tempFile);
        List<String> passwords = List.of(
                KEYSTORE_PASSWORD_VARIABLE_KEY + "=" + APP_FULL_CONFIG.getKeyStorePassword(),
                TRUSTSTORE_PASSWORD_VARIABLE_KEY + "=" + APP_FULL_CONFIG.getTrustStorePassword());
        FileUtils.writeLines(tempFile.getParent().resolve(tempFile.getFileName().toString() + ".variables").toFile(), "UTF-8", passwords);
        final InstallationData loadedData = serializer.deserialize(tempFile, Optional.empty());

        final ApplicationHttpsConfig config = loadedData.getConfig(ApplicationHttpsConfig.class);
        assertEquals(APP_MINIMAL_CONFIG, config);
    }
}
