/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2022 Red Hat, Inc., and individual contributors
 * as indicated by the @author tags.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.jboss.installer.validators;

import org.jboss.installer.core.ValidationResult;
import org.jboss.installer.test.utils.MockLanguageUtils;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;

import java.io.IOException;
import java.nio.file.Paths;

import static org.jboss.installer.validators.FileValidator.IS_DIRECTORY_ERROR;
import static org.jboss.installer.validators.FileValidator.NOT_ABSOLUTE_ERROR;
import static org.jboss.installer.validators.FileValidator.NOT_EXISTS_ERROR;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;


public class FileValidatorTest {

    @Rule
    public TemporaryFolder temp = new TemporaryFolder();
    private final FileValidator validator = new FileValidator(new MockLanguageUtils());
    private final FileValidator validatorNotExistAllowed = new FileValidator(new MockLanguageUtils(), true);

    @Test
    public void testFileIsNotAbsolute() {
        assertTrue(validator.validate(Paths.get("not/absolute/path")).getMessage().contains(NOT_ABSOLUTE_ERROR));
    }

    @Test
    public void testFileDoesNotExist() {
        assertTrue(validator.validate(Paths.get(temp.toString(), "nonexistent").toAbsolutePath()).getMessage().contains(NOT_EXISTS_ERROR));
    }

    @Test
    public void testFileDoesNotExistAllowed() {
        assertEquals(ValidationResult.ok().getResult(), validatorNotExistAllowed.validate(Paths.get(temp.toString(), "nonexistent").toAbsolutePath()).getResult());
    }

    @Test
    public void testFileIsDirectory() {
        assertTrue(validator.validate(temp.getRoot().toPath()).getMessage().contains(IS_DIRECTORY_ERROR));
    }

    @Test
    public void testGoodFile() throws IOException {
        assertEquals(ValidationResult.ok().getResult(), validator.validate(temp.newFile("tmp").toPath().toAbsolutePath()).getResult());
    }
}
