/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2022 Red Hat, Inc., and individual contributors
 * as indicated by the @author tags.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.jboss.installer.validators;

import org.apache.commons.lang3.SystemUtils;
import org.jboss.installer.core.ValidationResult;
import org.jboss.installer.postinstall.task.secdom.PropertiesFileConfig;
import org.jboss.installer.test.utils.MockLanguageUtils;
import org.junit.Assume;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;

import java.io.IOException;
import java.nio.file.Path;

import static org.assertj.core.api.Assertions.assertThat;
import static org.jboss.installer.screens.domain.PropertiesSecurityDomainPanel.GROUPS_FILE_LABEL;
import static org.jboss.installer.screens.domain.PropertiesSecurityDomainPanel.HASH_SET_DESCRIPTION;
import static org.jboss.installer.screens.domain.PropertiesSecurityDomainPanel.USERS_FILE_LABEL;
import static org.jboss.installer.validators.CertificateSecurityDomainValidator.GENERIC_VALIDATION_EMPTY_FIELD;
import static org.jboss.installer.validators.FileValidator.IS_DIRECTORY_ERROR;
import static org.junit.Assert.assertEquals;

public class PropertiesSecurityDomainValidatorTest {

    @Rule
    public TemporaryFolder temp = new TemporaryFolder();
    private final MockLanguageUtils langUtils = new MockLanguageUtils();
    private final PropertiesSecurityDomainValidator validator = new PropertiesSecurityDomainValidator(langUtils);
    private PropertiesFileConfig config;

    @Before
    public void setUp() throws IOException {
        config = defaultConfig();
    }

    @Test
    public void testDefaultConfigOK() {
        assertEquals(ValidationResult.ok(), validator.validate(config));
    }

    @Test
    public void testUsersFilePathPresentExistsAndNonEmpty() {
        final String expectedError = langUtils.getString(GENERIC_VALIDATION_EMPTY_FIELD, langUtils.getString(USERS_FILE_LABEL));
        config.setUsersFilePath(null);
        assertEquals(ValidationResult.error(expectedError), validator.validate(config));

    }

    @Test
    public void testGroupsFilePathPresentExistsAndNonEmpty() {
        final String expectedError = langUtils.getString(GENERIC_VALIDATION_EMPTY_FIELD, langUtils.getString(GROUPS_FILE_LABEL));
        config.setGroupsFilePath(null);
        assertEquals(ValidationResult.error(expectedError), validator.validate(config));
    }

    @Test
    public void testHashCharsetPresentAndNonEmpty() {
        final String expectedError = langUtils.getString(GENERIC_VALIDATION_EMPTY_FIELD, langUtils.getString(HASH_SET_DESCRIPTION));
        config.setHashCharset(null);
        assertEquals(ValidationResult.error(expectedError), validator.validate(config));
    }

    @Test
    public void testUserFileIsDirectory() throws Exception {
        final Path testFolder = temp.newFolder("imafolder").toPath();
        final String expectedError = langUtils.getString(IS_DIRECTORY_ERROR, testFolder.toString());
        config.setUsersFilePath(testFolder);
        assertEquals(ValidationResult.error(expectedError), validator.validate(config));
    }

    @Test
    public void testGroupFileIsDirectory() throws Exception {
        final Path testFolder = temp.newFolder("imafolder").toPath();
        final String expectedError = langUtils.getString(IS_DIRECTORY_ERROR, testFolder.toString());
        config.setGroupsFilePath(testFolder);
        assertEquals(ValidationResult.error(expectedError), validator.validate(config));
    }

    @Test
    public void testInvalidCharacterLinux() throws Exception {
        Assume.assumeFalse(SystemUtils.IS_OS_WINDOWS);
        final Path testFolder = temp.newFolder("imafolder").toPath();
        config.setGroupsFilePath(testFolder.resolve(";"));
        assertThat(validator.validate(config).getMessage())
                .startsWith("properties-domain.invalid_char");
    }

    @Test
    public void testInvalidCharacterWindows() throws Exception {
        Assume.assumeTrue(SystemUtils.IS_OS_WINDOWS);
        final Path testFolder = temp.newFolder("imafolder").toPath();
        config.setGroupsFilePath(testFolder.resolve("wrong  name"));
        assertThat(validator.validate(config).getMessage())
                .startsWith("properties-domain.invalid_char");
    }

    private PropertiesFileConfig defaultConfig() throws IOException {
        final PropertiesFileConfig config = new PropertiesFileConfig();
        config.setUsersFilePath(temp.newFile("users.properties").toPath());
        config.setGroupsFilePath(temp.newFile("groups.properties").toPath());
        config.setPlainText(true);
        config.setHashEncoding("Hex");
        config.setHashCharset("UTF-8");
        return config;
    }
}
