/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2021 Red Hat, Inc., and individual contributors
 * as indicated by the @author tags.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.jboss.installer.postinstall.server;

import static org.jboss.installer.postinstall.task.utils.ModelUtils.createEmptyOperation;
import static org.jboss.installer.postinstall.task.utils.ModelUtils.pathAddress;

import org.jboss.installer.postinstall.task.utils.ModelDescriptionConstants;
import org.jboss.dmr.ModelNode;
import org.jboss.installer.postinstall.task.utils.ModelUtils;

import java.io.FileFilter;
import java.nio.file.Path;
import java.util.List;

public class DomainServer extends AbstractEmbeddedServer {

    private static final String DOMAIN_FOLDER = "domain";
    private static final String CONFIGURATION_FOLDER = "configuration";
    private static final String HOST_CONFIG_FILE_PATTERN = "host";
    public static final List<String> PROFILES = List.of("default", "full", "full-ha", "ha");
    public static final String HOST_PRIMARY_XML = "host-primary.xml";
    public static final String HOST_SECONDARY_XML = "host-secondary.xml";

    public static final Path CONFIG_DIR = Path.of("domain", "configuration");

    public DomainServer(Path path) {
        super(path);
    }

    @Override
    protected String getStartupCommand() {
        return "embed-host-controller --jboss-home=\"%s\" --host-config=%s";
    }

    @Override
    protected String getShutdownCommand() {
        return "stop-embedded-host-controller";
    }

    @Override
    protected FileFilter getConfigFileFilter() {
        return file -> {
            final String fileName = file.getName();
            return file.isFile() && fileName.startsWith(HOST_CONFIG_FILE_PATTERN);
        };
    }

    @Override
    protected Path getConfigDirectory() {
        return path.resolve(DOMAIN_FOLDER).resolve(CONFIGURATION_FOLDER);
    }

    @Override
    public boolean isDomain() {
        return true;
    }

    @Override
    public ServerPath toRelativePathNode(Path absolutePath) {
        Path configFolder = getConfigDirectory();
        if (absolutePath.toAbsolutePath().startsWith(configFolder.toAbsolutePath())) {
            final String path = configFolder.relativize(absolutePath).toString();
            final String relativeTo = "jboss.domain.config.dir";
            return new ServerPath(relativeTo, path);
        } else {
            return new ServerPath(absolutePath.toAbsolutePath().toString());
        }
    }

    /**
     * Executes the operation on each of supported profiles. The {@code op} will be prefixed with {@code /profile="PROFILE"}
     * and executed multiple times.
     *
     * If the results of operation are needed, use {@link DomainServer#execute(ModelNode, String)} instead.
     *
     * @param op
     * @param desc
     * @throws ServerOperationException - if any of the operations fail
     */
    public void executeOnProfiles(ModelNode op, String desc) throws ServerOperationException {
        for (String profile : PROFILES) {
            final ModelNode profileOp = ModelUtils.prependToOperationAddress(op, "profile", profile);

            execute(profileOp, desc);
        }
    }

    public String readHostGroupName() throws ServerOperationException {
        final ModelNode readHostName = createEmptyOperation("read-children-names");
        readHostName.get("child-type").set("host");
        final ModelNode res = execute(readHostName, "Read host name");
        if (res.asList().size() > 1) {
            throw new IllegalStateException("Too many hosts");
        }
        return res.asList().get(0).asString();
    }

    @Override
    public boolean aliasExists(String alias, String storeName) throws ServerOperationException {
        final ModelNode queryAliasOp = createEmptyOperation("read-aliases",
                pathAddress("subsystem", "elytron").add("credential-store", storeName));

        final List<ModelNode> aliases = this.execute(wrapInHost(queryAliasOp), "Query aliases").asList();
        for (ModelNode modelNode : aliases) {
            if (alias.matches(modelNode.asString())) {
                return true;
            }
        }

        return false;
    }

    @Override
    public ModelNode encryptPassword(String alias, String storeName, String keystorePassword) throws ServerOperationException {
        final ModelNode encryptOp = createEmptyOperation("add-alias",
                pathAddress("subsystem", "elytron").add("credential-store", storeName));
        encryptOp.get("alias").set(alias);
        encryptOp.get("secret-value").set(keystorePassword);
        encryptOp.get(ModelDescriptionConstants.OPERATION_HEADERS).get("allow-resource-service-restart").set(true);
        return this.execute(wrapInHost(encryptOp), "Encrypt password");
    }

    public ModelNode wrapInHost(ModelNode op) throws ServerOperationException {
        final String hostGroup = readHostGroupName();
        return ModelUtils.prependToOperationAddress(op, "host", hostGroup);
    }

    @Override
    public List<Path> getTemporaryPaths() {
        final Path domainDir = path.resolve("domain");
        return List.of(domainDir.resolve("configuration").resolve("domain_xml_history"),
                domainDir.resolve("configuration").resolve("host_xml_history"),
                domainDir.resolve("data"),
                domainDir.resolve("log"),
                domainDir.resolve("servers"));
    }
}
