/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * <p>
 * http://www.apache.org/licenses/LICENSE-2.0
 * <p>
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.jboss.installer.validators;

import org.jboss.installer.core.LanguageUtils;
import org.jboss.installer.core.ValidationResult;
import org.jboss.installer.postinstall.ldap.LDAPUtils;
import org.jboss.installer.postinstall.task.LDAPManagementAuthConfig;

import javax.naming.directory.DirContext;

public class SearchDnValidator {

    public static final String LDAP_REALM_SCREEN_TEST_SEARCH_ERROR = "ldap_realm_screen.test_search.error";
    private LanguageUtils langUtils;

    public SearchDnValidator(LanguageUtils langUtils) {
        this.langUtils = langUtils;
    }

    public ValidationResult validateData(LDAPManagementAuthConfig config) {
        String ldapServerDn = config.getDn();
        String ldapServerUrl = config.getUrl();
        String ldapServerPassword = config.getPassword();
        String searchDn = config.getBaseDN();
        String searchFilter = config.getFilter();
        LDAPManagementAuthConfig.FilterType filterType = config.getFilterType();
        boolean isRecursive = config.isRecursive();

        DirContext ctx = LDAPUtils.makeConnection(ldapServerUrl, ldapServerDn, ldapServerPassword);
        if (ctx == null) {
            return ValidationResult.warning(langUtils.getString(LDAPValidator.LDAP_SCREEN_CONNECTION_ERROR));
        }

        if (!LDAPUtils.validateBaseDn(ctx, searchDn, buildFilter(searchFilter, filterType), isRecursive)) {
            return ValidationResult.warning(langUtils.getString(LDAP_REALM_SCREEN_TEST_SEARCH_ERROR));
        }
        return ValidationResult.ok();
    }

    private String buildFilter(String searchFilter, LDAPManagementAuthConfig.FilterType filterType) {
        return  (filterType == LDAPManagementAuthConfig.FilterType.USERNAME) ? "("+searchFilter+"=*)" : searchFilter;
    }
}
