/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors. 
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.picketlink.identity.federation.core.saml.v2.metadata.store;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.StringTokenizer;

import javax.xml.bind.JAXBElement;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Marshaller;
import javax.xml.bind.Unmarshaller;

import org.apache.log4j.Logger;
import org.picketlink.identity.federation.core.constants.PicketLinkFederationConstants;
import org.picketlink.identity.federation.core.util.JAXBUtil;
import org.picketlink.identity.federation.saml.v2.metadata.EntityDescriptorType;
import org.picketlink.identity.federation.saml.v2.metadata.IDPSSODescriptorType;
import org.picketlink.identity.federation.saml.v2.metadata.ObjectFactory;
import org.picketlink.identity.federation.saml.v2.metadata.RoleDescriptorType;
import org.picketlink.identity.federation.saml.v2.metadata.SPSSODescriptorType;

/**
 * File based metadata store that uses
 * the ${user.home}/jbid-store location to
 * persist the data
 * @author Anil.Saldhana@redhat.com
 * @since Apr 27, 2009
 */
public class FileBasedMetadataConfigurationStore implements IMetadataConfigurationStore
{
   private static Logger log = Logger.getLogger(FileBasedMetadataConfigurationStore.class);
   private boolean trace = log.isTraceEnabled(); 
   
   private String userHome = null;
   
   private String baseDirectory = null;
   
   private String pkgName = "org.picketlink.identity.federation.saml.v2.metadata"; 
   
   public FileBasedMetadataConfigurationStore()
   {
      bootstrap();
   }
   
   /**
    * @see {@code IMetadataConfigurationStore#bootstrap()}
    */
   public void bootstrap()
   {  
      userHome = SecurityActions.getSystemProperty("user.home");
      if(userHome == null)
         throw new RuntimeException("user.home system property not set");
      
      StringBuilder builder = new StringBuilder( userHome );
      builder.append( PicketLinkFederationConstants.FILE_STORE_DIRECTORY );
      baseDirectory = builder.toString();
      
      File plStore = new File( baseDirectory );
      if(plStore.exists() == false)
      {
         if(trace)
            log.trace(plStore.getPath() + " does not exist. Hence creating.");
         plStore.mkdir();
      }
   }
   
   /**
    * @see IMetadataConfigurationStore#getIdentityProviderID()
    */
   public Set<String> getIdentityProviderID()
   { 
      Set<String> identityProviders = new HashSet<String>();
      
      Properties idp = new Properties();
      
      StringBuilder builder = new StringBuilder( baseDirectory );
      builder.append( PicketLinkFederationConstants.IDP_PROPERTIES );
      
      File identityProviderFile = new File( builder.toString() );
      if( identityProviderFile.exists() )
      {
         try
         {
            idp.load( new FileInputStream( identityProviderFile  ));
            String listOfIDP = (String) idp.get("IDP");
            
            //Comma separated list
            StringTokenizer st = new StringTokenizer( listOfIDP, ",");
            while( st.hasMoreTokens() )
            {
               String token = st.nextToken();
               identityProviders.add( token );
            }
         }
         catch (Exception e)
         {
            log.error( "Exception loading the identity providers:", e );
         } 
      } 
      return identityProviders;
   }
   
   /**
    * @see IMetadataConfigurationStore#getServiceProviderID()
    */
   public Set<String> getServiceProviderID()
   { 
      Set<String> serviceProviders = new HashSet<String>();
      
      Properties sp = new Properties();  
      StringBuilder builder = new StringBuilder( baseDirectory );
      builder.append( PicketLinkFederationConstants.SP_PROPERTIES );
      
      File serviceProviderFile = new File( builder.toString() ); 
       
      if( serviceProviderFile.exists() )
      {
         try
         {
            sp.load( new FileInputStream( serviceProviderFile  ));
            String listOfSP = (String) sp.get("SP");
            
            //Comma separated list
            StringTokenizer st = new StringTokenizer( listOfSP, "," );
            while( st.hasMoreTokens() )
            {
               String token = st.nextToken();
               serviceProviders.add( token );
            }
         }
         catch (Exception e)
         {
            log.error( "Exception loading the service providers:", e );
         } 
      } 
      return serviceProviders;
   }
   
   /** 
    * @see IMetadataConfigurationStore#load(String)
    */
   @SuppressWarnings("unchecked")
   public EntityDescriptorType load(String id) throws IOException
   {
      File persistedFile = validateIdAndReturnMDFile(id);
      
      Unmarshaller un;
      try
      {
         un = JAXBUtil.getUnmarshaller(pkgName);
         JAXBElement<EntityDescriptorType> je = 
            (JAXBElement<EntityDescriptorType>) un.unmarshal(persistedFile);
        return je.getValue();
      }
      catch (JAXBException e)
      {
         IOException ioe =new IOException(e.getLocalizedMessage());
         ioe.initCause(e);
         throw ioe;
      }
      
   }

   /**  
    * @see IMetadataConfigurationStore#persist(EntityDescriptorType, String)
    */
   public void persist(EntityDescriptorType entity, String id) throws IOException
   {
      boolean isIDP = false;
      boolean isSP = false;
      
      File persistedFile = validateIdAndReturnMDFile(id);
      
      ObjectFactory of = new ObjectFactory();
      
      JAXBElement<?> jentity = of.createEntityDescriptor(entity);
      
      Marshaller m;
      try
      {
         m = JAXBUtil.getMarshaller(pkgName);
         m.marshal(jentity, persistedFile);
      }
      catch (JAXBException e)
      {
         IOException ioe =new IOException(e.getLocalizedMessage());
         ioe.initCause(e);
         throw ioe;
      } 
      if(trace) log.trace("Persisted into " + persistedFile.getPath());
      
      //We need to figure out whether this is sp or idp from the entity data
      List<RoleDescriptorType> roleDescriptorTypes = entity.getRoleDescriptorOrIDPSSODescriptorOrSPSSODescriptor();
      for( RoleDescriptorType rdt: roleDescriptorTypes )
      {
         if( rdt instanceof IDPSSODescriptorType )
         {
            isIDP = true;
            break;
         }
         if( rdt instanceof SPSSODescriptorType )
         {
            isSP = true;
            break;
         }
      }
      
      if( isSP )
      {
         addServiceProvider(id); 
      }
      else if( isIDP )
      {
         addIdentityProvider( id);
      }
         
   }

   /**
    * @see IMetadataConfigurationStore#delete(String)
    */
   public void delete(String id) 
   {
      File persistedFile = validateIdAndReturnMDFile(id);
      
      if(persistedFile.exists())
         persistedFile.delete(); 
   }

   /**
    * @throws IOException  
    * @throws ClassNotFoundException 
    * @see IMetadataConfigurationStore#loadTrustedProviders(String)
    */
   @SuppressWarnings("unchecked")
   public Map<String, String> loadTrustedProviders(String id) throws IOException, ClassNotFoundException 
   {
      File trustedFile = validateIdAndReturnTrustedProvidersFile(id);
      ObjectInputStream ois = null;
      try
      {
         ois = new ObjectInputStream(new FileInputStream(trustedFile));
         Map<String, String> trustedMap = (Map<String, String>) ois.readObject();
         return trustedMap; 
      }
      finally
      {
         if(ois != null)
            ois.close();
      } 
   }

   /**
    * @throws IOException   
    * @see IMetadataConfigurationStore#persistTrustedProviders(Map)
    */
   public void persistTrustedProviders(String id, Map<String, String> trusted) 
   throws IOException 
   {  
      File trustedFile = validateIdAndReturnTrustedProvidersFile(id);
      ObjectOutputStream oos = null;

      try
      {
         oos = new ObjectOutputStream(new FileOutputStream(trustedFile));
         oos.writeObject(trusted); 
      }
      finally
      {
         if(oos != null)
            oos.close();
      }
      if(trace) log.trace("Persisted trusted map into "+ trustedFile.getPath());
   }
   
   /**
    * @see IMetadataConfigurationStore#deleteTrustedProviders(String)
    */
   public void deleteTrustedProviders(String id) 
   {
      File persistedFile = validateIdAndReturnTrustedProvidersFile(id);
      
      if(persistedFile.exists())
         persistedFile.delete();  
   }
   
   private File validateIdAndReturnMDFile(String id)
   {
      String serializationExtension = PicketLinkFederationConstants.SERIALIZATION_EXTENSION;
      
      if(id == null)
         throw new IllegalArgumentException("id is null");
      if( !id.endsWith( serializationExtension ))
         id += serializationExtension;
      
      StringBuilder builder = new StringBuilder( baseDirectory );
      builder.append( "/").append( id );
      
      return new File( builder.toString() ); 
   }
   
   private File validateIdAndReturnTrustedProvidersFile(String id)
   {
      if(id == null)
         throw new IllegalArgumentException("id is null");
      
      id += "-trusted" + PicketLinkFederationConstants.SERIALIZATION_EXTENSION; 

      StringBuilder builder = new StringBuilder( baseDirectory );
      builder.append( "/").append( id );
      
      return new File( builder.toString() ); 
   }
   
   private void addServiceProvider( String id )
   {  
      Properties sp = new Properties(); 

      StringBuilder builder = new StringBuilder( baseDirectory );
      builder.append( PicketLinkFederationConstants.SP_PROPERTIES );
      
      File serviceProviderFile = new File( builder.toString() ); 
       
      try
      {
         if( serviceProviderFile.exists() == false )
            serviceProviderFile.createNewFile();
         
            sp.load( new FileInputStream( serviceProviderFile  ));
            String listOfSP = (String) sp.get("SP");
            if( listOfSP == null )
            {
               listOfSP = id; 
            }
            else
            {
               listOfSP += "," + id; 
            } 
            sp.put( "SP", listOfSP );
            
            sp.store( new FileWriter( serviceProviderFile ), ""); 
      }
      catch (Exception e)
      {
          log.error( "Exception loading the service providers:", e );
      }   
   }
   
   private void addIdentityProvider( String id )
   {  
      Properties idp = new Properties(); 

      StringBuilder builder = new StringBuilder( baseDirectory );
      builder.append( PicketLinkFederationConstants.IDP_PROPERTIES );
      
      File idpProviderFile = new File( builder.toString() ); 
       
      try
      {
         if( idpProviderFile.exists() == false )
            idpProviderFile.createNewFile();
         
            idp.load( new FileInputStream( idpProviderFile  ));
            String listOfIDP = (String) idp.get("IDP");
            if( listOfIDP == null )
            {
               listOfIDP = id; 
            }
            else
            {
               listOfIDP += "," + id; 
            } 
            idp.put( "IDP", listOfIDP );
            
            idp.store( new FileWriter( idpProviderFile ), ""); 
      }
      catch (Exception e)
      {
          log.error( "Exception loading the identity providers:", e );
      }   
   }


   /**
    * @see {@code IMetadataConfigurationStore#cleanup()}
    */
   public void cleanup()
   { 
   } 
}