/*
* JBoss, Home of Professional Open Source
* Copyright 2005, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.web.tomcat.service.session.distributedcache.spi;

import java.io.Serializable;

import org.jboss.ha.framework.interfaces.ObjectStreamSource;
import org.jboss.ha.framework.server.MarshalledValueObjectStreamSource;
import org.jboss.ha.framework.server.SimpleCachableMarshalledValue;
import org.jboss.logging.Logger;

/**
 * Factory class for creating object output and input streams, 
 * switching between JDK Serialization and JBoss Serialization classes. 
 * Using MarshalledValue to replace Serializable used inside different 
 * web app class loader context. Assuming that the caller classes are already
 * handling the switch between web app class loader context. 
 *  
 * 
 * @author <a href="hmesha@novell.com">Hany Mesha</a>
 * @author Brian Stansberry
 */
public class SessionSerializationFactory
{
   /** 
    * System property that, if set to <code>true</code>, indicates that 
    * JBoss Serialization should be used for session serialization.
    */
   public static final String JBOSS_SERIALIZATION_SYS_PROP = "session.serialization.jboss";
   private static Logger log_ = Logger.getLogger(SessionSerializationFactory.class);
   private static boolean useJBossSerialization = false;
   private static ObjectStreamSource objectStreamSource; 

   static
   {
       String useJBossSerializationStr = System.getProperty(JBOSS_SERIALIZATION_SYS_PROP, "false");
       useJBossSerialization = Boolean.valueOf(useJBossSerializationStr).booleanValue();
       objectStreamSource = useJBossSerialization ? new JBossSerializationObjectStreamSource()
                                                  : new MarshalledValueObjectStreamSource(); 
       try
       {
          if (useJBossSerialization)
          {
             log_.debug("Using JBossSerialization for web session replication");
          }
       }
       catch (Throwable ignored) 
       {
          // we don't fail a static initializer due to a debug log stmt
       }
   }
   
   public static SimpleCachableMarshalledValue createMarshalledValue(Serializable o)
   {
      return new SimpleCachableMarshalledValue(o, objectStreamSource);
   }
   
   public static ObjectStreamSource getObjectStreamSource()
   {
      return objectStreamSource;
   }

   public static boolean useJBossSerialization()
   {
       return useJBossSerialization;
   }

}
