/*
 * JBoss, Home of Professional Open Source
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.metatype.plugins.values.mappers;

import java.lang.reflect.Type;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Hashtable;
import java.util.Enumeration;

import javax.management.ObjectName;
import javax.management.MalformedObjectNameException;

import org.jboss.metatype.api.types.MetaType;
import org.jboss.metatype.api.types.PropertiesMetaType;
import org.jboss.metatype.api.values.MetaValue;
import org.jboss.metatype.api.values.PropertiesMetaValue;
import org.jboss.metatype.spi.values.MetaMapper;

/**
 * An ObjectName MetaMapper that maps to a PropertiesMetaType/PropertiesMetaValue.
 *
 * @author Scott.Stark@jboss.org
 * @author Ales.Justin@jboss.org
 * @version $Revision: 87758 $
 */
public class PropertiesObjectNameMetaMapper extends MetaMapper<ObjectName>
{
   PropertiesMetaType type;

   public PropertiesObjectNameMetaMapper()
   {
      this((String[])null);
   }
   public PropertiesObjectNameMetaMapper(String... args)
   {
      Map<String, String> propertyNameDescriptions = Collections.emptyMap();
      if(args != null)
      {
         if(((args.length %2) == 0) == false)
            throw new IllegalStateException("args counts must be 2*n for n key,key-description pairs");
         propertyNameDescriptions = new HashMap<String,String>();
         propertyNameDescriptions.put("domain", "the object name domain");
         for(int n = 0; n < args.length; n += 2)
         {
            String key = args[n];
            String description = args[n+1];
            propertyNameDescriptions.put(key, description);
         }
      }

      type = new PropertiesMetaType("java.util.Properties", propertyNameDescriptions);
   }

   @Override
   public MetaType getMetaType()
   {
      return type;
   }

   @Override
   public Type mapToType()
   {
      return ObjectName.class;
   }

   @Override
   public MetaValue createMetaValue(MetaType metaType, ObjectName object)
   {
      String domain = object.getDomain();
      Map<String, String> props = object.getKeyPropertyList();
      PropertiesMetaValue map = PropertiesMetaValue.wrap(props);
      map.put("domain", domain);
      return map;
   }

   @Override
   public ObjectName unwrapMetaValue(MetaValue metaValue)
   {
      PropertiesMetaValue map = (PropertiesMetaValue) metaValue;
      String domain = (String) map.remove("domain");
      try
      {
         return createObjectName(domain, map);
      }
      catch(Exception e)
      {
         throw new IllegalArgumentException("Failed to build ObjectName from MetaValue: "+metaValue, e);
      }
   }

   /**
    * Create object name from PropertiesMetaValue.
    * Make it JDK 1.6 compatible.
    *
    * @param domain the domain name
    * @param props the props
    * @return new ObjectName instance
    * @throws javax.management.MalformedObjectNameException for any error
    */
   public static ObjectName createObjectName(String domain, PropertiesMetaValue props) throws MalformedObjectNameException
   {
      if (props == null)
         return new ObjectName(domain);

      Hashtable<String, String> table = new Hashtable<String, String>();
      Enumeration<?> enumeration = props.propertyNames();
      while (enumeration.hasMoreElements())
      {
         Object key = enumeration.nextElement();
         if (key instanceof String)
         {
            String stringKey = (String)key;
            String property = props.getProperty(stringKey);
            table.put(stringKey, property);
         }
      }
      return new ObjectName(domain, table);
   }
}
