/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.mx.metadata;

import java.util.Arrays;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.Set;

import javax.management.DynamicMBean;
import javax.management.MXBean;
import javax.management.NotCompliantMBeanException;

/**
 * Holds the type of an MBean class.
 *
 * The introspection algorithm used is the following:
 *
 * 1. If MyClass is an instance of the DynamicMBean interface, then the return value of its getMBeanInfo method will
 *    list the attributes and operations of the resource. In other words, MyClass is a dynamic MBean.
 *
 * 2. If the MyClass MBean is an instance of a MyClassMBean interface, then only the methods listed in, or inherited by,
 *    the interface are considered among all the methods of, or inherited by, the MBean. The design patterns are then used to
 *    identify the attributes and operations from the method names in the MyClassMBean interface and its ancestors.
 *    In other words, MyClass is a standard MBean.
 *
 * 3. If MyClass is an instance of the DynamicMBean interface, then MyClassMBean is ignored.
 *    If MyClassMBean is not a public interface, it is not a JMX manageable resource.
 *    If the MBean is an instance of neither MyClassMBean nor DynamicMBean, the inheritance tree of MyClass is examined,
 *    looking for the nearest superclass that implements its own MBean interface.
 *
 *    a. If there is an ancestor called SuperClass that is an instance of SuperClassMBean, the design patterns
 *       are used to derive the attributes and operations from SuperClassMBean. In this case, the MBean MyClass then
 *       has the same management interface as the MBean SuperClass. If SuperClassMBean is not a public interface,
 *       it is not a JMX manageable resource.
 *
 *    b. When there is no superclass with its own MBean interface, MyClass is not a JMX manageable resource.
 *
 * @author  <a href="mailto:trevor@protocool.com">Trevor Squires</a>.
 * @author  thomas.diesler@jboss.org
 */
public class MBeanCapability
{
   public static final int DYNAMIC_MBEAN = 0x321;
   public static final int STANDARD_MBEAN = 0x123;
   public static final int NOT_AN_MBEAN = 0xc0de;

   protected int mbeanType = NOT_AN_MBEAN;

   private MBeanCapability(int type)
   {
      mbeanType = type;
   }

   public int getMBeanType()
   {
      return mbeanType;
   }

   public static MBeanCapability of(Class mbeanClass) throws NotCompliantMBeanException
   {
       MBeanCapability cap = ofMBean(mbeanClass);
       
       //if null, check for MXBean, if it there is, return it, else, throw exception
       if(cap == null){
           //check for MXBeans
           cap = ofMXBean(mbeanClass);
           if(cap != null){
               return cap;
           } 
       } else {
           return cap;
       }
       throw new NotCompliantMBeanException("Class does not expose a management interface: " + mbeanClass.getName());
   }
   
   private static MBeanCapability ofMBean(Class mbeanClass)
   {
      if (null == mbeanClass)
      {
         throw new IllegalArgumentException("MBean class cannot be null");
      }

      // If MyClass is an instance of the DynamicMBean interface, MyClass is a dynamic MBean.
      if (DynamicMBean.class.isAssignableFrom(mbeanClass))
      {
         return new MBeanCapability(DYNAMIC_MBEAN);
      }

      // If the MyClass MBean is an instance of a MyClassMBean interface, MyClass is a standard MBean
      Class [] interfaces = mbeanClass.getInterfaces();
      for (int i = 0; i < interfaces.length; i++)
      {
         Class anInterface = interfaces[i];
         if (anInterface.getName().equals(mbeanClass.getName() + "MBean"))
         {
            return new MBeanCapability(STANDARD_MBEAN);
         }
      }

      // If there is an ancestor called SuperClass that is an instance of SuperClassMBean
      Class superClass = mbeanClass.getSuperclass();
      if (superClass != null)
         return ofMBean(superClass);

      return null;
   }
   
    @SuppressWarnings("rawtypes")
    private static MBeanCapability ofMXBean(final Class mbeanClass) throws NotCompliantMBeanException {
        // this is a bit different, than ofMBean ( I hope ). We need to dive into interfaces.
        if (null == mbeanClass) {
            throw new IllegalArgumentException("MBean class cannot be null");
        }
        Class workOnMe = mbeanClass;
        // gather all interfaces(top defined - this could be done with recursion.
        LinkedList<Class> ifaces = new LinkedList<Class>();
        while (true) {
            Class[] declaredIfaces = workOnMe.getInterfaces();
            ifaces.addAll(Arrays.asList(declaredIfaces));
            workOnMe = workOnMe.getSuperclass();
            if (workOnMe == null || workOnMe.equals(Object.class))
                break;
        }
        // set, to avoid duplicates
        final Set<Class> score = new HashSet<Class>();
        for (Class iface : ifaces) {
            ofMXBeanInterface(mbeanClass,iface, score);
        }
        switch (score.size()) {
            case 0:
                return null;
            case 1:
                // Note ... in perfect world we would return: MX_BEAN defined above..( if it has been defined )
                // This value will use AS5 mx.mxbean package and classes implemented there.
                return new MBeanCapability(DYNAMIC_MBEAN);
            default:
                throw new NotCompliantMBeanException("Class '" + mbeanClass.getName()
                        + "' exposes more than one MXBean interface '" + score + "'");

        }

    }

    @SuppressWarnings("rawtypes")
    private static void ofMXBeanInterface(final Class baseClass,final Class mbeanInterface,final Set<Class> score) throws NotCompliantMBeanException {
        //this is a bit messy, we have two annotations...
        boolean failedWithaAnnotation=false;
        //we actually should mess with this?
        org.jboss.mx.mxbean.MXBean jbossAnnotation = (org.jboss.mx.mxbean.MXBean) mbeanInterface
                .getAnnotation(org.jboss.mx.mxbean.MXBean.class);        
        // we need to check for:
        // - @MXBean || @MXBean(true)
        // - zxcMXBean interface
        MXBean jdkAnnodation = (MXBean) mbeanInterface.getAnnotation(MXBean.class);
        if(jdkAnnodation!=null && jbossAnnotation!=null){
            //thats just wrong
            throw new NotCompliantMBeanException("Class '" + baseClass.getName()
                    + "' has interface '" + mbeanInterface.getName() + "' which exposes more than on MXBean annotation!");
        }
        
        if(jbossAnnotation!=null){
            if(jbossAnnotation.value()){
                score.add(mbeanInterface);
            }else{
                failedWithaAnnotation = true;
            }
        }
        
        if(jdkAnnodation!=null){
            if(jdkAnnodation.value()){
                score.add(mbeanInterface);
            }else{
                failedWithaAnnotation = true;
            }
        }
        //consider name only in case annotation dont deny
        if(!failedWithaAnnotation)
       {
            // check interface name
            final String ifaceName = mbeanInterface.getName();
            if (ifaceName.endsWith("MXBean") && ifaceName.lastIndexOf("MXBean") > 0) {
                score.add(mbeanInterface);
            }

        }
        // now we need to check super interfaces
        // NOTE: there is no word on double def -
        // @MXBean public interface XX{}
        // @MXBean public interface YY extends XX{} ...
        // so we assume thats a no no. Same for class names?
        Class[] ifaces = mbeanInterface.getInterfaces();
        if (ifaces != null) // just in case some JVMs dont behave here
            for (Class i : ifaces) {
                ofMXBeanInterface(baseClass,i, score);
            }
    }
}