package org.hibernate.plugins;

import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.hibernate.plugins.util.Util;

import java.io.*;
import java.util.*;

/**
 * Consolidates SQL files from a directory into one
 *
 * @goal schemaConsolidate
 * @phase process-classes
 * @requiresDependencyResolution compile
 */
public class SchemaConsolidate extends AbstractMojo {

    /**
     * Set input directory
     *
     * @parameter
     * @required
     * @readonly
     */
    private String inputDir;

    /**
     * Set output directory
     *
     * @parameter
     * @required
     * @readonly
     */
    private String outputDir;

    /**
     * Set to false if you don't want to preserve drop statements (including "alter table drop").
     *
     * @parameter default-value="true"
     */
    private boolean preserveDrops;

    public void execute() throws MojoExecutionException, MojoFailureException {
        Set<File> sqlFiles = Util.getSqlFiles(inputDir);
        Map<String, Set<File>> groupedSqlFiles = Util.groupSqlFilesByDialects(Util.DIALECTS, sqlFiles);
        Map<String, String> groupedConsolidatedSqls = new HashMap<String, String>();

        for (String dialect : groupedSqlFiles.keySet()) {
            String consolidatedSqls = consolidateSqlFiles(groupedSqlFiles.get(dialect), this.preserveDrops);
            groupedConsolidatedSqls.put(dialect, consolidatedSqls);
        }

        for (String dialect : groupedConsolidatedSqls.keySet()) {
            String outputFile = outputDir + "/" + dialect + ".sql";
            Util.writeFile(new File(outputFile), groupedConsolidatedSqls.get(dialect));
        }
    }

    private String consolidateSqlFiles(Set<File> sqlFiles, boolean preserveDrops)
            throws MojoExecutionException, MojoFailureException {
        Set<String> uniqueSqls = new HashSet<String>();
        List<String> resultSqls = new ArrayList<String>();
        Set<String> objectNames = new HashSet<String>();

        for (File sqlFile : sqlFiles) {
            List<String> sqlStatements = Util.loadSqlStatementsFromFile(sqlFile);

            for (String sqlStatement : sqlStatements) {
                if (preserveDrops || !Util.statementContains(sqlStatement, "drop")) {
                    String canonicalSql = createCanonicalSql(sqlStatement);
                    if (!uniqueSqls.contains(canonicalSql)) {
                        uniqueSqls.add(canonicalSql);
                        resultSqls.add(sqlStatement);
                        String objectName = Util.getObjectName(sqlStatement, "create\\s+table\\s+(\\w+)");
                        if (objectName != null) {
                            if (objectNames.contains(objectName)) {
                                throw new MojoFailureException(
                                        "There are two DDL statements with the same name but different mappings. " +
                                        "The name of the table is '" + objectName + "'.");
                            }
                            objectNames.add(objectName);
                        }
                    }
                }
            }
        }

        String consolidatedSqls = Util.join(resultSqls, Util.SQL_SEPARATOR);
        return consolidatedSqls.toString();
    }

    private String createCanonicalSql(String sql) {
        String result = sql;
        result = result.trim();
        if (!Util.statementContains(sql, "drop")) {
            result = result.replaceAll("FK[A-F0-9]+", "FK");
        }
        return result;
    }
}
