package com.redhat.installer.validator;
import java.util.Properties;

import com.izforge.izpack.installer.AutomatedInstallData;
import com.izforge.izpack.installer.DataValidator;
import com.redhat.installer.util.KeystoreUtils;

/**
 *  Validates the security domain JSSE element panel.
 *  If user selects to add a JSSE element, then this validator will 
 *  check that either situation is true:
 *  1) securitydomain.keystore.password != null && securitydomain.keystore.url != null
 *      OR
 *  2) securitydomain.truststore.password != null && securitydomain.truststore.url != null
 *        If both of these are false, the user is prompted to change their input.
 * @author fcanas
 */
public class SecurityDomainValidator implements DataValidator
{
    String error;
    
    public Status validateData(AutomatedInstallData idata) {
        Properties variables = idata.getVariables();
        String isKeyStore = (String) variables.get("securityDomainJsseAddKeystore");
        String isTrustStore = (String) variables.get("securityDomainJsseAddTruststore");
        String keyPwd = (String)variables.get("securitydomain.jsse.keystore.password");
        String keyUrl = (String)variables.get("securitydomain.jsse.keystore.url");
        String storePwd = (String)variables.get("securitydomain.jsse.truststore.password");
        String storeUrl = (String)variables.get("securitydomain.jsse.truststore.url");
        
        if (idata.getVariable("securityDomainAddJsse").contains("false")) {
            return Status.OK;
        }
        
        // If user selected the keystore, validate their keystore input.
        if (isKeyStore.contains("true")) {
        	if (nullOrEmpty(keyPwd, keyUrl)){
        		error = idata.langpack.getString("securitydomain.jsse.keystore.reqs");
        		return Status.ERROR;
        	}
        	int result = KeystoreUtils.isValidKeystore(keyUrl, keyPwd.toCharArray()); 
            if (result == 0) {
                return Status.OK;
            } else if (result == 1) {
            	error = idata.langpack.getString("securitydomain.jsse.keystore.passincorrect");
	            return Status.WARNING;
            } else if (result == 2) {
            	error = idata.langpack.getString("securitydomain.jsse.keystore.inaccessible");
	            return Status.WARNING;
            }
        }
        // If user selected the truststore, validate their truststore input.
        if (isTrustStore.contains("true")) {
        	if (nullOrEmpty(storePwd, storeUrl)){
                error = idata.langpack.getString("securitydomain.jsse.truststore.reqs");
                return Status.ERROR;	
        	}
        	int result = KeystoreUtils.isValidKeystore(storeUrl, storePwd.toCharArray()); 
            if (result == 0) {
                return Status.OK;
            } else if (result == 1) {
            	error = idata.langpack.getString("securitydomain.jsse.truststore.passincorrect");
            	return Status.WARNING;
            } else if (result == 2){
            	error = idata.langpack.getString("securitydomain.jsse.truststore.inaccessible");
            	return Status.WARNING;           	
            }
        }
        
        if (!(isTrustStore.contains("true") || isKeyStore.contains("true")))
        {
	        error = idata.langpack.getString("securitydomain.jsse.requirements");
	        return Status.ERROR;
        }
        
        return Status.OK;
    }
    
    /**
     * Returns true if any string in the parameter list is either null or empty
     * @param values
     * @return
     */
    private boolean nullOrEmpty(String ... values)
    {
    	boolean anyFailure = false;
    	for (String value : values){
    		if (value == null || value.isEmpty()){
    			anyFailure = true;
    		}
    	}
        return anyFailure;
    }

    public String getErrorMessageId() 
    {
        return error;
    }

    public String getWarningMessageId() 
    {
        return error;
    }

    public boolean getDefaultAnswer() 
    {
        return false;
    }

}

