package org.jboss.ip.dbtool;

import org.jboss.as.cli.CommandLineException;
import org.jboss.ip.dbtool.cliutil.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.ConnectException;
import java.net.URISyntaxException;
import java.util.*;
import java.util.jar.JarFile;


/**
 * Main script of the tool.
 *
 * @author Alex Creasy <acreasy@redhat.com>
 */
public final class Main {

    // Hard coded default values
    public static final int DEFAULT_MIN_POOL_SIZE = 15;
    public static final int DEFAULT_MAX_POOL_SIZE = 50;
    public static final String DEFAULT_SERVER_HOSTNAME = "localhost";
    public static final int DEFAULT_NATIVE_INTERFACE_PORT = 9999;

    public static final String LOGGER_NAME = "org.jboss.ip.dbtool";
    public static final String TOOL_DATABASE_DIR_NAME = "databases";
    public static final String AUTORUN_PROPERTIES_FILENAME = "autorun.properties";

    public static final int MAX_ATTEMPTS = 10;
    public static final int TIME_TO_SLEEP = 2;

    public static final String ORACLE_RAC_ID = "oraclerac";

    private static final Logger logger = LoggerFactory.getLogger(LOGGER_NAME);

    private static final Map<String, Database> dbMap = new HashMap<String, Database>();

    // Properties loaded from script file to drive datasource configuration
    private static final Properties autorun = new Properties();

    // For connecting to the JBoss CLI
    private static String jbossHostname;
    private static int jbossNativeManagementPort;
    private static String jbossNativeManagementUsername;
    private static char[] jbossNativeManagementPassword;

    // Dir's the tool needs to be aware of.
    private static File jbossHomeDir;
    private static File toolHomeDir;
    private static File toolDriverDir;
    private static JarFile toolJarFile;

    private static boolean isDomain;
    private static String[] domainProfiles;
    private static String[] defaultDatasources;


    public static void main(final String[] args) {
        ServerLauncher serverLauncher;
        CommandExecutor cli = null;
        try {
            bootstrap();
            welcome();

            /*
             * Startup the JBoss server.
             */
            serverLauncher = new ServerLauncher(jbossHomeDir);
            logger.info("Starting JBoss server in dir '{}'", jbossHomeDir);
            try {
                serverLauncher.startServer(isDomain);
            } catch (ConnectException e) {
                throw new FatalException("Unable to start JBoss server at: " + jbossHomeDir, e);
            }


            cli = new CommandExecutor(jbossNativeManagementUsername, jbossNativeManagementPassword, jbossHostname,
                    jbossNativeManagementPort);

            logger.info("Connecting to Cli using {}", cli);
            int attempt = 1;
            boolean connected = false;
            while (!connected) {
                try {
                    cli.connect();
                    connected = true;
                    logger.info("Attempt number {} successful.", attempt);
                } catch (CommandLineException cle) {
                    logger.info("Attempt number {} failed, trying again...", attempt);
                    if (attempt >= MAX_ATTEMPTS) {
                        logger.error("Maximum number of attempts exceeded.");
                        throw new FatalException(String.format("Unable to connect to JBoss native CLI at: %s:%d with " +
                                "username=%s and supplied password.", jbossHostname, jbossNativeManagementPort,
                                jbossNativeManagementUsername));
                    }
                    // Wait for a few seconds and try again
                    Thread.sleep(TIME_TO_SLEEP * 1000);
                    attempt++;
                }
            }

            // Attempt to backup the user's existing configuration
            try {
                cli.backupConfig();
                System.out.println("Sucessfully backed up existing configuration");
                logger.info("Sucessfully backed up existing configuration");
            } catch (CommandLineException cliEx) {
                throw new FatalException("Unable to backup existing configuration");
            }

            /*
             * Get the database the user wishes to switch to and install / register the
             * JDBC driver with JBoss.
             */
            final Database db = dbMap.get(getProperty("datasource.product").trim());

            if (db == null) {
                throw new FatalException("datasource.product value: '" + getProperty("datasource.product") + "' is " +
                        "not available");
            }

            System.out.printf("Datasources will be updated to use: '%s'\n", db.getDescriptiveName());
            logger.info("Datasources will be updated to use: {}", db);

            // Find driver in tool driver dir, copy to server dir and create module.xml file.
            // Any exceptions thrown are considered fatal.
            final File drv = DriverInstaller.install(db, toolDriverDir, jbossHomeDir);

            System.out.printf("Found driver: %s", drv.getName());

            /*
             * Populate a list of datasources we need to update, and ascertain what values
             * they need to be updated with.
             */

            // Check to see if the user has overridden the default datasources to update.
            final String tmp = getProperty("datasources").trim();
            final String[] datasources = tmp.isEmpty() ? defaultDatasources : tmp.split(",");

            // Load the values to update the chosen datasources with
            final DatasourceValues.Builder dsBuilder = new DatasourceValues.Builder();

            // Build a connection URL, hard coded support for oracle RAC is provided.
            if (ORACLE_RAC_ID.equals(db.getId())) {
                // TODO: Error check input.
                final String[] hostnames = getProperty("datasource.hostsports").split(";");

                dsBuilder.setConnectionUrl(ConnectionUrlCreator.createOracleRacUrl(getProperty("datasource.dbname"),
                        hostnames));
            } else {
                dsBuilder.setConnectionUrl(ConnectionUrlCreator.createUrl(db, getProperty("datasource.dbname"),
                        getProperty("datasource.hostname"), getProperty("datasource.port")));
            }

            dsBuilder.setDriverStaleConnectionChecker(db.getDriverStaleConnectionChecker())
                    .setDriverName(db.getDriverName())
                    .setDriverExceptionSorter(db.getDriverExceptionSorter())
                    .setDriverValidConnectionChecker(db.getDriverValidConnectionChecker())
                    .setUsername(getProperty("datasource.username"))
                    .setPassword(getProperty("datasource.password").toCharArray());

            // Deal with properties that need to be parsed as ints.
            int minPoolSize;
            try {
                minPoolSize = Integer.parseInt(getProperty("datasource.min.pool.size"));
            } catch (NumberFormatException nfe) {
                minPoolSize = DEFAULT_MIN_POOL_SIZE;
            }

            int maxPoolSize;
            try {
                maxPoolSize = Integer.parseInt(getProperty("datasource.max.pool.size"));
            } catch (NumberFormatException nfe) {
                maxPoolSize = DEFAULT_MAX_POOL_SIZE;
            }

            dsBuilder.setMinPoolSize(minPoolSize).setMaxPoolSize(maxPoolSize);

            final DatasourceValues dsValues = dsBuilder.build();

            /*
             * Update the selected Datasources (for each domain profile if domain mode).
             */
            System.out.println("\n----------------------");
            System.out.println("Updating Datasources");
            System.out.println("----------------------");

            if (isDomain) {
                for (final String profile : domainProfiles) {

                    if (isDomain)
                        System.out.printf("\nUpdating profile: %s\n", profile);

                    if (isDomain)
                        System.out.print("    ");

                    System.out.print("Registering driver: ");

                    try {
                        cli.registerJdbcDriver(profile, db.getDriverName(), db.getDriverModuleName(),
                                db.getDriverXaDatasourceClass());
                        System.out.println("Success");
                    } catch (CommandLineException cliEx) {
                        System.out.println("Failed");
                        logger.warn("Driver installation failed, unable to register with JBoss cli", cliEx);
                    }

                    updateDatasources(cli, profile, datasources, dsValues);
                    if (Arrays.asList(datasources).contains("BpelDS")) {
                        updateBpel(cli, profile, db.getHibernateSqlDialect());
                    }
                }
            } else {

                System.out.print("Registering driver: ");

                try {
                    cli.registerJdbcDriver(db.getDriverName(), db.getDriverModuleName(),
                            db.getDriverXaDatasourceClass());
                    System.out.println("Success");
                } catch (CommandLineException cliEx) {
                    System.out.println("Failed");
                    logger.warn("Driver installation failed, unable to register with JBoss cli", cliEx);
                }

                updateDatasources(cli, datasources, dsValues);
                if (Arrays.asList(datasources).contains("BpelDS")) {
                    updateBpel(cli, null, db.getHibernateSqlDialect());
                }
            }

            /*
             * Create schemas on the DB server if the user has requested it
             */
            if ("yes".equalsIgnoreCase(getProperty("create.db.schemas", "no").trim())) {
                // Export schema
                final DbSchema schema = new DbSchema(jbossHomeDir);
                System.out.printf("Exporting schema to: '%s'\n", dsValues.getConnectionUrl());
                schema.create(dsValues.getUsername(), String.valueOf(dsValues.getPassword()),
                        dsValues.getConnectionUrl(), db.getDriverClass(), db.getSqlDialect(), drv);

                // Redeploy s-ramp config
                System.out.print("Deploying S-RAMP config...");
                try {
                    if (isDomain) {
                        for (String profile : domainProfiles) {
                            cli.redeploySrampConfig(profile, jbossHomeDir, db.getDriverName());
                        }
                    } else {
                        cli.redeploySrampConfig(null, jbossHomeDir, db.getDriverName());
                    }
                    System.out.println("Success");
                } catch (CommandLineException cliEx) {
                    System.out.println("Failed");
                    logger.error("Error while attempting to deploy S-RAMP config", cliEx);
                }

                // Reload server
                System.out.println("Reloading server...");
                try {
                    cli.reloadServer();
                } catch (CommandLineException cliEx) {
                    logger.error("Error while attempting to reload server", cliEx);
                }

                // Seed S-RAMP
                System.out.println("Seeding S-RAMP...");
                File srampShell = getSrampShellJar();
                File srampCliCommands = new File(jbossHomeDir.getAbsolutePath()
                        + "/dtgov-sramp-repo-seed-cli-commands.txt");

                try {
                    cli.seedSramp(srampShell, srampCliCommands);
                    System.out.println("Seeding S-RAMP succeeded");
                } catch (Exception e) {
                    System.out.println("Seeding S-RAMP failed");
                    logger.warn("Seeding S-RAMP failed", e);
                }
            }

            System.out.println("\nInstallation complete.");

        } catch (FatalException fe) {
            // Log and notify the user.
            fatal(fe.getMessage(), fe.getCause());
        } catch (SecurityException se) {
            fatal("A SecurityException prevented the tool from performing a required operation.", se);
        } catch (Throwable t) {
            fatal("A fatal error prevented the tool from running", t);
        } finally {
            if (cli != null)
                try {
                    cli.shutdownServer();
                } catch (CommandLineException cliEx) {
                    logger.error("Error while attempting to shut down server", cliEx);
                }
        }

        //TODO: Remove this line, once patched jboss-as-cli lib available.
        System.exit(0);
    }

    private static File getSrampShellJar() {
        File jbossBin = new File(jbossHomeDir.getAbsolutePath() + "/bin");
        if (!jbossBin.isDirectory()) {
            throw new IllegalStateException("Directory " + jbossBin.getAbsolutePath() + " not found.");
        }
        File[] files = jbossBin.listFiles();

        for (File file : files) {
            if (file.getName().matches("(?i)^s-ramp-shell.*\\.jar$")){
                return file;
            }
        }
        return null;
    }

    private static void updateBpel(CommandExecutor cli, String profile, String dialect) {
        try {
            cli.updateBpelDialect(profile, dialect);
        } catch (WriteAttributeException e) {
            logger.error("Unable to configure BPEL", e);
        }
    }

    private static void updateDatasources(final CommandExecutor cmdExec, final String[] datasources,
                                          final DatasourceValues dsValues) {
        updateDatasources(cmdExec, null, datasources, dsValues);
    }

    /*
     * For the given domain profile, iterate over a list of datasources updating them with the values contained in a
     * given DatasourceValues object.
     */
    private static void updateDatasources(final CommandExecutor cmdExec, final String domainProfile,
            final String[] datasources, final DatasourceValues dsValues) {

        for (final String ds : datasources) {

            if (isDomain)
                System.out.print("    ");

            System.out.printf("Updating %s: ", ds);
            try {
                cmdExec.updateDatasource(domainProfile, ds, dsValues);
                System.out.print("Success\n");
            } catch (NoSuchDatasourceException ex) {
                System.out.print("Not found, skipping\n");
            } catch (WriteAttributeException ex) {
                System.out.print("Failed\n");
            }
        }
    }

    /*
     * Loads values from properties files etc.
     */
    private static void bootstrap() {
        logger.info("JBoss EAP DBTool: Initializing...");

        // Find the tool's home dir
        try {
            toolHomeDir = new File(Main.class.getProtectionDomain().getCodeSource().getLocation().toURI())
                    .getParentFile();
        } catch (URISyntaxException ex) {
            throw new FatalException();
        }

        logger.info("Tool home directory: '{}'", toolHomeDir);

        // Load the tool's Jar file.
        try {
            toolJarFile = new JarFile(new File(Main.class.getProtectionDomain().getCodeSource().getLocation().toURI()));
        } catch (URISyntaxException ex) {
            throw new FatalException();
        } catch (IOException ex) {
            throw new FatalException();
        }

        logger.info("Tool file: '{}'", toolJarFile.getName());

        // Load the directory to search for database drivers
        final String drvDir = System.getProperty("dbtool.db.driver.dir", "").trim();

        if (!drvDir.isEmpty()) {
            toolDriverDir = new File(drvDir);

            logger.trace("dbtool.db.driver.dir = '{}', toolDriverDir = '{}'", drvDir, toolDriverDir);

            if (!toolDriverDir.isDirectory())
                throw new FatalException("Given property dbtool.db.driver.dir: '" + drvDir + "' does not exist or is " +
                        "not a directory");
        } else {
            toolDriverDir = toolHomeDir;
        }

        logger.info("Tool will search for database driver in: '{}'", toolDriverDir);

        // Find the home directory of the JBoss server to configure in the following order of precedence:
        // 1) If the user has specified a directory by setting the system property jboss.home.dir
        // 2) If the user has set an OS environment variable JBOSS_HOME
        logger.debug("JBOSS_HOME='{}', jboss.home.dir='{}'", System.getenv("JBOSS_HOME"), System.getProperty(
                "jboss.home.dir"));

        String jbossDir = System.getProperty("jboss.home.dir");

        if (jbossDir == null || jbossDir.isEmpty())
            jbossDir = System.getenv("JBOSS_HOME");

        if (jbossDir != null && !jbossDir.isEmpty()) {
            jbossDir = jbossDir.trim();
            try {
                jbossHomeDir = new File(jbossDir).getCanonicalFile();
            } catch (IOException ex) {
                logger.warn("Unable to resolve symbolic links to path: '{}'", jbossDir);
                jbossHomeDir = new File(jbossDir);
            }

            if (!jbossHomeDir.isDirectory())
                throw new FatalException("Given property jboss.home.dir: '" + jbossHomeDir + "' does not exist or is" +
                        " not a directory");
        } else {
            throw new FatalException("A JBoss installation directory could not be found to configure. Either set " +
                    "envoronment variable JBOSS_HOME or system property jboss.home.dir to refer to the correct " +
                    "directory");
        }

        /*
         * Load autorun properties file, try to find it through 3 hierarchical methods.
         *
         * 1) If the system property dbtool.autorun.file is set, use that.
         * 2) Use a slight hack to find out the dir of the dbtool.jar file and try that.
         *
         */

        final String autorunProperty = System.getProperty("dbtool.autorun.file", "").trim();

        if (!autorunProperty.isEmpty()) {
            final File autorunFile = new File(autorunProperty);

            try {
                autorun.load(new FileInputStream(autorunFile));
            } catch (FileNotFoundException ex) {
                throw new FatalException("Given property dbtool.autorun.file: '" + autorunProperty + "' could not " +
                        "be found", ex);
            } catch (IOException ex) {
                throw new FatalException("Unable to load autorun file: " + autorunProperty);
            }
        } else {

            final File dbToolDir;
            try {
                dbToolDir = new File(Main.class.getProtectionDomain().getCodeSource().getLocation().toURI())
                        .getParentFile();
            } catch (URISyntaxException ex) {
                throw new FatalException("Unable to load autorun.properties file", ex);
            }

            try {
                autorun.load(new FileInputStream(new File(dbToolDir, AUTORUN_PROPERTIES_FILENAME)));
            } catch (FileNotFoundException ex) {
                throw new FatalException("Unable to find " + AUTORUN_PROPERTIES_FILENAME + " in dbtool directory and " +
                        "dbtool.autorun.file system property is not set", ex);
            } catch (IOException ex) {
                throw new FatalException("Unable to load autorun file");
            }
        }

        if (logger.isInfoEnabled()) {
            final StringBuilder sb = new StringBuilder();
            sb.append("Successfully loaded autorun script, with values: ");
            for (String key : autorun.stringPropertyNames())
                sb.append("\n").append(key).append(" = ").append("'").append(autorun.getProperty(key)).append("'");
            logger.info(sb.toString());
        }

        /*
         * Load tool supported databases.
         */
        loadDatabases();
        logger.debug("Loaded the following Databases: " + dbMap.values());

        /*
         * Load default set of datasources that the tool will update if none are specified.
         */
        Properties x;
        try {
            x = Utils.loadPropertiesFileFromClasspath("datasources.properties");
        } catch (IOException ioe) {
            throw new FatalException("Unable to load datasources.properties", ioe);
        }

        defaultDatasources = x.getProperty("datasources.defaults", "").split(",");

        logger.debug("Loaded the following default datasources: " + Arrays.toString(defaultDatasources));

        /*
         * Get details required to start JBoss server and connect to CLI
         */
        jbossHostname = getProperty("jboss.hostname", DEFAULT_SERVER_HOSTNAME);
        try {
            jbossNativeManagementPort = Integer.parseInt(getProperty("jboss.native.interface.port",
                    String.valueOf(DEFAULT_NATIVE_INTERFACE_PORT)));
        } catch (NumberFormatException nfe) {
            logger.error("{} contains illegal value for property: jboss.native.interface.port",
                    AUTORUN_PROPERTIES_FILENAME);
        }
        jbossNativeManagementUsername = getProperty("jboss.native.interface.username");
        jbossNativeManagementPassword = getProperty("jboss.native.interface.password")
                .toCharArray();

        logger.debug("jbossHostname='{}', jbossNativeManagemenPort='{}', jbossNativeManagementUsername='{}'," +
                "jbossNativeManagementPassword=*****", jbossHostname, jbossNativeManagementPort,
                jbossNativeManagementUsername);

        /*
         * Domain or standalone?
         */
        String jbossMode = autorun.getProperty("jboss.mode", "").trim();
        if ("domain".equalsIgnoreCase(jbossMode)) {
            isDomain = true;

            final String rawProfiles = autorun.getProperty("jboss.domain.profiles", "").trim();

            if (rawProfiles.isEmpty())
                throw new FatalException("Property jboss.domain.profiles must be set when jboss.mode=domain");

            // Extract and clean domain profiles from raw property.
            domainProfiles = rawProfiles.split(",");
            for (int i = 0; i < domainProfiles.length; i++)
                domainProfiles[i] = domainProfiles[i].trim();
        } else if ("standalone".equalsIgnoreCase(jbossMode)) {
            // For standalone mode provide an array with a single null value.
            // When iterating over the profiles this will cause a single
            // iteration, with the value null being passed to CommandExecutor
            // methods.
//            domainProfiles = EMPTY_PROFILE_SET;
            isDomain = false;
        } else {
            throw new FatalException("jboss.mode must be either 'domain' or 'standalone', supplied '" + jbossMode + "'");
        }
    }

    /*
     * Display welcome message
     */
    private static void welcome() {
        System.out.println("JBoss SOA Platform Database Configuration Tool");
        System.out.println("------------------------------------------------\n");
        System.out.println("This tool is used to configure the SOA platform and all its");
        System.out.println("constituent components against a new RDBMS.  Deployment scripts");
        System.out.println("are currently available for the following databases:");

        Utils.prettyPrinter(dbMap.values(), 3, System.out, new Utils.Transformer<Database>() {
            @Override
            public String prettyToString(Database d) {
                return d.getDescriptiveName();
            }
        });

        System.out.println("\n");

        System.out.println("** Warnings **");
        System.out.println("  This tool may not work correctly if you have made manual");
        System.out.println("  changes to the database configuration.");
        System.out.println("  This tool is only intended to do initial configuration.");
        System.out.println("  This tool only creates empty initial database tables, it does not migrate data.");
        System.out.println("  Databases marked (*) are not certified for new installations.\n");
    }

    // Method abstracts from the property object in preparation for addition
    // of interactive mode.
    private static String getProperty(final String key, final String defaultVal) {
        return autorun.getProperty(key, defaultVal).trim();
    }

    private static String getProperty(final String key) {
        return getProperty(key, "").trim();
    }

    // Populate a map of database platforms that the tool can configure for the user.
    private static void loadDatabases() {
        logger.debug("Main.loadDatabases()");

        final Map<String, Properties> propsMap = Utils.getAllPropsFilesFromDirInJar(toolJarFile,
                TOOL_DATABASE_DIR_NAME);

        logger.trace("Loaded databes configuration scripts: {}", propsMap);

        for (final String s : propsMap.keySet()) {
            // Create a database object from the properties file,

            try {
                final Database db = Database.newInstance(s, propsMap.get(s));
                dbMap.put(db.getId(), db);
            } catch (RuntimeException re) {
                logger.warn("Unable to load database from properties file: " + s, re);
            }
        }
    }

    private static void fatal(final String message, final Throwable t) {
        final String messagef = "[FATAL] " + message;
        logger.error(messagef, t);
        System.err.printf("\n\n%s\n", messagef);
        if (t != null && t.getMessage() != null)
            System.err.printf("Error message: %s\n", t.getMessage());
        System.err.printf("More information may be available in %s\n", System.getProperty("dbtool.log.file",
                "dbtool.log"));
    }

    /*
     * Signals an unrecoverable runtime error.
     */
    private static class FatalException extends RuntimeException {
        public FatalException() {
            super();
        }

        public FatalException(String message) {
            super(message);
        }

        public FatalException(String message, Throwable cause) {
            super(message, cause);
        }

        public FatalException(Throwable cause) {
            super(cause);
        }
    }
}
