package org.jboss.ip.dbtool.cliutil;

/**
 * A DatasourceValues instance is an immutable representation of the configurable parameters of a JBoss datasource.
 *
 * The class is designed using the builder pattern and should be instantiated by first creating an instance of its
 * embedded Builder class - DatasourceValues.Builder.
 *
 * @author Alex Creasy <acreasy@redhat.com>
 */
public final class DatasourceValues {

    private final String jndiName;
    private final String connectionUrl;
    private final int minPoolSize;
    private final int maxPoolSize;
    private final String username;
    private final char[] password;

    private final String driverName;
    private final String driverExceptionSorter;
    private final String driverValidConnectionChecker;
    private final String driverStaleConnectionChecker;


    /*
     * Use Builder nested class to instantiate (allows immutability without having to set all params immediately)
     * e.g. DatasourceValues ds = new DatasourceValues.Builder().setConnectionUrl("foo").setJndiName("bah").build();
     *
     */
    private DatasourceValues(String jndiName, String connectionUrl, String driverName, String driverExceptionSorter,
                             String driverValidConnectionChecker, String driverStaleConnectionChecker, int minPoolSize,
                             int maxPoolSize, String username, char[] password) {

        this.jndiName = jndiName;
        this.connectionUrl = connectionUrl;
        this.driverName = driverName;
        this.driverExceptionSorter = driverExceptionSorter;
        this.driverValidConnectionChecker = driverValidConnectionChecker;
        this.driverStaleConnectionChecker = driverStaleConnectionChecker;
        this.minPoolSize = minPoolSize;
        this.maxPoolSize = maxPoolSize;
        this.username = username;
        this.password = password;
    }

    public String getJndiName() {
        return jndiName;
    }

    public String getConnectionUrl() {
        return connectionUrl;
    }

    public String getDriverName() {
        return driverName;
    }

    public String getDriverExceptionSorter() {
        return driverExceptionSorter;
    }

    public String getDriverValidConnectionChecker() {
        return driverValidConnectionChecker;
    }

    public String getDriverStaleConnectionChecker() {
        return driverStaleConnectionChecker;
    }

    public int getMinPoolSize() {
        return minPoolSize;
    }

    public int getMaxPoolSize() {
        return maxPoolSize;
    }

    public String getUsername() {
        return username;
    }

    public char[] getPassword() {
        return password;
    }

    @Override
    public String toString() {
        final StringBuilder sb = new StringBuilder("DatasourceValues{");
        sb.append(", jndiName='").append(jndiName).append('\'');
        sb.append(", connectionUrl='").append(connectionUrl).append('\'');
        sb.append(", driverName='").append(driverName).append('\'');
        sb.append(", driverExceptionSorter='").append(driverExceptionSorter).append('\'');
        sb.append(", driverValidConnectionChecker='").append(driverValidConnectionChecker).append('\'');
        sb.append(", driverStaleConnectionChecker='").append(driverStaleConnectionChecker).append('\'');
        sb.append(", minPoolSize=").append(minPoolSize);
        sb.append(", maxPoolSize=").append(maxPoolSize);
        sb.append(", username='").append(username).append('\'');
        sb.append(", password=").append("'REDACTED'");
        sb.append('}');
        return sb.toString();
    }

    /**
     * Builder class for DatasourceValues.
     *
     * A DatasourceValues instance is obtained by first creating a new DatasourceValues.Builder instance. The builder's setter
     * methods can then be used to set the values of the datasource before calling the build() method to obtain an
     * immutable DatasourceValues containing the given parameters.
     *
     * Example usage:
     *
     * <code>
     * // Obtain builder instance
     * DatasourceValues.Builder dsBuilder = new DatasourceValues.Builder();
     *
     * // Set parameter PoolName, this parameter must be set before calling build().
     * dsBuilder.setJndiName("java:/jboss/db2");
     *
     * // Builder setter methods can be chained like so:
     * dsBuilder.setConnectionURL("jdbc:db2://mydb.com:50000/mydb").setDriverName("db2");
     *
     * // Create DatasourceValues instance from builder.
     * DatasourceValues ds = dsBuilder.build();
     * </code>
     *
     * @author Alex Creasy <acreasy@redhat.com>
     */
    public static class Builder {
        private String jndiName;
        private String connectionUrl;
        private String driverName;
        private String driverExceptionSorter;
        private String driverValidConnectionChecker;
        private String driverStaleConnectionChecker;
        private int minPoolSize;
        private int maxPoolSize;
        private String username;
        private char[] password;

        public Builder setJndiName(String jndiName) {
            this.jndiName = jndiName;
            return this;
        }

        public Builder setConnectionUrl(String connectionUrl) {
            this.connectionUrl = connectionUrl;
            return this;
        }

        public Builder setDriverName(String driverName) {
            this.driverName = driverName;
            return this;
        }

        public Builder setDriverExceptionSorter(String driverExceptionSorter) {
            this.driverExceptionSorter = driverExceptionSorter;
            return this;
        }

        public Builder setDriverValidConnectionChecker(String driverValidConnectionChecker) {
            this.driverValidConnectionChecker = driverValidConnectionChecker;
            return this;
        }

        public Builder setDriverStaleConnectionChecker(String driverStaleConnectionChecker) {
            this.driverStaleConnectionChecker = driverStaleConnectionChecker;
            return this;
        }

        public Builder setMinPoolSize(int minPoolSize) {
            this.minPoolSize = minPoolSize;
            return this;
        }

        public Builder setMaxPoolSize(int maxPoolSize) {
            this.maxPoolSize = maxPoolSize;
            return this;
        }

        public Builder setUsername(String username) {
            this.username = username;
            return this;
        }

        public Builder setPassword(char[] password) {
            this.password = password;
            return this;
        }

        /**
         * Constructs an immutable DatasourceValues instance with the current values of the builder.
         *
         * @return a DatasourceValues populated with the values given to the builder.
         */
        public DatasourceValues build() {
            return new DatasourceValues(jndiName, connectionUrl, driverName, driverExceptionSorter,
                    driverValidConnectionChecker, driverStaleConnectionChecker, minPoolSize, maxPoolSize, username,
                    password);
        }
    }
}

