package org.jboss.qe.shared.util.jms;

import javax.jms.BytesMessage;
import javax.jms.Connection;
import javax.jms.ConnectionFactory;
import javax.jms.Destination;
import javax.jms.JMSException;
import javax.jms.MapMessage;
import javax.jms.Message;
import javax.jms.MessageConsumer;
import javax.jms.MessageProducer;
import javax.jms.ObjectMessage;
import javax.jms.Session;
import javax.jms.StreamMessage;
import javax.jms.TextMessage;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

public final class JmsClient {
	
	private static final Logger log = LoggerFactory.getLogger(JmsClient.class);
	
	private static final long MAX_WAIT_FOR_MESSAGES_WIPE = 100L;

	private enum DestinationType {
		QUEUE {
			@Override
			public Destination createDestination(Session session, String destinationName) throws JMSException {
				return session.createQueue(destinationName);
			}
		},
		TOPIC {
			@Override
			public Destination createDestination(Session session, String destinationName) throws JMSException {
				return session.createTopic(destinationName);
			}
		};

		public abstract Destination createDestination(Session session, String destinationName) throws JMSException;
	}

	/**
	 * Receives messages from queue until it's empty.
	 *
	 * @param cf the ConnectionFactory used to create a connection
	 * @param queueName the queue name
	 * @throws JMSException
	 */
	public static void clearQueue(final ConnectionFactory cf, final String queueName) throws JMSException {
		String message;
		while ((message = receiveFromQueue(cf, queueName, MAX_WAIT_FOR_MESSAGES_WIPE)) != null) {
			final String msg = "From queue " + queueName + " cleared message: " + message;
			log.info(msg);
		}
	}

	/**
	 * Receives all messages from a queue.
	 *
	 * @param cf the ConnectionFactory used to create a connection
	 * @param queueName the queue name
	 * @return String representations of all messages
	 * @throws JMSException
	 */
	public static List<String> receiveAllFromQueue(final ConnectionFactory cf, final String queueName)
			throws JMSException {
		final List<String> result = new ArrayList<>();
		String message;
		while ((message = receiveFromQueue(cf, queueName, MAX_WAIT_FOR_MESSAGES_WIPE)) != null) {
			result.add(message);
		}
		return result;
	}

	/**
	 * Receives a message from a queue within a specified timeout interval.
	 *
	 * @param cf the ConnectionFactory used to create a connection
	 * @param queueName the queue name
	 * @param timeout the timeout
	 * @return the String representation of the message
	 * @throws JMSException
	 */
	public static String receiveFromQueue(ConnectionFactory cf, String queueName, long timeout) throws JMSException {
		return receiveFromDestination(cf, queueName, DestinationType.QUEUE, timeout);
	}

	/**
	 * Receives a message from a queue within a specified timeout interval.
	 *
	 * @param cf the ConnectionFactory used to create a connection
	 * @param queueName the queue name
	 * @param timeout the timeout
	 * @return the received {@link Message}
	 * @throws JMSException
	 */
	public static Message receiveMessageFromQueue(ConnectionFactory cf, String queueName, long timeout) throws JMSException {
		return receiveMessageFromDestination(cf, queueName, DestinationType.QUEUE, timeout);
	}

	/**
	 * Receives a message from a topic within a specified timeout interval.
	 *
	 * @param cf the ConnectionFactory used to create a connection
	 * @param topicName the topic name
	 * @param timeout the timeout
	 * @return the String representation of the message
	 * @throws JMSException
	 */
	public static String receiveFromTopic(ConnectionFactory cf, String topicName, long timeout)
			throws JMSException {
		return receiveFromDestination(cf, topicName, DestinationType.TOPIC, timeout);
	}

	/**
	 * Receives a message from a topic within a specified timeout interval.
	 *
	 * @param cf the ConnectionFactory used to create a connection
	 * @param topicName the topic name
	 * @param timeout the timeout
	 * @return the received {@link Message}
	 * @throws JMSException
	 */
	public static Message receiveMessageFromTopic(ConnectionFactory cf, String topicName, long timeout)
			throws JMSException {
		return receiveMessageFromDestination(cf, topicName, DestinationType.TOPIC, timeout);
	}

	private static String receiveFromDestination(ConnectionFactory cf, String destinationName,
			DestinationType destinationType, long timeout) throws JMSException {
		final Message message = receiveMessageFromDestination(cf, destinationName, destinationType, timeout);

		String messageContent = null;
		if (message instanceof TextMessage) {
			final TextMessage textMessage = (TextMessage) message;
			messageContent = textMessage.getText();
		} else if (message != null) {
			messageContent = message.toString();
		}
		log.info("Message \"{}\" received from {} {} ", messageContent,
				destinationType.toString().toLowerCase(), destinationName);
		return messageContent;
	}

	private static Message receiveMessageFromDestination(ConnectionFactory cf, String destinationName,
			DestinationType destinationType, long timeout) throws JMSException {
		Connection connection = null;
		try {
			connection = createConnection(cf);
			connection.start();
			final Session session = connection.createSession(false, Session.AUTO_ACKNOWLEDGE);
			final Destination destination = destinationType.createDestination(session, destinationName);
			final MessageConsumer consumer = session.createConsumer(destination);
			final Message message = consumer.receive(timeout);
			log.info("Message \"{}\" received from {} {} ", message,
					destinationType.toString().toLowerCase(), destinationName);
			return message;
		} finally {
			if (connection != null) {
				connection.close();
			}
		}
	}

	/**
	 * Sends a text message to a queue.
	 *
	 * @param cf the ConnectionFactory used to create a connection
	 * @param queueName the queue name
	 * @param message the message to send
	 * @throws JMSException
	 */
	public static void sendToQueue(ConnectionFactory cf, String queueName, String message) throws JMSException {
		send(cf, DestinationType.QUEUE, queueName, message);
	}

	/**
	 * Sends a text message to a queue.
	 *
	 * @param cf the ConnectionFactory used to create a connection
	 * @param queueName the queue name
	 * @param message the message to send
	 * @param enricher jms message enricher
	 * @throws JMSException
	 */
	public static void sendToQueue(ConnectionFactory cf, String queueName, String message, MessageEnricher enricher)
			throws JMSException {
		send(cf, DestinationType.QUEUE, queueName, message, enricher);
	}

	/**
	 * Sends a message to a queue.
	 *
	 * @param cf the ConnectionFactory used to create a connection
	 * @param queueName the queue name
	 * @param message the message to send
	 * @throws JMSException
	 */
	public static void sendToQueue(ConnectionFactory cf, String queueName, Message message)
			throws JMSException {
		send(cf, DestinationType.QUEUE, queueName, message);
	}

	/**
	 * Sends a text message to a topic.
	 *
	 * @param cf the ConnectionFactory used to create a connection
	 * @param queueName the topic name
	 * @param message the message to send
	 * @throws JMSException
	 */
	public static void sendToTopic(ConnectionFactory cf, String queueName, String message)
			throws JMSException {
		send(cf, DestinationType.TOPIC, queueName, message);
	}

	/**
	 * Sends a message to a topic.
	 *
	 * @param cf the ConnectionFactory used to create a connection
	 * @param queueName the topic name
	 * @param message the message to send
	 * @throws JMSException
	 */
	public static void sendToTopic(ConnectionFactory cf, String queueName, Message message)
			throws JMSException {
		send(cf, DestinationType.TOPIC, queueName, message);
	}

	private static void send(ConnectionFactory cf, DestinationType destinationType, String destinationName,
			String message) throws JMSException {
		send(cf, destinationType, destinationName, message, null);
	}

	private static void send(ConnectionFactory cf, DestinationType destinationType, String destinationName,
			String message, MessageEnricher enricher) throws JMSException {
		Connection connection = null;
		try {
			connection = createConnection(cf);
			connection.start();
			final Session session = connection.createSession(false, Session.AUTO_ACKNOWLEDGE);
			final Destination destination = destinationType.createDestination(session, destinationName);
			final MessageProducer producer = session.createProducer(destination);
			final TextMessage textMessage = session.createTextMessage(message);
			if (enricher != null) {
				enricher.enrich(textMessage);
			}
			producer.send(textMessage);
			log.info("Message \"{}\" sent to {} {} ", message, destinationType.toString().toLowerCase(),
					destinationName);
		} finally {
			if (connection != null) {
				connection.close();
			}
		}
	}

	private static void send(ConnectionFactory cf, DestinationType destinationType, String destinationName,
			Message message) throws JMSException {
		Connection connection = null;
		try {
			connection = createConnection(cf);
			connection.start();
			final Session session = connection.createSession(false, Session.AUTO_ACKNOWLEDGE);
			final Destination destination = destinationType.createDestination(session, destinationName);
			final MessageProducer producer = session.createProducer(destination);
			producer.send(message);
			log.info("Message \"{}\" sent to {} {} ", message, destinationType.toString().toLowerCase(),
					destinationName);
		} finally {
			if (connection != null) {
				connection.close();
			}
		}
	}

	/**
	 * Create JMS message. The message type is based on payload argument.
	 * For each Message type you must provide method with proper payload:
	 * <p>
	 * <ul>
	 * <li>String for TextMessage</li>
	 * <li>byte[] for BytesMessage</li>
	 * <li>Map for MapMessage</li>
	 * <li>List for StreamMessage</li>
	 * <li>Arbitrary object for ObjectMessage</li>
	 * <li></li>
	 * </ul>
	 *
	 * @param cf
	 * @param payload
	 * @param messageType
	 * @return
	 * @throws JMSException
	 */
	public static <T extends Message> T createMessage(ConnectionFactory cf, Object payload, Class<T> messageType) throws JMSException {
		Message message = null;
		Connection connection = null;
		try {
			connection = createConnection(cf);
			connection.start();
			final Session session = connection.createSession(false, Session.AUTO_ACKNOWLEDGE);

			if (messageType == TextMessage.class) {
				if (!(payload instanceof String)) {
					throw new IllegalArgumentException("Payload must be String for textMessage");
				}
				message = session.createTextMessage((String) payload);
			}
			if (messageType == BytesMessage.class) {
				if (!(payload instanceof byte[])) {
					throw new IllegalArgumentException("Payload must be Byte[] for BytesMessage");
				}
				final BytesMessage bytesMessage = session.createBytesMessage();
				bytesMessage.writeBytes((byte[]) payload);
				message = bytesMessage;
			}
			if (messageType == MapMessage.class) {
				if (!(payload instanceof Map<?, ?>)) {
					throw new IllegalArgumentException("Payload must be Map<?,?> for MapMessage");
				}
				final Map<?, ?> map = (Map<?, ?>) payload;
				final MapMessage mapMessage = session.createMapMessage();
				for (Object key : map.keySet()) {
					mapMessage.setObject(String.valueOf(key), map.get(key));
				}
				message = mapMessage;
			}
			if (messageType == StreamMessage.class) {
				if (!(payload instanceof List<?>)) {
					throw new IllegalArgumentException("Payload must be List<?> for stream message");
				}
				final StreamMessage streamMessage = session.createStreamMessage();
				final List<?> list = (List<?>) payload;
				for (Object o : list) {
					streamMessage.writeObject(o);
				}
				message = streamMessage;
			}
			if (messageType == ObjectMessage.class) {
				if (!(payload instanceof Serializable)) {
					throw new IllegalArgumentException("Payload must be serializable for object message");
				}
				message = session.createObjectMessage((Serializable) payload);
			}
			return messageType.cast(message);
		} finally {
			if (connection != null) {
				connection.close();
			}
		}
	}

	public static Connection createConnection(ConnectionFactory cf) throws JMSException {
		//final String username = System.getProperty("external.broker.username", "admin");
		//final String password = System.getProperty("external.broker.password", "admin");
		return cf.createConnection();
	}

	/*public static ConnectionFactory createAmqConnectionFactory() {
		final String defaultUsername = System.getProperty("jboss.qa.username", "admin");
		final String defaultPassword = System.getProperty("jboss.qa.password", "admin");
		final String defaultBrokerUrl = System.getProperty("jboss.qa.brokerUrl", "tcp://localhost:61616");

		try {
			return ActiveMQJMSClient.createConnectionFactory(
					String.format("%s?user=%s&password=%s", defaultBrokerUrl, defaultUsername, defaultPassword),
					"broker");
		} catch (Exception e) {
			log.error(e.getMessage(), e);
			throw new RuntimeException(e);
		}
	}*/

	private JmsClient() {
	}
}
